import codecs
import os
import re

from setuptools import find_packages, setup


#  https://packaging.python.org/guides/single-sourcing-package-version/
#  #single-sourcing-the-version


def read_file(filename: str) -> str:
    """Read package file as text to get name and version."""
    # intentionally *not* adding an encoding option to open
    # see here:
    # https://github.com/pypa/virtualenv/issues/201#issuecomment-3145690
    here = os.path.abspath(os.path.dirname(__file__))
    with codecs.open(os.path.join(here, "typing_copilot", filename), "r") as f:
        return f.read()


def find_version() -> str:
    """Only define version in one place."""
    version_file = read_file("__init__.py")
    version_match = re.search(r'^__version__ = ["\']([^"\']*)["\']', version_file, re.M)
    if version_match:
        return version_match.group(1)
    raise RuntimeError("Unable to find version string.")


def find_name() -> str:
    """Only define name in one place."""
    name_file = read_file("__init__.py")
    name_match = re.search(r'^__package_name__ = ["\']([^"\']*)["\']', name_file, re.M)
    if name_match:
        return name_match.group(1)
    raise RuntimeError("Unable to find name string.")


def find_long_description() -> str:
    """Return the content of the README.md file."""
    return read_file("../README.md")


setup(
    name=find_name(),
    version=find_version(),
    description="Helper for starting to type-hint large codebases with mypy.",
    long_description=find_long_description(),
    long_description_content_type="text/markdown",
    url="https://github.com/obi1kenobi/typing-copilot",
    author="Predrag Gruevski",
    author_email="obi1kenobi82@gmail.com",
    license="MIT",
    packages=find_packages(),
    entry_points={"console_scripts": ["typing_copilot=typing_copilot.cli:cli"]},
    install_requires=[  # Make sure to keep in sync with poetry requirements.
        "click>=8,<9",
        "mypy>=0.782",
        "toml>=0.10.2,<1",
        "types-toml>=0.10.4,<1",
    ],
    python_requires=">=3.7",
)
