from typing import Any, Dict, List

from .error_tracker import MypyErrorSetting
from .own_config import TypingCopilotConfig
from .validation import validate_module_name
from . import __package_name__, __version__


AUTOGENERATED_LINE_PREFIX = f"# Autogenerated by {__package_name__}"


def _make_mypy_config(mypy_global_config_values: Dict[str, Any]) -> str:
    header = f"""\
{AUTOGENERATED_LINE_PREFIX} v{__version__}
[mypy]"""

    config_lines = [header]
    config_lines.extend(f"{key} = {value}" for key, value in mypy_global_config_values.items())

    return "\n".join(config_lines) + "\n"


def make_strict_baseline_mypy_config(own_config: TypingCopilotConfig) -> str:
    default_values: Dict[str, Any] = {
        "no_implicit_optional": True,
        "strict_optional": True,
        "warn_redundant_casts": True,
        "check_untyped_defs": True,
        "disallow_untyped_calls": True,
        "disallow_incomplete_defs": True,
        "disallow_untyped_defs": True,
        "disallow_untyped_decorators": True,
        "ignore_missing_imports": False,
    }

    # Start with the defaults, then add/override any explicitly-configured settings.
    config_values = dict(default_values)
    config_values.update(own_config.mypy_global_config)

    return _make_mypy_config(config_values)


def make_lax_baseline_mypy_config(own_config: TypingCopilotConfig) -> str:
    default_values: Dict[str, Any] = {
        "no_implicit_optional": False,
        "strict_optional": False,
        "warn_redundant_casts": False,
        "check_untyped_defs": False,
        "disallow_untyped_calls": False,
        "disallow_incomplete_defs": False,
        "disallow_untyped_defs": False,
        "disallow_untyped_decorators": False,
        "ignore_missing_imports": True,
    }

    # Start with the defaults, then add/override any explicitly-configured settings.
    config_values = dict(default_values)
    config_values.update(own_config.mypy_global_config)

    return _make_mypy_config(config_values)


def make_unused_ignores_config_line(unused_ignores_setting: bool) -> str:
    # N.B.: As of version 0.782, mypy only reports these errors if other checks pass.
    return f"warn_unused_ignores = {unused_ignores_setting}\n"


def make_ignore_missing_imports_block(module_name: str) -> str:
    validate_module_name(module_name)

    return f"""
[mypy-{module_name}.*]
ignore_missing_imports = True
"""


def make_1st_party_module_rule_block(module_name: str, rules: List[MypyErrorSetting]) -> str:
    validate_module_name(module_name)

    section_header = f"\n[mypy-{module_name}.*]\n"
    rule_lines = [f"{rule_name} = {value}" for (rule_name, value) in rules]

    return section_header + "\n".join(rule_lines) + "\n"
