"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cdk = require("@aws-cdk/core");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const application_listener_action_1 = require("./application-listener-action");
/**
 * The content type for a fixed response
 */
var ContentType;
(function (ContentType) {
    ContentType["TEXT_PLAIN"] = "text/plain";
    ContentType["TEXT_CSS"] = "text/css";
    ContentType["TEXT_HTML"] = "text/html";
    ContentType["APPLICATION_JAVASCRIPT"] = "application/javascript";
    ContentType["APPLICATION_JSON"] = "application/json";
})(ContentType = exports.ContentType || (exports.ContentType = {}));
/**
 * Define a new listener rule
 */
class ApplicationListenerRule extends cdk.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.conditions = {};
        const hasPathPatterns = props.pathPatterns || props.pathPattern;
        if (!props.hostHeader && !hasPathPatterns) {
            throw new Error('At least one of \'hostHeader\', \'pathPattern\' or \'pathPatterns\' is required when defining a load balancing rule.');
        }
        const possibleActions = ['action', 'targetGroups', 'fixedResponse', 'redirectResponse'];
        const providedActions = possibleActions.filter(action => props[action] !== undefined);
        if (providedActions.length > 1) {
            throw new Error(`'${providedActions}' specified together, specify only one`);
        }
        if (props.priority <= 0) {
            throw new Error('Priority must have value greater than or equal to 1');
        }
        this.listener = props.listener;
        const resource = new elasticloadbalancingv2_generated_1.CfnListenerRule(this, 'Resource', {
            listenerArn: props.listener.listenerArn,
            priority: props.priority,
            conditions: cdk.Lazy.anyValue({ produce: () => this.renderConditions() }),
            actions: cdk.Lazy.anyValue({ produce: () => this.action ? this.action.renderActions() : [] }),
        });
        if (props.hostHeader) {
            this.setCondition('host-header', [props.hostHeader]);
        }
        if (hasPathPatterns) {
            if (props.pathPattern && props.pathPatterns) {
                throw new Error('Both `pathPatterns` and `pathPattern` are specified, specify only one');
            }
            const pathPattern = props.pathPattern ? [props.pathPattern] : props.pathPatterns;
            this.setCondition('path-pattern', pathPattern);
        }
        if (props.action) {
            this.configureAction(props.action);
        }
        (props.targetGroups || []).forEach(this.addTargetGroup.bind(this));
        if (props.fixedResponse) {
            this.addFixedResponse(props.fixedResponse);
        }
        else if (props.redirectResponse) {
            this.addRedirectResponse(props.redirectResponse);
        }
        this.listenerRuleArn = resource.ref;
    }
    /**
     * Add a non-standard condition to this rule
     */
    setCondition(field, values) {
        this.conditions[field] = values;
    }
    /**
     * Configure the action to perform for this rule
     */
    configureAction(action) {
        // It might make sense to 'throw' here.
        //
        // However, programs may already exist out there which configured an action twice,
        // in which case the second action accidentally overwrite the initial action, and in some
        // way ended up with a program that did what the author intended. If we were to add throw now,
        // the previously working program would be broken.
        //
        // Instead, signal this through a warning.
        // @deprecate: upon the next major version bump, replace this with a `throw`
        if (this.action) {
            this.node.addWarning('An Action already existed on this ListenerRule and was replaced. Configure exactly one default Action.');
        }
        action.bind(this, this.listener, this);
        this.action = action;
    }
    /**
     * Add a TargetGroup to load balance to
     *
     * @deprecated Use configureAction instead
     */
    addTargetGroup(targetGroup) {
        this.configureAction(application_listener_action_1.ListenerAction.forward([targetGroup]));
    }
    /**
     * Add a fixed response
     *
     * @deprecated Use configureAction instead
     */
    addFixedResponse(fixedResponse) {
        validateFixedResponse(fixedResponse);
        this.configureAction(application_listener_action_1.ListenerAction.fixedResponse(cdk.Token.asNumber(fixedResponse.statusCode), {
            contentType: fixedResponse.contentType,
            messageBody: fixedResponse.messageBody,
        }));
    }
    /**
     * Add a redirect response
     *
     * @deprecated Use configureAction instead
     */
    addRedirectResponse(redirectResponse) {
        validateRedirectResponse(redirectResponse);
        this.configureAction(application_listener_action_1.ListenerAction.redirect({
            host: redirectResponse.host,
            path: redirectResponse.path,
            permanent: redirectResponse.statusCode === 'HTTP_301',
            port: redirectResponse.port,
            protocol: redirectResponse.protocol,
            query: redirectResponse.query,
        }));
    }
    /**
     * Validate the rule
     */
    validate() {
        if (this.action === undefined) {
            return ['Listener rule needs at least one action'];
        }
        return [];
    }
    /**
     * Render the conditions for this rule
     */
    renderConditions() {
        const ret = new Array();
        for (const [field, values] of Object.entries(this.conditions)) {
            if (values !== undefined) {
                ret.push({ field, values });
            }
        }
        return ret;
    }
}
exports.ApplicationListenerRule = ApplicationListenerRule;
/**
 * Validate the status code and message body of a fixed response
 *
 * @internal
 */
function validateFixedResponse(fixedResponse) {
    if (fixedResponse.statusCode && !/^(2|4|5)\d\d$/.test(fixedResponse.statusCode)) {
        throw new Error('`statusCode` must be 2XX, 4XX or 5XX.');
    }
    if (fixedResponse.messageBody && fixedResponse.messageBody.length > 1024) {
        throw new Error('`messageBody` cannot have more than 1024 characters.');
    }
}
exports.validateFixedResponse = validateFixedResponse;
/**
 * Validate the status code and message body of a redirect response
 *
 * @internal
 */
function validateRedirectResponse(redirectResponse) {
    if (redirectResponse.protocol && !/^(HTTPS?|#\{protocol\})$/i.test(redirectResponse.protocol)) {
        throw new Error('`protocol` must be HTTP, HTTPS, or #{protocol}.');
    }
    if (!redirectResponse.statusCode || !/^HTTP_30[12]$/.test(redirectResponse.statusCode)) {
        throw new Error('`statusCode` must be HTTP_301 or HTTP_302.');
    }
}
exports.validateRedirectResponse = validateRedirectResponse;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwbGljYXRpb24tbGlzdGVuZXItcnVsZS5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImFwcGxpY2F0aW9uLWxpc3RlbmVyLXJ1bGUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7QUFBQSxxQ0FBcUM7QUFDckMsMEZBQXNFO0FBR3RFLCtFQUErRDtBQWlHL0Q7O0dBRUc7QUFDSCxJQUFZLFdBTVg7QUFORCxXQUFZLFdBQVc7SUFDckIsd0NBQXlCLENBQUE7SUFDekIsb0NBQXFCLENBQUE7SUFDckIsc0NBQXdCLENBQUE7SUFDeEIsZ0VBQWlELENBQUE7SUFDakQsb0RBQXFDLENBQUE7QUFDdkMsQ0FBQyxFQU5XLFdBQVcsR0FBWCxtQkFBVyxLQUFYLG1CQUFXLFFBTXRCO0FBc0VEOztHQUVHO0FBQ0gsTUFBYSx1QkFBd0IsU0FBUSxHQUFHLENBQUMsU0FBUztJQVd4RCxZQUFZLEtBQW9CLEVBQUUsRUFBVSxFQUFFLEtBQW1DO1FBQy9FLEtBQUssQ0FBQyxLQUFLLEVBQUUsRUFBRSxDQUFDLENBQUM7UUFORixlQUFVLEdBQTBDLEVBQUUsQ0FBQztRQVF0RSxNQUFNLGVBQWUsR0FBRyxLQUFLLENBQUMsWUFBWSxJQUFJLEtBQUssQ0FBQyxXQUFXLENBQUM7UUFDaEUsSUFBSSxDQUFDLEtBQUssQ0FBQyxVQUFVLElBQUksQ0FBQyxlQUFlLEVBQUU7WUFDekMsTUFBTSxJQUFJLEtBQUssQ0FBQyxzSEFBc0gsQ0FBQyxDQUFDO1NBQ3pJO1FBRUQsTUFBTSxlQUFlLEdBQThDLENBQUMsUUFBUSxFQUFFLGNBQWMsRUFBRSxlQUFlLEVBQUUsa0JBQWtCLENBQUMsQ0FBQztRQUNuSSxNQUFNLGVBQWUsR0FBRyxlQUFlLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQyxFQUFFLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxLQUFLLFNBQVMsQ0FBQyxDQUFDO1FBQ3RGLElBQUksZUFBZSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDOUIsTUFBTSxJQUFJLEtBQUssQ0FBQyxJQUFJLGVBQWUsd0NBQXdDLENBQUMsQ0FBQztTQUM5RTtRQUVELElBQUksS0FBSyxDQUFDLFFBQVEsSUFBSSxDQUFDLEVBQUU7WUFDdkIsTUFBTSxJQUFJLEtBQUssQ0FBQyxxREFBcUQsQ0FBQyxDQUFDO1NBQ3hFO1FBRUQsSUFBSSxDQUFDLFFBQVEsR0FBRyxLQUFLLENBQUMsUUFBUSxDQUFDO1FBRS9CLE1BQU0sUUFBUSxHQUFHLElBQUksa0RBQWUsQ0FBQyxJQUFJLEVBQUUsVUFBVSxFQUFFO1lBQ3JELFdBQVcsRUFBRSxLQUFLLENBQUMsUUFBUSxDQUFDLFdBQVc7WUFDdkMsUUFBUSxFQUFFLEtBQUssQ0FBQyxRQUFRO1lBQ3hCLFVBQVUsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLEVBQUUsRUFBRSxDQUFDO1lBQ3pFLE9BQU8sRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsTUFBTSxDQUFDLGFBQWEsRUFBRSxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUUsQ0FBQztTQUM5RixDQUFDLENBQUM7UUFFSCxJQUFJLEtBQUssQ0FBQyxVQUFVLEVBQUU7WUFDcEIsSUFBSSxDQUFDLFlBQVksQ0FBQyxhQUFhLEVBQUUsQ0FBQyxLQUFLLENBQUMsVUFBVSxDQUFDLENBQUMsQ0FBQztTQUN0RDtRQUVELElBQUksZUFBZSxFQUFFO1lBQ25CLElBQUksS0FBSyxDQUFDLFdBQVcsSUFBSSxLQUFLLENBQUMsWUFBWSxFQUFFO2dCQUMzQyxNQUFNLElBQUksS0FBSyxDQUFDLHVFQUF1RSxDQUFDLENBQUM7YUFDMUY7WUFDRCxNQUFNLFdBQVcsR0FBRyxLQUFLLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMsS0FBSyxDQUFDLFlBQVksQ0FBQztZQUNqRixJQUFJLENBQUMsWUFBWSxDQUFDLGNBQWMsRUFBRSxXQUFXLENBQUMsQ0FBQztTQUNoRDtRQUVELElBQUksS0FBSyxDQUFDLE1BQU0sRUFBRTtZQUNoQixJQUFJLENBQUMsZUFBZSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsQ0FBQztTQUNwQztRQUVELENBQUMsS0FBSyxDQUFDLFlBQVksSUFBSSxFQUFFLENBQUMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQztRQUVuRSxJQUFJLEtBQUssQ0FBQyxhQUFhLEVBQUU7WUFDdkIsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxhQUFhLENBQUMsQ0FBQztTQUM1QzthQUFNLElBQUksS0FBSyxDQUFDLGdCQUFnQixFQUFFO1lBQ2pDLElBQUksQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLLENBQUMsZ0JBQWdCLENBQUMsQ0FBQztTQUNsRDtRQUVELElBQUksQ0FBQyxlQUFlLEdBQUcsUUFBUSxDQUFDLEdBQUcsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7O09BRUc7SUFDSSxZQUFZLENBQUMsS0FBYSxFQUFFLE1BQTRCO1FBQzdELElBQUksQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUFDLEdBQUcsTUFBTSxDQUFDO0lBQ2xDLENBQUM7SUFFRDs7T0FFRztJQUNJLGVBQWUsQ0FBQyxNQUFzQjtRQUMzQyx1Q0FBdUM7UUFDdkMsRUFBRTtRQUNGLGtGQUFrRjtRQUNsRix5RkFBeUY7UUFDekYsOEZBQThGO1FBQzlGLGtEQUFrRDtRQUNsRCxFQUFFO1FBQ0YsMENBQTBDO1FBQzFDLDRFQUE0RTtRQUM1RSxJQUFJLElBQUksQ0FBQyxNQUFNLEVBQUU7WUFDZixJQUFJLENBQUMsSUFBSSxDQUFDLFVBQVUsQ0FBQyx3R0FBd0csQ0FBQyxDQUFDO1NBQ2hJO1FBRUQsTUFBTSxDQUFDLElBQUksQ0FBQyxJQUFJLEVBQUUsSUFBSSxDQUFDLFFBQVEsRUFBRSxJQUFJLENBQUMsQ0FBQztRQUN2QyxJQUFJLENBQUMsTUFBTSxHQUFHLE1BQU0sQ0FBQztJQUN2QixDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLGNBQWMsQ0FBQyxXQUFvQztRQUN4RCxJQUFJLENBQUMsZUFBZSxDQUFDLDRDQUFjLENBQUMsT0FBTyxDQUFDLENBQUMsV0FBVyxDQUFDLENBQUMsQ0FBQyxDQUFDO0lBQzlELENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksZ0JBQWdCLENBQUMsYUFBNEI7UUFDbEQscUJBQXFCLENBQUMsYUFBYSxDQUFDLENBQUM7UUFFckMsSUFBSSxDQUFDLGVBQWUsQ0FBQyw0Q0FBYyxDQUFDLGFBQWEsQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLFFBQVEsQ0FBQyxhQUFhLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDOUYsV0FBVyxFQUFFLGFBQWEsQ0FBQyxXQUFXO1lBQ3RDLFdBQVcsRUFBRSxhQUFhLENBQUMsV0FBVztTQUN2QyxDQUFDLENBQUMsQ0FBQztJQUNOLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksbUJBQW1CLENBQUMsZ0JBQWtDO1FBQzNELHdCQUF3QixDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFFM0MsSUFBSSxDQUFDLGVBQWUsQ0FBQyw0Q0FBYyxDQUFDLFFBQVEsQ0FBQztZQUMzQyxJQUFJLEVBQUUsZ0JBQWdCLENBQUMsSUFBSTtZQUMzQixJQUFJLEVBQUUsZ0JBQWdCLENBQUMsSUFBSTtZQUMzQixTQUFTLEVBQUUsZ0JBQWdCLENBQUMsVUFBVSxLQUFLLFVBQVU7WUFDckQsSUFBSSxFQUFFLGdCQUFnQixDQUFDLElBQUk7WUFDM0IsUUFBUSxFQUFFLGdCQUFnQixDQUFDLFFBQVE7WUFDbkMsS0FBSyxFQUFFLGdCQUFnQixDQUFDLEtBQUs7U0FDOUIsQ0FBQyxDQUFDLENBQUM7SUFDTixDQUFDO0lBRUQ7O09BRUc7SUFDTyxRQUFRO1FBQ2hCLElBQUksSUFBSSxDQUFDLE1BQU0sS0FBSyxTQUFTLEVBQUU7WUFDN0IsT0FBTyxDQUFDLHlDQUF5QyxDQUFDLENBQUM7U0FDcEQ7UUFDRCxPQUFPLEVBQUUsQ0FBQztJQUNaLENBQUM7SUFFRDs7T0FFRztJQUNLLGdCQUFnQjtRQUN0QixNQUFNLEdBQUcsR0FBRyxJQUFJLEtBQUssRUFBdUMsQ0FBQztRQUM3RCxLQUFLLE1BQU0sQ0FBQyxLQUFLLEVBQUUsTUFBTSxDQUFDLElBQUksTUFBTSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsVUFBVSxDQUFDLEVBQUU7WUFDN0QsSUFBSSxNQUFNLEtBQUssU0FBUyxFQUFFO2dCQUN4QixHQUFHLENBQUMsSUFBSSxDQUFDLEVBQUUsS0FBSyxFQUFFLE1BQU0sRUFBRSxDQUFDLENBQUM7YUFDN0I7U0FDRjtRQUNELE9BQU8sR0FBRyxDQUFDO0lBQ2IsQ0FBQztDQUNGO0FBNUpELDBEQTRKQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFnQixxQkFBcUIsQ0FBQyxhQUE0QjtJQUNoRSxJQUFJLGFBQWEsQ0FBQyxVQUFVLElBQUksQ0FBQyxlQUFlLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxVQUFVLENBQUMsRUFBRTtRQUMvRSxNQUFNLElBQUksS0FBSyxDQUFDLHVDQUF1QyxDQUFDLENBQUM7S0FDMUQ7SUFFRCxJQUFJLGFBQWEsQ0FBQyxXQUFXLElBQUksYUFBYSxDQUFDLFdBQVcsQ0FBQyxNQUFNLEdBQUcsSUFBSSxFQUFFO1FBQ3hFLE1BQU0sSUFBSSxLQUFLLENBQUMsc0RBQXNELENBQUMsQ0FBQztLQUN6RTtBQUNILENBQUM7QUFSRCxzREFRQztBQUVEOzs7O0dBSUc7QUFDSCxTQUFnQix3QkFBd0IsQ0FBQyxnQkFBa0M7SUFDekUsSUFBSSxnQkFBZ0IsQ0FBQyxRQUFRLElBQUksQ0FBQywyQkFBMkIsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsUUFBUSxDQUFDLEVBQUU7UUFDN0YsTUFBTSxJQUFJLEtBQUssQ0FBQyxpREFBaUQsQ0FBQyxDQUFDO0tBQ3BFO0lBRUQsSUFBSSxDQUFDLGdCQUFnQixDQUFDLFVBQVUsSUFBSSxDQUFDLGVBQWUsQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsVUFBVSxDQUFDLEVBQUU7UUFDdEYsTUFBTSxJQUFJLEtBQUssQ0FBQyw0Q0FBNEMsQ0FBQyxDQUFDO0tBQy9EO0FBQ0gsQ0FBQztBQVJELDREQVFDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY2RrIGZyb20gJ0Bhd3MtY2RrL2NvcmUnO1xuaW1wb3J0IHsgQ2ZuTGlzdGVuZXJSdWxlIH0gZnJvbSAnLi4vZWxhc3RpY2xvYWRiYWxhbmNpbmd2Mi5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSUxpc3RlbmVyQWN0aW9uIH0gZnJvbSAnLi4vc2hhcmVkL2xpc3RlbmVyLWFjdGlvbic7XG5pbXBvcnQgeyBJQXBwbGljYXRpb25MaXN0ZW5lciB9IGZyb20gJy4vYXBwbGljYXRpb24tbGlzdGVuZXInO1xuaW1wb3J0IHsgTGlzdGVuZXJBY3Rpb24gfSBmcm9tICcuL2FwcGxpY2F0aW9uLWxpc3RlbmVyLWFjdGlvbic7XG5pbXBvcnQgeyBJQXBwbGljYXRpb25UYXJnZXRHcm91cCB9IGZyb20gJy4vYXBwbGljYXRpb24tdGFyZ2V0LWdyb3VwJztcblxuLyoqXG4gKiBCYXNpYyBwcm9wZXJ0aWVzIGZvciBkZWZpbmluZyBhIHJ1bGUgb24gYSBsaXN0ZW5lclxuICovXG5leHBvcnQgaW50ZXJmYWNlIEJhc2VBcHBsaWNhdGlvbkxpc3RlbmVyUnVsZVByb3BzIHtcbiAgLyoqXG4gICAqIFByaW9yaXR5IG9mIHRoZSBydWxlXG4gICAqXG4gICAqIFRoZSBydWxlIHdpdGggdGhlIGxvd2VzdCBwcmlvcml0eSB3aWxsIGJlIHVzZWQgZm9yIGV2ZXJ5IHJlcXVlc3QuXG4gICAqXG4gICAqIFByaW9yaXRpZXMgbXVzdCBiZSB1bmlxdWUuXG4gICAqL1xuICByZWFkb25seSBwcmlvcml0eTogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUYXJnZXQgZ3JvdXBzIHRvIGZvcndhcmQgcmVxdWVzdHMgdG8uXG4gICAqXG4gICAqIE9ubHkgb25lIG9mIGBhY3Rpb25gLCBgZml4ZWRSZXNwb25zZWAsIGByZWRpcmVjdFJlc3BvbnNlYCBvciBgdGFyZ2V0R3JvdXBzYCBjYW4gYmUgc3BlY2lmaWVkLlxuICAgKlxuICAgKiBJbXBsaWVzIGEgYGZvcndhcmRgIGFjdGlvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyB0YXJnZXQgZ3JvdXBzLlxuICAgKi9cbiAgcmVhZG9ubHkgdGFyZ2V0R3JvdXBzPzogSUFwcGxpY2F0aW9uVGFyZ2V0R3JvdXBbXTtcblxuICAvKipcbiAgICogQWN0aW9uIHRvIHBlcmZvcm0gd2hlbiByZXF1ZXN0cyBhcmUgcmVjZWl2ZWRcbiAgICpcbiAgICogT25seSBvbmUgb2YgYGFjdGlvbmAsIGBmaXhlZFJlc3BvbnNlYCwgYHJlZGlyZWN0UmVzcG9uc2VgIG9yIGB0YXJnZXRHcm91cHNgIGNhbiBiZSBzcGVjaWZpZWQuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gYWN0aW9uXG4gICAqL1xuICByZWFkb25seSBhY3Rpb24/OiBMaXN0ZW5lckFjdGlvbjtcblxuICAvKipcbiAgICogRml4ZWQgcmVzcG9uc2UgdG8gcmV0dXJuLlxuICAgKlxuICAgKiBPbmx5IG9uZSBvZiBgYWN0aW9uYCwgYGZpeGVkUmVzcG9uc2VgLCBgcmVkaXJlY3RSZXNwb25zZWAgb3IgYHRhcmdldEdyb3Vwc2AgY2FuIGJlIHNwZWNpZmllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBmaXhlZCByZXNwb25zZS5cbiAgICogQGRlcHJlY2F0ZWQgVXNlIGBhY3Rpb25gIGluc3RlYWQuXG4gICAqL1xuICByZWFkb25seSBmaXhlZFJlc3BvbnNlPzogRml4ZWRSZXNwb25zZTtcblxuICAvKipcbiAgICogUmVkaXJlY3QgcmVzcG9uc2UgdG8gcmV0dXJuLlxuICAgKlxuICAgKiBPbmx5IG9uZSBvZiBgYWN0aW9uYCwgYGZpeGVkUmVzcG9uc2VgLCBgcmVkaXJlY3RSZXNwb25zZWAgb3IgYHRhcmdldEdyb3Vwc2AgY2FuIGJlIHNwZWNpZmllZC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyByZWRpcmVjdCByZXNwb25zZS5cbiAgICogQGRlcHJlY2F0ZWQgVXNlIGBhY3Rpb25gIGluc3RlYWQuXG4gICAqL1xuICByZWFkb25seSByZWRpcmVjdFJlc3BvbnNlPzogUmVkaXJlY3RSZXNwb25zZTtcblxuICAvKipcbiAgICogUnVsZSBhcHBsaWVzIGlmIHRoZSByZXF1ZXN0ZWQgaG9zdCBtYXRjaGVzIHRoZSBpbmRpY2F0ZWQgaG9zdFxuICAgKlxuICAgKiBNYXkgY29udGFpbiB1cCB0byB0aHJlZSAnKicgd2lsZGNhcmRzLlxuICAgKlxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9lbGFzdGljbG9hZGJhbGFuY2luZy9sYXRlc3QvYXBwbGljYXRpb24vbG9hZC1iYWxhbmNlci1saXN0ZW5lcnMuaHRtbCNob3N0LWNvbmRpdGlvbnNcbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBob3N0IGNvbmRpdGlvbi5cbiAgICovXG4gIHJlYWRvbmx5IGhvc3RIZWFkZXI/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFJ1bGUgYXBwbGllcyBpZiB0aGUgcmVxdWVzdGVkIHBhdGggbWF0Y2hlcyB0aGUgZ2l2ZW4gcGF0aCBwYXR0ZXJuXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNsb2FkYmFsYW5jaW5nL2xhdGVzdC9hcHBsaWNhdGlvbi9sb2FkLWJhbGFuY2VyLWxpc3RlbmVycy5odG1sI3BhdGgtY29uZGl0aW9uc1xuICAgKiBAZGVmYXVsdCAtIE5vIHBhdGggY29uZGl0aW9uLlxuICAgKiBAZGVwcmVjYXRlZCBVc2UgYHBhdGhQYXR0ZXJuc2AgaW5zdGVhZC5cbiAgICovXG4gIHJlYWRvbmx5IHBhdGhQYXR0ZXJuPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBSdWxlIGFwcGxpZXMgaWYgdGhlIHJlcXVlc3RlZCBwYXRoIG1hdGNoZXMgYW55IG9mIHRoZSBnaXZlbiBwYXR0ZXJucy5cbiAgICpcbiAgICogUGF0aHMgbWF5IGNvbnRhaW4gdXAgdG8gdGhyZWUgJyonIHdpbGRjYXJkcy5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2xvYWRiYWxhbmNpbmcvbGF0ZXN0L2FwcGxpY2F0aW9uL2xvYWQtYmFsYW5jZXItbGlzdGVuZXJzLmh0bWwjcGF0aC1jb25kaXRpb25zXG4gICAqIEBkZWZhdWx0IC0gTm8gcGF0aCBjb25kaXRpb25zLlxuICAgKi9cbiAgcmVhZG9ubHkgcGF0aFBhdHRlcm5zPzogc3RyaW5nW107XG59XG5cbi8qKlxuICogUHJvcGVydGllcyBmb3IgZGVmaW5pbmcgYSBsaXN0ZW5lciBydWxlXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQXBwbGljYXRpb25MaXN0ZW5lclJ1bGVQcm9wcyBleHRlbmRzIEJhc2VBcHBsaWNhdGlvbkxpc3RlbmVyUnVsZVByb3BzIHtcbiAgLyoqXG4gICAqIFRoZSBsaXN0ZW5lciB0byBhdHRhY2ggdGhlIHJ1bGUgdG9cbiAgICovXG4gIHJlYWRvbmx5IGxpc3RlbmVyOiBJQXBwbGljYXRpb25MaXN0ZW5lcjtcbn1cblxuLyoqXG4gKiBUaGUgY29udGVudCB0eXBlIGZvciBhIGZpeGVkIHJlc3BvbnNlXG4gKi9cbmV4cG9ydCBlbnVtIENvbnRlbnRUeXBlIHtcbiAgVEVYVF9QTEFJTiA9ICd0ZXh0L3BsYWluJyxcbiAgVEVYVF9DU1MgPSAndGV4dC9jc3MnLFxuICBURVhUX0hUTUwgPSAgJ3RleHQvaHRtbCcsXG4gIEFQUExJQ0FUSU9OX0pBVkFTQ1JJUFQgPSAnYXBwbGljYXRpb24vamF2YXNjcmlwdCcsXG4gIEFQUExJQ0FUSU9OX0pTT04gPSAnYXBwbGljYXRpb24vanNvbidcbn1cblxuLyoqXG4gKiBBIGZpeGVkIHJlc3BvbnNlXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgRml4ZWRSZXNwb25zZSB7XG4gIC8qKlxuICAgKiBUaGUgSFRUUCByZXNwb25zZSBjb2RlICgyWFgsIDRYWCBvciA1WFgpXG4gICAqL1xuICByZWFkb25seSBzdGF0dXNDb2RlOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBjb250ZW50IHR5cGVcbiAgICpcbiAgICogQGRlZmF1bHQgdGV4dC9wbGFpblxuICAgKi9cbiAgcmVhZG9ubHkgY29udGVudFR5cGU/OiBDb250ZW50VHlwZTtcblxuICAvKipcbiAgICogVGhlIG1lc3NhZ2VcbiAgICpcbiAgICogQGRlZmF1bHQgbm8gbWVzc2FnZVxuICAgKi9cbiAgcmVhZG9ubHkgbWVzc2FnZUJvZHk/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogQSByZWRpcmVjdCByZXNwb25zZVxuICovXG5leHBvcnQgaW50ZXJmYWNlIFJlZGlyZWN0UmVzcG9uc2Uge1xuICAvKipcbiAgICogVGhlIGhvc3RuYW1lLiBUaGlzIGNvbXBvbmVudCBpcyBub3QgcGVyY2VudC1lbmNvZGVkLiBUaGUgaG9zdG5hbWUgY2FuIGNvbnRhaW4gI3tob3N0fS5cbiAgICpcbiAgICogQGRlZmF1bHQgb3JpZ2luIGhvc3Qgb2YgcmVxdWVzdFxuICAgKi9cbiAgcmVhZG9ubHkgaG9zdD86IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBhYnNvbHV0ZSBwYXRoLCBzdGFydGluZyB3aXRoIHRoZSBsZWFkaW5nIFwiL1wiLiBUaGlzIGNvbXBvbmVudCBpcyBub3QgcGVyY2VudC1lbmNvZGVkLlxuICAgKiBUaGUgcGF0aCBjYW4gY29udGFpbiAje2hvc3R9LCAje3BhdGh9LCBhbmQgI3twb3J0fS5cbiAgICpcbiAgICogQGRlZmF1bHQgb3JpZ2luIHBhdGggb2YgcmVxdWVzdFxuICAgKi9cbiAgcmVhZG9ubHkgcGF0aD86IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBwb3J0LiBZb3UgY2FuIHNwZWNpZnkgYSB2YWx1ZSBmcm9tIDEgdG8gNjU1MzUgb3IgI3twb3J0fS5cbiAgICpcbiAgICogQGRlZmF1bHQgb3JpZ2luIHBvcnQgb2YgcmVxdWVzdFxuICAgKi9cbiAgcmVhZG9ubHkgcG9ydD86IHN0cmluZztcbiAgLyoqXG4gICAqIFRoZSBwcm90b2NvbC4gWW91IGNhbiBzcGVjaWZ5IEhUVFAsIEhUVFBTLCBvciAje3Byb3RvY29sfS4gWW91IGNhbiByZWRpcmVjdCBIVFRQIHRvIEhUVFAsXG4gICAqIEhUVFAgdG8gSFRUUFMsIGFuZCBIVFRQUyB0byBIVFRQUy4gWW91IGNhbm5vdCByZWRpcmVjdCBIVFRQUyB0byBIVFRQLlxuICAgKlxuICAgKiBAZGVmYXVsdCBvcmlnaW4gcHJvdG9jb2wgb2YgcmVxdWVzdFxuICAgKi9cbiAgcmVhZG9ubHkgcHJvdG9jb2w/OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgcXVlcnkgcGFyYW1ldGVycywgVVJMLWVuY29kZWQgd2hlbiBuZWNlc3NhcnksIGJ1dCBub3QgcGVyY2VudC1lbmNvZGVkLlxuICAgKiBEbyBub3QgaW5jbHVkZSB0aGUgbGVhZGluZyBcIj9cIiwgYXMgaXQgaXMgYXV0b21hdGljYWxseSBhZGRlZC5cbiAgICogWW91IGNhbiBzcGVjaWZ5IGFueSBvZiB0aGUgcmVzZXJ2ZWQga2V5d29yZHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IG9yaWdpbiBxdWVyeSBzdHJpbmcgb2YgcmVxdWVzdFxuICAgKi9cbiAgcmVhZG9ubHkgcXVlcnk/OiBzdHJpbmc7XG4gIC8qKlxuICAgKiBUaGUgSFRUUCByZWRpcmVjdCBjb2RlIChIVFRQXzMwMSBvciBIVFRQXzMwMilcbiAgICovXG4gIHJlYWRvbmx5IHN0YXR1c0NvZGU6IHN0cmluZztcbn1cblxuLyoqXG4gKiBEZWZpbmUgYSBuZXcgbGlzdGVuZXIgcnVsZVxuICovXG5leHBvcnQgY2xhc3MgQXBwbGljYXRpb25MaXN0ZW5lclJ1bGUgZXh0ZW5kcyBjZGsuQ29uc3RydWN0IHtcbiAgLyoqXG4gICAqIFRoZSBBUk4gb2YgdGhpcyBydWxlXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgbGlzdGVuZXJSdWxlQXJuOiBzdHJpbmc7XG5cbiAgcHJpdmF0ZSByZWFkb25seSBjb25kaXRpb25zOiB7W2tleTogc3RyaW5nXTogc3RyaW5nW10gfCB1bmRlZmluZWR9ID0ge307XG5cbiAgcHJpdmF0ZSByZWFkb25seSBsaXN0ZW5lcjogSUFwcGxpY2F0aW9uTGlzdGVuZXI7XG4gIHByaXZhdGUgYWN0aW9uPzogSUxpc3RlbmVyQWN0aW9uO1xuXG4gIGNvbnN0cnVjdG9yKHNjb3BlOiBjZGsuQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogQXBwbGljYXRpb25MaXN0ZW5lclJ1bGVQcm9wcykge1xuICAgIHN1cGVyKHNjb3BlLCBpZCk7XG5cbiAgICBjb25zdCBoYXNQYXRoUGF0dGVybnMgPSBwcm9wcy5wYXRoUGF0dGVybnMgfHwgcHJvcHMucGF0aFBhdHRlcm47XG4gICAgaWYgKCFwcm9wcy5ob3N0SGVhZGVyICYmICFoYXNQYXRoUGF0dGVybnMpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQXQgbGVhc3Qgb25lIG9mIFxcJ2hvc3RIZWFkZXJcXCcsIFxcJ3BhdGhQYXR0ZXJuXFwnIG9yIFxcJ3BhdGhQYXR0ZXJuc1xcJyBpcyByZXF1aXJlZCB3aGVuIGRlZmluaW5nIGEgbG9hZCBiYWxhbmNpbmcgcnVsZS4nKTtcbiAgICB9XG5cbiAgICBjb25zdCBwb3NzaWJsZUFjdGlvbnM6IEFycmF5PGtleW9mIEFwcGxpY2F0aW9uTGlzdGVuZXJSdWxlUHJvcHM+ID0gWydhY3Rpb24nLCAndGFyZ2V0R3JvdXBzJywgJ2ZpeGVkUmVzcG9uc2UnLCAncmVkaXJlY3RSZXNwb25zZSddO1xuICAgIGNvbnN0IHByb3ZpZGVkQWN0aW9ucyA9IHBvc3NpYmxlQWN0aW9ucy5maWx0ZXIoYWN0aW9uID0+IHByb3BzW2FjdGlvbl0gIT09IHVuZGVmaW5lZCk7XG4gICAgaWYgKHByb3ZpZGVkQWN0aW9ucy5sZW5ndGggPiAxKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYCcke3Byb3ZpZGVkQWN0aW9uc30nIHNwZWNpZmllZCB0b2dldGhlciwgc3BlY2lmeSBvbmx5IG9uZWApO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5wcmlvcml0eSA8PSAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ1ByaW9yaXR5IG11c3QgaGF2ZSB2YWx1ZSBncmVhdGVyIHRoYW4gb3IgZXF1YWwgdG8gMScpO1xuICAgIH1cblxuICAgIHRoaXMubGlzdGVuZXIgPSBwcm9wcy5saXN0ZW5lcjtcblxuICAgIGNvbnN0IHJlc291cmNlID0gbmV3IENmbkxpc3RlbmVyUnVsZSh0aGlzLCAnUmVzb3VyY2UnLCB7XG4gICAgICBsaXN0ZW5lckFybjogcHJvcHMubGlzdGVuZXIubGlzdGVuZXJBcm4sXG4gICAgICBwcmlvcml0eTogcHJvcHMucHJpb3JpdHksXG4gICAgICBjb25kaXRpb25zOiBjZGsuTGF6eS5hbnlWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHRoaXMucmVuZGVyQ29uZGl0aW9ucygpIH0pLFxuICAgICAgYWN0aW9uczogY2RrLkxhenkuYW55VmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLmFjdGlvbiA/IHRoaXMuYWN0aW9uLnJlbmRlckFjdGlvbnMoKSA6IFtdIH0pLFxuICAgIH0pO1xuXG4gICAgaWYgKHByb3BzLmhvc3RIZWFkZXIpIHtcbiAgICAgIHRoaXMuc2V0Q29uZGl0aW9uKCdob3N0LWhlYWRlcicsIFtwcm9wcy5ob3N0SGVhZGVyXSk7XG4gICAgfVxuXG4gICAgaWYgKGhhc1BhdGhQYXR0ZXJucykge1xuICAgICAgaWYgKHByb3BzLnBhdGhQYXR0ZXJuICYmIHByb3BzLnBhdGhQYXR0ZXJucykge1xuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0JvdGggYHBhdGhQYXR0ZXJuc2AgYW5kIGBwYXRoUGF0dGVybmAgYXJlIHNwZWNpZmllZCwgc3BlY2lmeSBvbmx5IG9uZScpO1xuICAgICAgfVxuICAgICAgY29uc3QgcGF0aFBhdHRlcm4gPSBwcm9wcy5wYXRoUGF0dGVybiA/IFtwcm9wcy5wYXRoUGF0dGVybl0gOiBwcm9wcy5wYXRoUGF0dGVybnM7XG4gICAgICB0aGlzLnNldENvbmRpdGlvbigncGF0aC1wYXR0ZXJuJywgcGF0aFBhdHRlcm4pO1xuICAgIH1cblxuICAgIGlmIChwcm9wcy5hY3Rpb24pIHtcbiAgICAgIHRoaXMuY29uZmlndXJlQWN0aW9uKHByb3BzLmFjdGlvbik7XG4gICAgfVxuXG4gICAgKHByb3BzLnRhcmdldEdyb3VwcyB8fCBbXSkuZm9yRWFjaCh0aGlzLmFkZFRhcmdldEdyb3VwLmJpbmQodGhpcykpO1xuXG4gICAgaWYgKHByb3BzLmZpeGVkUmVzcG9uc2UpIHtcbiAgICAgIHRoaXMuYWRkRml4ZWRSZXNwb25zZShwcm9wcy5maXhlZFJlc3BvbnNlKTtcbiAgICB9IGVsc2UgaWYgKHByb3BzLnJlZGlyZWN0UmVzcG9uc2UpIHtcbiAgICAgIHRoaXMuYWRkUmVkaXJlY3RSZXNwb25zZShwcm9wcy5yZWRpcmVjdFJlc3BvbnNlKTtcbiAgICB9XG5cbiAgICB0aGlzLmxpc3RlbmVyUnVsZUFybiA9IHJlc291cmNlLnJlZjtcbiAgfVxuXG4gIC8qKlxuICAgKiBBZGQgYSBub24tc3RhbmRhcmQgY29uZGl0aW9uIHRvIHRoaXMgcnVsZVxuICAgKi9cbiAgcHVibGljIHNldENvbmRpdGlvbihmaWVsZDogc3RyaW5nLCB2YWx1ZXM6IHN0cmluZ1tdIHwgdW5kZWZpbmVkKSB7XG4gICAgdGhpcy5jb25kaXRpb25zW2ZpZWxkXSA9IHZhbHVlcztcbiAgfVxuXG4gIC8qKlxuICAgKiBDb25maWd1cmUgdGhlIGFjdGlvbiB0byBwZXJmb3JtIGZvciB0aGlzIHJ1bGVcbiAgICovXG4gIHB1YmxpYyBjb25maWd1cmVBY3Rpb24oYWN0aW9uOiBMaXN0ZW5lckFjdGlvbikge1xuICAgIC8vIEl0IG1pZ2h0IG1ha2Ugc2Vuc2UgdG8gJ3Rocm93JyBoZXJlLlxuICAgIC8vXG4gICAgLy8gSG93ZXZlciwgcHJvZ3JhbXMgbWF5IGFscmVhZHkgZXhpc3Qgb3V0IHRoZXJlIHdoaWNoIGNvbmZpZ3VyZWQgYW4gYWN0aW9uIHR3aWNlLFxuICAgIC8vIGluIHdoaWNoIGNhc2UgdGhlIHNlY29uZCBhY3Rpb24gYWNjaWRlbnRhbGx5IG92ZXJ3cml0ZSB0aGUgaW5pdGlhbCBhY3Rpb24sIGFuZCBpbiBzb21lXG4gICAgLy8gd2F5IGVuZGVkIHVwIHdpdGggYSBwcm9ncmFtIHRoYXQgZGlkIHdoYXQgdGhlIGF1dGhvciBpbnRlbmRlZC4gSWYgd2Ugd2VyZSB0byBhZGQgdGhyb3cgbm93LFxuICAgIC8vIHRoZSBwcmV2aW91c2x5IHdvcmtpbmcgcHJvZ3JhbSB3b3VsZCBiZSBicm9rZW4uXG4gICAgLy9cbiAgICAvLyBJbnN0ZWFkLCBzaWduYWwgdGhpcyB0aHJvdWdoIGEgd2FybmluZy5cbiAgICAvLyBAZGVwcmVjYXRlOiB1cG9uIHRoZSBuZXh0IG1ham9yIHZlcnNpb24gYnVtcCwgcmVwbGFjZSB0aGlzIHdpdGggYSBgdGhyb3dgXG4gICAgaWYgKHRoaXMuYWN0aW9uKSB7XG4gICAgICB0aGlzLm5vZGUuYWRkV2FybmluZygnQW4gQWN0aW9uIGFscmVhZHkgZXhpc3RlZCBvbiB0aGlzIExpc3RlbmVyUnVsZSBhbmQgd2FzIHJlcGxhY2VkLiBDb25maWd1cmUgZXhhY3RseSBvbmUgZGVmYXVsdCBBY3Rpb24uJyk7XG4gICAgfVxuXG4gICAgYWN0aW9uLmJpbmQodGhpcywgdGhpcy5saXN0ZW5lciwgdGhpcyk7XG4gICAgdGhpcy5hY3Rpb24gPSBhY3Rpb247XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgVGFyZ2V0R3JvdXAgdG8gbG9hZCBiYWxhbmNlIHRvXG4gICAqXG4gICAqIEBkZXByZWNhdGVkIFVzZSBjb25maWd1cmVBY3Rpb24gaW5zdGVhZFxuICAgKi9cbiAgcHVibGljIGFkZFRhcmdldEdyb3VwKHRhcmdldEdyb3VwOiBJQXBwbGljYXRpb25UYXJnZXRHcm91cCkge1xuICAgIHRoaXMuY29uZmlndXJlQWN0aW9uKExpc3RlbmVyQWN0aW9uLmZvcndhcmQoW3RhcmdldEdyb3VwXSkpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIGZpeGVkIHJlc3BvbnNlXG4gICAqXG4gICAqIEBkZXByZWNhdGVkIFVzZSBjb25maWd1cmVBY3Rpb24gaW5zdGVhZFxuICAgKi9cbiAgcHVibGljIGFkZEZpeGVkUmVzcG9uc2UoZml4ZWRSZXNwb25zZTogRml4ZWRSZXNwb25zZSkge1xuICAgIHZhbGlkYXRlRml4ZWRSZXNwb25zZShmaXhlZFJlc3BvbnNlKTtcblxuICAgIHRoaXMuY29uZmlndXJlQWN0aW9uKExpc3RlbmVyQWN0aW9uLmZpeGVkUmVzcG9uc2UoY2RrLlRva2VuLmFzTnVtYmVyKGZpeGVkUmVzcG9uc2Uuc3RhdHVzQ29kZSksIHtcbiAgICAgIGNvbnRlbnRUeXBlOiBmaXhlZFJlc3BvbnNlLmNvbnRlbnRUeXBlLFxuICAgICAgbWVzc2FnZUJvZHk6IGZpeGVkUmVzcG9uc2UubWVzc2FnZUJvZHksXG4gICAgfSkpO1xuICB9XG5cbiAgLyoqXG4gICAqIEFkZCBhIHJlZGlyZWN0IHJlc3BvbnNlXG4gICAqXG4gICAqIEBkZXByZWNhdGVkIFVzZSBjb25maWd1cmVBY3Rpb24gaW5zdGVhZFxuICAgKi9cbiAgcHVibGljIGFkZFJlZGlyZWN0UmVzcG9uc2UocmVkaXJlY3RSZXNwb25zZTogUmVkaXJlY3RSZXNwb25zZSkge1xuICAgIHZhbGlkYXRlUmVkaXJlY3RSZXNwb25zZShyZWRpcmVjdFJlc3BvbnNlKTtcblxuICAgIHRoaXMuY29uZmlndXJlQWN0aW9uKExpc3RlbmVyQWN0aW9uLnJlZGlyZWN0KHtcbiAgICAgIGhvc3Q6IHJlZGlyZWN0UmVzcG9uc2UuaG9zdCxcbiAgICAgIHBhdGg6IHJlZGlyZWN0UmVzcG9uc2UucGF0aCxcbiAgICAgIHBlcm1hbmVudDogcmVkaXJlY3RSZXNwb25zZS5zdGF0dXNDb2RlID09PSAnSFRUUF8zMDEnLFxuICAgICAgcG9ydDogcmVkaXJlY3RSZXNwb25zZS5wb3J0LFxuICAgICAgcHJvdG9jb2w6IHJlZGlyZWN0UmVzcG9uc2UucHJvdG9jb2wsXG4gICAgICBxdWVyeTogcmVkaXJlY3RSZXNwb25zZS5xdWVyeSxcbiAgICB9KSk7XG4gIH1cblxuICAvKipcbiAgICogVmFsaWRhdGUgdGhlIHJ1bGVcbiAgICovXG4gIHByb3RlY3RlZCB2YWxpZGF0ZSgpIHtcbiAgICBpZiAodGhpcy5hY3Rpb24gPT09IHVuZGVmaW5lZCkge1xuICAgICAgcmV0dXJuIFsnTGlzdGVuZXIgcnVsZSBuZWVkcyBhdCBsZWFzdCBvbmUgYWN0aW9uJ107XG4gICAgfVxuICAgIHJldHVybiBbXTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW5kZXIgdGhlIGNvbmRpdGlvbnMgZm9yIHRoaXMgcnVsZVxuICAgKi9cbiAgcHJpdmF0ZSByZW5kZXJDb25kaXRpb25zKCkge1xuICAgIGNvbnN0IHJldCA9IG5ldyBBcnJheTx7IGZpZWxkOiBzdHJpbmcsIHZhbHVlczogc3RyaW5nW10gfT4oKTtcbiAgICBmb3IgKGNvbnN0IFtmaWVsZCwgdmFsdWVzXSBvZiBPYmplY3QuZW50cmllcyh0aGlzLmNvbmRpdGlvbnMpKSB7XG4gICAgICBpZiAodmFsdWVzICE9PSB1bmRlZmluZWQpIHtcbiAgICAgICAgcmV0LnB1c2goeyBmaWVsZCwgdmFsdWVzIH0pO1xuICAgICAgfVxuICAgIH1cbiAgICByZXR1cm4gcmV0O1xuICB9XG59XG5cbi8qKlxuICogVmFsaWRhdGUgdGhlIHN0YXR1cyBjb2RlIGFuZCBtZXNzYWdlIGJvZHkgb2YgYSBmaXhlZCByZXNwb25zZVxuICpcbiAqIEBpbnRlcm5hbFxuICovXG5leHBvcnQgZnVuY3Rpb24gdmFsaWRhdGVGaXhlZFJlc3BvbnNlKGZpeGVkUmVzcG9uc2U6IEZpeGVkUmVzcG9uc2UpIHtcbiAgaWYgKGZpeGVkUmVzcG9uc2Uuc3RhdHVzQ29kZSAmJiAhL14oMnw0fDUpXFxkXFxkJC8udGVzdChmaXhlZFJlc3BvbnNlLnN0YXR1c0NvZGUpKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKCdgc3RhdHVzQ29kZWAgbXVzdCBiZSAyWFgsIDRYWCBvciA1WFguJyk7XG4gIH1cblxuICBpZiAoZml4ZWRSZXNwb25zZS5tZXNzYWdlQm9keSAmJiBmaXhlZFJlc3BvbnNlLm1lc3NhZ2VCb2R5Lmxlbmd0aCA+IDEwMjQpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoJ2BtZXNzYWdlQm9keWAgY2Fubm90IGhhdmUgbW9yZSB0aGFuIDEwMjQgY2hhcmFjdGVycy4nKTtcbiAgfVxufVxuXG4vKipcbiAqIFZhbGlkYXRlIHRoZSBzdGF0dXMgY29kZSBhbmQgbWVzc2FnZSBib2R5IG9mIGEgcmVkaXJlY3QgcmVzcG9uc2VcbiAqXG4gKiBAaW50ZXJuYWxcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIHZhbGlkYXRlUmVkaXJlY3RSZXNwb25zZShyZWRpcmVjdFJlc3BvbnNlOiBSZWRpcmVjdFJlc3BvbnNlKSB7XG4gIGlmIChyZWRpcmVjdFJlc3BvbnNlLnByb3RvY29sICYmICEvXihIVFRQUz98I1xce3Byb3RvY29sXFx9KSQvaS50ZXN0KHJlZGlyZWN0UmVzcG9uc2UucHJvdG9jb2wpKSB7XG4gICAgdGhyb3cgbmV3IEVycm9yKCdgcHJvdG9jb2xgIG11c3QgYmUgSFRUUCwgSFRUUFMsIG9yICN7cHJvdG9jb2x9LicpO1xuICB9XG5cbiAgaWYgKCFyZWRpcmVjdFJlc3BvbnNlLnN0YXR1c0NvZGUgfHwgIS9eSFRUUF8zMFsxMl0kLy50ZXN0KHJlZGlyZWN0UmVzcG9uc2Uuc3RhdHVzQ29kZSkpIHtcbiAgICB0aHJvdyBuZXcgRXJyb3IoJ2BzdGF0dXNDb2RlYCBtdXN0IGJlIEhUVFBfMzAxIG9yIEhUVFBfMzAyLicpO1xuICB9XG59XG4iXX0=