"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const base_listener_1 = require("../shared/base-listener");
const enums_1 = require("../shared/enums");
const listener_certificate_1 = require("../shared/listener-certificate");
const util_1 = require("../shared/util");
const application_listener_action_1 = require("./application-listener-action");
const application_listener_certificate_1 = require("./application-listener-certificate");
const application_listener_rule_1 = require("./application-listener-rule");
const application_target_group_1 = require("./application-target-group");
/**
 * Define an ApplicationListener
 *
 * @resource AWS::ElasticLoadBalancingV2::Listener
 */
class ApplicationListener extends base_listener_1.BaseListener {
    constructor(scope, id, props) {
        const [protocol, port] = util_1.determineProtocolAndPort(props.protocol, props.port);
        if (protocol === undefined || port === undefined) {
            throw new Error('At least one of \'port\' or \'protocol\' is required');
        }
        super(scope, id, {
            loadBalancerArn: props.loadBalancer.loadBalancerArn,
            certificates: core_1.Lazy.anyValue({ produce: () => this.certificateArns.map(certificateArn => ({ certificateArn })) }, { omitEmptyArray: true }),
            protocol,
            port,
            sslPolicy: props.sslPolicy,
        });
        this.loadBalancer = props.loadBalancer;
        this.protocol = protocol;
        this.certificateArns = [];
        // Attach certificates
        if (props.certificateArns && props.certificateArns.length > 0) {
            this.addCertificateArns('ListenerCertificate', props.certificateArns);
        }
        if (props.certificates && props.certificates.length > 0) {
            this.addCertificates('DefaultCertificates', props.certificates);
        }
        // This listener edits the securitygroup of the load balancer,
        // but adds its own default port.
        this.connections = new ec2.Connections({
            securityGroups: props.loadBalancer.connections.securityGroups,
            defaultPort: ec2.Port.tcp(port),
        });
        if (props.defaultAction && props.defaultTargetGroups) {
            throw new Error('Specify at most one of \'defaultAction\' and \'defaultTargetGroups\'');
        }
        if (props.defaultAction) {
            this.setDefaultAction(props.defaultAction);
        }
        if (props.defaultTargetGroups) {
            this.setDefaultAction(application_listener_action_1.ListenerAction.forward(props.defaultTargetGroups));
        }
        if (props.open !== false) {
            this.connections.allowDefaultPortFrom(ec2.Peer.anyIpv4(), `Allow from anyone on port ${port}`);
        }
    }
    /**
     * Import an existing listener
     */
    static fromApplicationListenerAttributes(scope, id, attrs) {
        return new ImportedApplicationListener(scope, id, attrs);
    }
    /**
     * Add one or more certificates to this listener.
     *
     * After the first certificate, this creates ApplicationListenerCertificates
     * resources since cloudformation requires the certificates array on the
     * listener resource to have a length of 1.
     *
     * @deprecated Use `addCertificates` instead.
     */
    addCertificateArns(id, arns) {
        this.addCertificates(id, arns.map(listener_certificate_1.ListenerCertificate.fromArn));
    }
    /**
     * Add one or more certificates to this listener.
     *
     * After the first certificate, this creates ApplicationListenerCertificates
     * resources since cloudformation requires the certificates array on the
     * listener resource to have a length of 1.
     */
    addCertificates(id, certificates) {
        const additionalCerts = [...certificates];
        if (this.certificateArns.length === 0 && additionalCerts.length > 0) {
            const first = additionalCerts.splice(0, 1)[0];
            this.certificateArns.push(first.certificateArn);
        }
        if (additionalCerts.length > 0) {
            new application_listener_certificate_1.ApplicationListenerCertificate(this, id, {
                listener: this,
                certificates: additionalCerts,
            });
        }
    }
    /**
     * Perform the given default action on incoming requests
     *
     * This allows full control of the default action of the load balancer,
     * including Action chaining, fixed responses and redirect responses. See
     * the `ListenerAction` class for all options.
     *
     * It's possible to add routing conditions to the Action added in this way.
     * At least one Action must be added without conditions (which becomes the
     * default Action).
     */
    addAction(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            //
            // TargetGroup.registerListener is called inside ApplicationListenerRule.
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                hostHeader: props.hostHeader,
                pathPattern: props.pathPattern,
                pathPatterns: props.pathPatterns,
                priority: props.priority,
                action: props.action,
            });
        }
        else {
            // New default target with these targetgroups
            this.setDefaultAction(props.action);
        }
    }
    /**
     * Load balance incoming requests to the given target groups.
     *
     * All target groups will be load balanced to with equal weight and without
     * stickiness. For a more complex configuration than that, use `addAction()`.
     *
     * It's possible to add routing conditions to the TargetGroups added in this
     * way. At least one TargetGroup must be added without conditions (which will
     * become the default Action for this listener).
     */
    addTargetGroups(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            //
            // TargetGroup.registerListener is called inside ApplicationListenerRule.
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                hostHeader: props.hostHeader,
                pathPattern: props.pathPattern,
                pathPatterns: props.pathPatterns,
                priority: props.priority,
                targetGroups: props.targetGroups,
            });
        }
        else {
            // New default target with these targetgroups
            this.setDefaultAction(application_listener_action_1.ListenerAction.forward(props.targetGroups));
        }
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates an ApplicationTargetGroup for the targets
     * involved, and a 'forward' action to route traffic to the given TargetGroup.
     *
     * If you want more control over the precise setup, create the TargetGroup
     * and use `addAction` yourself.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     */
    addTargets(id, props) {
        if (!this.loadBalancer.vpc) {
            // tslint:disable-next-line:max-line-length
            throw new Error('Can only call addTargets() when using a constructed Load Balancer or an imported Load Balancer with specified vpc; construct a new TargetGroup and use addTargetGroup');
        }
        const group = new application_target_group_1.ApplicationTargetGroup(this, id + 'Group', {
            deregistrationDelay: props.deregistrationDelay,
            healthCheck: props.healthCheck,
            port: props.port,
            protocol: props.protocol,
            slowStart: props.slowStart,
            stickinessCookieDuration: props.stickinessCookieDuration,
            targetGroupName: props.targetGroupName,
            targets: props.targets,
            vpc: this.loadBalancer.vpc,
        });
        this.addTargetGroups(id, {
            hostHeader: props.hostHeader,
            pathPattern: props.pathPattern,
            pathPatterns: props.pathPatterns,
            priority: props.priority,
            targetGroups: [group],
        });
        return group;
    }
    /**
     * Add a fixed response
     *
     * @deprecated Use `addAction()` instead
     */
    addFixedResponse(id, props) {
        checkAddRuleProps(props);
        const fixedResponse = {
            statusCode: props.statusCode,
            contentType: props.contentType,
            messageBody: props.messageBody,
        };
        application_listener_rule_1.validateFixedResponse(fixedResponse);
        if (props.priority) {
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                fixedResponse,
                ...props,
            });
        }
        else {
            this.setDefaultAction(application_listener_action_1.ListenerAction.fixedResponse(core_1.Token.asNumber(props.statusCode), {
                contentType: props.contentType,
                messageBody: props.messageBody,
            }));
        }
    }
    /**
     * Add a redirect response
     *
     * @deprecated Use `addAction()` instead
     */
    addRedirectResponse(id, props) {
        checkAddRuleProps(props);
        const redirectResponse = {
            host: props.host,
            path: props.path,
            port: props.port,
            protocol: props.protocol,
            query: props.query,
            statusCode: props.statusCode,
        };
        application_listener_rule_1.validateRedirectResponse(redirectResponse);
        if (props.priority) {
            new application_listener_rule_1.ApplicationListenerRule(this, id + 'Rule', {
                listener: this,
                priority: props.priority,
                redirectResponse,
                ...props,
            });
        }
        else {
            this.setDefaultAction(application_listener_action_1.ListenerAction.redirect({
                host: props.host,
                path: props.path,
                port: props.port,
                protocol: props.protocol,
                query: props.query,
                permanent: props.statusCode === 'HTTP_301',
            }));
        }
    }
    /**
     * Register that a connectable that has been added to this load balancer.
     *
     * Don't call this directly. It is called by ApplicationTargetGroup.
     */
    registerConnectable(connectable, portRange) {
        connectable.connections.allowFrom(this.loadBalancer, portRange, 'Load balancer to target');
    }
    /**
     * Validate this listener.
     */
    validate() {
        const errors = super.validate();
        if (this.protocol === enums_1.ApplicationProtocol.HTTPS && this.certificateArns.length === 0) {
            errors.push('HTTPS Listener needs at least one certificate (call addCertificates)');
        }
        return errors;
    }
    /**
     * Wrapper for _setDefaultAction which does a type-safe bind
     */
    setDefaultAction(action) {
        action.bind(this, this);
        this._setDefaultAction(action);
    }
}
exports.ApplicationListener = ApplicationListener;
class ImportedApplicationListener extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.listenerArn = props.listenerArn;
        const defaultPort = props.defaultPort !== undefined ? ec2.Port.tcp(props.defaultPort) : undefined;
        let securityGroup;
        if (props.securityGroup) {
            securityGroup = props.securityGroup;
        }
        else if (props.securityGroupId) {
            securityGroup = ec2.SecurityGroup.fromSecurityGroupId(scope, 'SecurityGroup', props.securityGroupId, {
                allowAllOutbound: props.securityGroupAllowsAllOutbound,
            });
        }
        else {
            throw new Error('Either `securityGroup` or `securityGroupId` must be specified to import an application listener.');
        }
        this.connections = new ec2.Connections({
            securityGroups: [securityGroup],
            defaultPort,
        });
    }
    /**
     * Add one or more certificates to this listener.
     */
    addCertificateArns(id, arns) {
        new application_listener_certificate_1.ApplicationListenerCertificate(this, id, {
            listener: this,
            certificateArns: arns,
        });
    }
    /**
     * Load balance incoming requests to the given target groups.
     *
     * It's possible to add conditions to the TargetGroups added in this way.
     * At least one TargetGroup must be added without conditions.
     */
    addTargetGroups(id, props) {
        checkAddRuleProps(props);
        if (props.priority !== undefined) {
            // New rule
            new application_listener_rule_1.ApplicationListenerRule(this, id, {
                listener: this,
                hostHeader: props.hostHeader,
                pathPattern: props.pathPattern,
                pathPatterns: props.pathPatterns,
                priority: props.priority,
                targetGroups: props.targetGroups,
            });
        }
        else {
            throw new Error('Cannot add default Target Groups to imported ApplicationListener');
        }
    }
    /**
     * Load balance incoming requests to the given load balancing targets.
     *
     * This method implicitly creates an ApplicationTargetGroup for the targets
     * involved.
     *
     * It's possible to add conditions to the targets added in this way. At least
     * one set of targets must be added without conditions.
     *
     * @returns The newly created target group
     */
    addTargets(_id, _props) {
        // tslint:disable-next-line:max-line-length
        throw new Error('Can only call addTargets() when using a constructed ApplicationListener; construct a new TargetGroup and use addTargetGroup.');
    }
    /**
     * Register that a connectable that has been added to this load balancer.
     *
     * Don't call this directly. It is called by ApplicationTargetGroup.
     */
    registerConnectable(connectable, portRange) {
        this.connections.allowTo(connectable, portRange, 'Load balancer to target');
    }
}
function checkAddRuleProps(props) {
    if ((props.hostHeader !== undefined || props.pathPattern !== undefined || props.pathPatterns !== undefined) !== (props.priority !== undefined)) {
        throw new Error('Setting \'pathPattern\' or \'hostHeader\' also requires \'priority\', and vice versa');
    }
}
//# sourceMappingURL=data:application/json;base64,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