"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
/**
 * What to do when a client makes a request to a listener
 *
 * Some actions can be combined with other ones (specifically,
 * you can perform authentication before serving the request).
 *
 * Multiple actions form a linked chain; the chain must always terminate in a
 * *(weighted)forward*, *fixedResponse* or *redirect* action.
 *
 * If an action supports chaining, the next action can be indicated
 * by passing it in the `next` property.
 */
class NetworkListenerAction {
    /**
     * Create an instance of NetworkListenerAction
     *
     * The default class should be good enough for most cases and
     * should be created by using one of the static factory functions,
     * but allow overriding to make sure we allow flexibility for the future.
     */
    constructor(actionJson, next) {
        this.actionJson = actionJson;
        this.next = next;
    }
    /**
     * Forward to one or more Target Groups
     */
    static forward(targetGroups, options = {}) {
        if (targetGroups.length === 0) {
            throw new Error('Need at least one targetGroup in a NetworkListenerAction.forward()');
        }
        if (targetGroups.length === 1 && options.stickinessDuration === undefined) {
            // Render a "simple" action for backwards compatibility with old templates
            return new TargetGroupListenerAction(targetGroups, {
                type: 'forward',
                targetGroupArn: targetGroups[0].targetGroupArn,
            });
        }
        return new TargetGroupListenerAction(targetGroups, {
            type: 'forward',
            forwardConfig: {
                targetGroups: targetGroups.map(g => ({ targetGroupArn: g.targetGroupArn })),
                targetGroupStickinessConfig: options.stickinessDuration ? {
                    durationSeconds: options.stickinessDuration.toSeconds(),
                    enabled: true,
                } : undefined,
            },
        });
    }
    /**
     * Forward to one or more Target Groups which are weighted differently
     */
    static weightedForward(targetGroups, options = {}) {
        if (targetGroups.length === 0) {
            throw new Error('Need at least one targetGroup in a NetworkListenerAction.weightedForward()');
        }
        return new TargetGroupListenerAction(targetGroups.map(g => g.targetGroup), {
            type: 'forward',
            forwardConfig: {
                targetGroups: targetGroups.map(g => ({ targetGroupArn: g.targetGroup.targetGroupArn, weight: g.weight })),
                targetGroupStickinessConfig: options.stickinessDuration ? {
                    durationSeconds: options.stickinessDuration.toSeconds(),
                    enabled: true,
                } : undefined,
            },
        });
    }
    /**
     * Render the actions in this chain
     */
    renderActions() {
        var _a, _b;
        return this.renumber([this.actionJson, ...(_b = (_a = this.next) === null || _a === void 0 ? void 0 : _a.renderActions()) !== null && _b !== void 0 ? _b : []]);
    }
    /**
     * Called when the action is being used in a listener
     */
    bind(scope, listener) {
        // Empty on purpose
        Array.isArray(scope);
        Array.isArray(listener);
    }
    /**
     * Renumber the "order" fields in the actions array.
     *
     * We don't number for 0 or 1 elements, but otherwise number them 1...#actions
     * so ELB knows about the right order.
     *
     * Do this in `NetworkListenerAction` instead of in `Listener` so that we give
     * users the opportunity to override by subclassing and overriding `renderActions`.
     */
    renumber(actions) {
        if (actions.length < 2) {
            return actions;
        }
        return actions.map((action, i) => ({ ...action, order: i + 1 }));
    }
}
exports.NetworkListenerAction = NetworkListenerAction;
/**
 * Listener Action that calls "registerListener" on TargetGroups
 */
class TargetGroupListenerAction extends NetworkListenerAction {
    constructor(targetGroups, actionJson) {
        super(actionJson);
        this.targetGroups = targetGroups;
    }
    bind(_scope, listener) {
        for (const tg of this.targetGroups) {
            tg.registerListener(listener);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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