"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ec2 = require("@aws-cdk/aws-ec2");
const iam = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
const elasticloadbalancingv2_generated_1 = require("../elasticloadbalancingv2.generated");
const util_1 = require("./util");
/**
 * Base class for both Application and Network Load Balancers
 */
class BaseLoadBalancer extends core_1.Resource {
    constructor(scope, id, baseProps, additionalProps) {
        super(scope, id, {
            physicalName: baseProps.loadBalancerName,
        });
        /**
         * Attributes set on this load balancer
         */
        this.attributes = {};
        const internetFacing = util_1.ifUndefined(baseProps.internetFacing, false);
        const vpcSubnets = util_1.ifUndefined(baseProps.vpcSubnets, (internetFacing ? { subnetType: ec2.SubnetType.PUBLIC } : {}));
        const { subnetIds, internetConnectivityEstablished } = baseProps.vpc.selectSubnets(vpcSubnets);
        this.vpc = baseProps.vpc;
        const resource = new elasticloadbalancingv2_generated_1.CfnLoadBalancer(this, 'Resource', {
            name: this.physicalName,
            subnets: subnetIds,
            scheme: internetFacing ? 'internet-facing' : 'internal',
            loadBalancerAttributes: core_1.Lazy.anyValue({ produce: () => util_1.renderAttributes(this.attributes) }, { omitEmptyArray: true }),
            ...additionalProps,
        });
        if (internetFacing) {
            resource.node.addDependency(internetConnectivityEstablished);
        }
        if (baseProps.deletionProtection) {
            this.setAttribute('deletion_protection.enabled', 'true');
        }
        this.loadBalancerCanonicalHostedZoneId = resource.attrCanonicalHostedZoneId;
        this.loadBalancerDnsName = resource.attrDnsName;
        this.loadBalancerFullName = resource.attrLoadBalancerFullName;
        this.loadBalancerName = resource.attrLoadBalancerName;
        this.loadBalancerArn = resource.ref;
        this.loadBalancerSecurityGroups = resource.attrSecurityGroups;
    }
    /**
     * Enable access logging for this load balancer.
     *
     * A region must be specified on the stack containing the load balancer; you cannot enable logging on
     * environment-agnostic stacks. See https://docs.aws.amazon.com/cdk/latest/guide/environments.html
     */
    logAccessLogs(bucket, prefix) {
        this.setAttribute('access_logs.s3.enabled', 'true');
        this.setAttribute('access_logs.s3.bucket', bucket.bucketName.toString());
        this.setAttribute('access_logs.s3.prefix', prefix);
        const region = core_1.Stack.of(this).region;
        if (core_1.Token.isUnresolved(region)) {
            throw new Error('Region is required to enable ELBv2 access logging');
        }
        const account = ELBV2_ACCOUNTS[region];
        if (!account) {
            throw new Error(`Cannot enable access logging; don't know ELBv2 account for region ${region}`);
        }
        prefix = prefix || '';
        bucket.grantPut(new iam.AccountPrincipal(account), `${(prefix ? prefix + '/' : '')}AWSLogs/${core_1.Stack.of(this).account}/*`);
        // make sure the bucket's policy is created before the ALB (see https://github.com/aws/aws-cdk/issues/1633)
        this.node.addDependency(bucket);
    }
    /**
     * Set a non-standard attribute on the load balancer
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/application-load-balancers.html#load-balancer-attributes
     */
    setAttribute(key, value) {
        this.attributes[key] = value;
    }
    /**
     * Remove an attribute from the load balancer
     */
    removeAttribute(key) {
        this.setAttribute(key, undefined);
    }
}
exports.BaseLoadBalancer = BaseLoadBalancer;
// https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-access-logs.html#access-logging-bucket-permissions
const ELBV2_ACCOUNTS = {
    'us-east-1': '127311923021',
    'us-east-2': '033677994240',
    'us-west-1': '027434742980',
    'us-west-2': '797873946194',
    'ca-central-1': '985666609251',
    'eu-central-1': '054676820928',
    'eu-west-1': '156460612806',
    'eu-west-2': '652711504416',
    'eu-west-3': '009996457667',
    'eu-north-1': '897822967062',
    'ap-east-1': '754344448648',
    'ap-northeast-1': '582318560864',
    'ap-northeast-2': '600734575887',
    'ap-northeast-3': '383597477331',
    'ap-southeast-1': '114774131450',
    'ap-southeast-2': '783225319266',
    'ap-south-1': '718504428378',
    'me-south-1': '076674570225',
    'sa-east-1': '507241528517',
    'us-gov-west-1': '048591011584',
    'us-gov-east-1': '190560391635',
    'cn-north-1': '638102146993',
    'cn-northwest-1': '037604701340',
};
//# sourceMappingURL=data:application/json;base64,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