"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const elbv2 = require("../../lib");
let stack;
let group1;
let group2;
let lb;
module.exports = {
    'setUp'(cb) {
        stack = new cdk.Stack();
        const vpc = new ec2.Vpc(stack, 'Stack');
        group1 = new elbv2.ApplicationTargetGroup(stack, 'TargetGroup1', { vpc, port: 80 });
        group2 = new elbv2.ApplicationTargetGroup(stack, 'TargetGroup2', { vpc, port: 80 });
        lb = new elbv2.ApplicationLoadBalancer(stack, 'LB', { vpc });
        cb();
    },
    'Forward action legacy rendering'(test) {
        // WHEN
        lb.addListener('Listener', {
            port: 80,
            defaultAction: elbv2.ListenerAction.forward([group1]),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    TargetGroupArn: { Ref: 'TargetGroup1E5480F51' },
                    Type: 'forward',
                },
            ],
        }));
        test.done();
    },
    'Forward to multiple targetgroups with an Action and stickiness'(test) {
        // WHEN
        lb.addListener('Listener', {
            port: 80,
            defaultAction: elbv2.ListenerAction.forward([group1, group2], {
                stickinessDuration: cdk.Duration.hours(1),
            }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    ForwardConfig: {
                        TargetGroupStickinessConfig: {
                            DurationSeconds: 3600,
                            Enabled: true,
                        },
                        TargetGroups: [
                            {
                                TargetGroupArn: { Ref: 'TargetGroup1E5480F51' },
                            },
                            {
                                TargetGroupArn: { Ref: 'TargetGroup2D571E5D7' },
                            },
                        ],
                    },
                    Type: 'forward',
                },
            ],
        }));
        test.done();
    },
    'Weighted forward to multiple targetgroups with an Action'(test) {
        // WHEN
        lb.addListener('Listener', {
            port: 80,
            defaultAction: elbv2.ListenerAction.weightedForward([
                { targetGroup: group1, weight: 10 },
                { targetGroup: group2, weight: 50 },
            ], {
                stickinessDuration: cdk.Duration.hours(1),
            }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    ForwardConfig: {
                        TargetGroupStickinessConfig: {
                            DurationSeconds: 3600,
                            Enabled: true,
                        },
                        TargetGroups: [
                            {
                                TargetGroupArn: { Ref: 'TargetGroup1E5480F51' },
                                Weight: 10,
                            },
                            {
                                TargetGroupArn: { Ref: 'TargetGroup2D571E5D7' },
                                Weight: 50,
                            },
                        ],
                    },
                    Type: 'forward',
                },
            ],
        }));
        test.done();
    },
    'Chaining OIDC authentication action'(test) {
        // WHEN
        lb.addListener('Listener', {
            port: 80,
            defaultAction: elbv2.ListenerAction.authenticateOidc({
                authorizationEndpoint: 'A',
                clientId: 'B',
                clientSecret: cdk.SecretValue.plainText('C'),
                issuer: 'D',
                tokenEndpoint: 'E',
                userInfoEndpoint: 'F',
                next: elbv2.ListenerAction.forward([group1]),
            }),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::Listener', {
            DefaultActions: [
                {
                    AuthenticateOidcConfig: {
                        AuthorizationEndpoint: 'A',
                        ClientId: 'B',
                        ClientSecret: 'C',
                        Issuer: 'D',
                        TokenEndpoint: 'E',
                        UserInfoEndpoint: 'F',
                    },
                    Order: 1,
                    Type: 'authenticate-oidc',
                },
                {
                    Order: 2,
                    TargetGroupArn: { Ref: 'TargetGroup1E5480F51' },
                    Type: 'forward',
                },
            ],
        }));
        test.done();
    },
    'Add default Action and add Action with conditions'(test) {
        // GIVEN
        const listener = lb.addListener('Listener', { port: 80 });
        // WHEN
        listener.addAction('Action1', {
            action: elbv2.ListenerAction.forward([group1]),
        });
        listener.addAction('Action2', {
            hostHeader: 'example.com',
            priority: 10,
            action: elbv2.ListenerAction.forward([group2]),
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::ListenerRule', {
            Actions: [
                {
                    TargetGroupArn: { Ref: 'TargetGroup2D571E5D7' },
                    Type: 'forward',
                },
            ],
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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