"use strict";
const assert_1 = require("@aws-cdk/assert");
const ec2 = require("@aws-cdk/aws-ec2");
const cdk = require("@aws-cdk/core");
const elbv2 = require("../../lib");
const helpers_1 = require("../helpers");
module.exports = {
    'Empty target Group without type still requires a VPC'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'Stack');
        // WHEN
        new elbv2.ApplicationTargetGroup(stack, 'LB', {});
        // THEN
        test.throws(() => {
            app.synth();
        }, /'vpc' is required for a non-Lambda TargetGroup/);
        test.done();
    },
    'Can add self-registering target to imported TargetGroup'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'Stack');
        const vpc = new ec2.Vpc(stack, 'Vpc');
        // WHEN
        const tg = elbv2.ApplicationTargetGroup.fromTargetGroupAttributes(stack, 'TG', {
            targetGroupArn: 'arn',
        });
        tg.addTarget(new helpers_1.FakeSelfRegisteringTarget(stack, 'Target', vpc));
        // THEN
        test.done();
    },
    'Cannot add direct target to imported TargetGroup'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'Stack');
        const tg = elbv2.ApplicationTargetGroup.fromTargetGroupAttributes(stack, 'TG', {
            targetGroupArn: 'arn',
        });
        // WHEN
        test.throws(() => {
            tg.addTarget(new elbv2.InstanceTarget('i-1234'));
        }, /Cannot add a non-self registering target to an imported TargetGroup/);
        test.done();
    },
    'HealthCheck fields set if provided'(test) {
        // GIVEN
        const app = new cdk.App();
        const stack = new cdk.Stack(app, 'Stack');
        const vpc = new ec2.Vpc(stack, 'VPC', {});
        const alb = new elbv2.ApplicationLoadBalancer(stack, 'ALB', { vpc });
        const listener = new elbv2.ApplicationListener(stack, 'Listener', {
            port: 80,
            loadBalancer: alb,
            open: false,
        });
        // WHEN
        const ipTarget = new elbv2.IpTarget('10.10.12.12');
        listener.addTargets('TargetGroup', {
            targets: [ipTarget],
            port: 80,
            healthCheck: {
                enabled: true,
                healthyHttpCodes: '255',
                interval: cdk.Duration.seconds(255),
                timeout: cdk.Duration.seconds(192),
                healthyThresholdCount: 29,
                unhealthyThresholdCount: 27,
                path: '/arbitrary',
            },
        });
        // THEN
        assert_1.expect(stack).to(assert_1.haveResource('AWS::ElasticLoadBalancingV2::TargetGroup', {
            HealthCheckEnabled: true,
            HealthCheckIntervalSeconds: 255,
            HealthCheckPath: '/arbitrary',
            HealthCheckTimeoutSeconds: 192,
            HealthyThresholdCount: 29,
            Matcher: {
                HttpCode: '255',
            },
            Port: 80,
            UnhealthyThresholdCount: 27,
        }));
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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