from django.apps import apps

from mayan.apps.converter.settings import setting_image_generation_timeout

from .events import event_ocr_document_version_submitted
from .literals import TASK_DOCUMENT_VERSION_PAGE_OCR_TIMEOUT
from .tasks import task_document_version_ocr_process


def method_document_ocr_content(self):
    version_active = self.version_active

    if version_active:
        return version_active.ocr_content()
    else:
        return ()


def method_document_ocr_submit(self, user=None):
    version_active = self.version_active
    # Don't error out if document has no version.
    if version_active:
        version_active.submit_for_ocr(user=user)


def method_document_version_ocr_content(self):
    DocumentVersionPageOCRContent = apps.get_model(
        app_label='ocr', model_name='DocumentVersionPageOCRContent'
    )

    for page in self.pages.all():
        try:
            page_content = page.ocr_content.content
        except DocumentVersionPageOCRContent.DoesNotExist:
            """Not critical, just ignore and go to next page."""
        else:
            yield page_content


def method_document_version_ocr_submit(self, user=None):
    if user:
        user_id = user.pk
    else:
        user_id = None

    # Timeout calculation logic:
    # Total timeout for the document version task should be the
    # file page render timeout + the version page timeout + version page OCR
    # timeout x the total version pages. This decreases the probability of
    # the OCR task getting killed before the document file and version page
    # rendering finishes.

    event_ocr_document_version_submitted.commit(
        action_object=self.document, actor=user, target=self
    )

    task_document_version_ocr_process.apply_async(
        kwargs={
            'document_version_id': self.pk, 'user_id': user_id
        }, timeout=(
            TASK_DOCUMENT_VERSION_PAGE_OCR_TIMEOUT + setting_image_generation_timeout.value * 2
        ) * self.pages.count()
    )
