"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.UnauthenticatedAction = exports.ListenerAction = void 0;
const core_1 = require("@aws-cdk/core");
/**
 * What to do when a client makes a request to a listener
 *
 * Some actions can be combined with other ones (specifically,
 * you can perform authentication before serving the request).
 *
 * Multiple actions form a linked chain; the chain must always terminate in a
 * *(weighted)forward*, *fixedResponse* or *redirect* action.
 *
 * If an action supports chaining, the next action can be indicated
 * by passing it in the `next` property.
 *
 * (Called `ListenerAction` instead of the more strictly correct
 * `ListenerAction` because this is the class most users interact
 * with, and we want to make it not too visually overwhelming).
 */
class ListenerAction {
    /**
     * Create an instance of ListenerAction
     *
     * The default class should be good enough for most cases and
     * should be created by using one of the static factory functions,
     * but allow overriding to make sure we allow flexibility for the future.
     */
    constructor(actionJson, next) {
        this.actionJson = actionJson;
        this.next = next;
    }
    /**
     * Authenticate using an identity provider (IdP) that is compliant with OpenID Connect (OIDC)
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/listener-authenticate-users.html#oidc-requirements
     */
    static authenticateOidc(options) {
        var _a;
        return new ListenerAction({
            type: 'authenticate-oidc',
            authenticateOidcConfig: {
                authorizationEndpoint: options.authorizationEndpoint,
                clientId: options.clientId,
                clientSecret: options.clientSecret.toString(),
                issuer: options.issuer,
                tokenEndpoint: options.tokenEndpoint,
                userInfoEndpoint: options.userInfoEndpoint,
                authenticationRequestExtraParams: options.authenticationRequestExtraParams,
                onUnauthenticatedRequest: options.onUnauthenticatedRequest,
                scope: options.scope,
                sessionCookieName: options.sessionCookieName,
                sessionTimeout: (_a = options.sessionTimeout) === null || _a === void 0 ? void 0 : _a.toSeconds(),
            },
        }, options.next);
    }
    /**
     * Forward to one or more Target Groups
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#forward-actions
     */
    static forward(targetGroups, options = {}) {
        if (targetGroups.length === 0) {
            throw new Error('Need at least one targetGroup in a ListenerAction.forward()');
        }
        if (targetGroups.length === 1 && options.stickinessDuration === undefined) {
            // Render a "simple" action for backwards compatibility with old templates
            return new TargetGroupListenerAction(targetGroups, {
                type: 'forward',
                targetGroupArn: targetGroups[0].targetGroupArn,
            });
        }
        return new TargetGroupListenerAction(targetGroups, {
            type: 'forward',
            forwardConfig: {
                targetGroups: targetGroups.map(g => ({ targetGroupArn: g.targetGroupArn })),
                targetGroupStickinessConfig: options.stickinessDuration ? {
                    durationSeconds: options.stickinessDuration.toSeconds(),
                    enabled: true,
                } : undefined,
            },
        });
    }
    /**
     * Forward to one or more Target Groups which are weighted differently
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#forward-actions
     */
    static weightedForward(targetGroups, options = {}) {
        if (targetGroups.length === 0) {
            throw new Error('Need at least one targetGroup in a ListenerAction.weightedForward()');
        }
        return new TargetGroupListenerAction(targetGroups.map(g => g.targetGroup), {
            type: 'forward',
            forwardConfig: {
                targetGroups: targetGroups.map(g => ({ targetGroupArn: g.targetGroup.targetGroupArn, weight: g.weight })),
                targetGroupStickinessConfig: options.stickinessDuration ? {
                    durationSeconds: options.stickinessDuration.toSeconds(),
                    enabled: true,
                } : undefined,
            },
        });
    }
    /**
     * Return a fixed response
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#fixed-response-actions
     */
    static fixedResponse(statusCode, options = {}) {
        return new ListenerAction({
            type: 'fixed-response',
            fixedResponseConfig: {
                statusCode: core_1.Tokenization.stringifyNumber(statusCode),
                contentType: options.contentType,
                messageBody: options.messageBody,
            },
        });
    }
    /**
     * Redirect to a different URI
     *
     * A URI consists of the following components:
     * protocol://hostname:port/path?query. You must modify at least one of the
     * following components to avoid a redirect loop: protocol, hostname, port, or
     * path. Any components that you do not modify retain their original values.
     *
     * You can reuse URI components using the following reserved keywords:
     *
     * - `#{protocol}`
     * - `#{host}`
     * - `#{port}`
     * - `#{path}` (the leading "/" is removed)
     * - `#{query}`
     *
     * For example, you can change the path to "/new/#{path}", the hostname to
     * "example.#{host}", or the query to "#{query}&value=xyz".
     *
     * @see https://docs.aws.amazon.com/elasticloadbalancing/latest/application/load-balancer-listeners.html#redirect-actions
     */
    static redirect(options) {
        if ([options.host, options.path, options.port, options.protocol, options.query].findIndex(x => x !== undefined) === -1) {
            throw new Error('To prevent redirect loops, set at least one of \'protocol\', \'host\', \'port\', \'path\', or \'query\'.');
        }
        return new ListenerAction({
            type: 'redirect',
            redirectConfig: {
                statusCode: options.permanent ? 'HTTP_301' : 'HTTP_302',
                host: options.host,
                path: options.path,
                port: options.port,
                protocol: options.protocol,
                query: options.query,
            },
        });
    }
    /**
     * Render the actions in this chain
     */
    renderActions() {
        var _a, _b;
        return this.renumber([this.actionJson, ...(_b = (_a = this.next) === null || _a === void 0 ? void 0 : _a.renderActions()) !== null && _b !== void 0 ? _b : []]);
    }
    /**
     * Called when the action is being used in a listener
     */
    bind(scope, listener, associatingConstruct) {
        // Empty on purpose
        Array.isArray(scope);
        Array.isArray(listener);
        Array.isArray(associatingConstruct);
    }
    /**
     * Renumber the "order" fields in the actions array.
     *
     * We don't number for 0 or 1 elements, but otherwise number them 1...#actions
     * so ELB knows about the right order.
     *
     * Do this in `ListenerAction` instead of in `Listener` so that we give
     * users the opportunity to override by subclassing and overriding `renderActions`.
     */
    renumber(actions) {
        if (actions.length < 2) {
            return actions;
        }
        return actions.map((action, i) => ({ ...action, order: i + 1 }));
    }
}
exports.ListenerAction = ListenerAction;
/**
 * What to do with unauthenticated requests
 */
var UnauthenticatedAction;
(function (UnauthenticatedAction) {
    /**
     * Return an HTTP 401 Unauthorized error.
     */
    UnauthenticatedAction["DENY"] = "deny";
    /**
     * Allow the request to be forwarded to the target.
     */
    UnauthenticatedAction["ALLOW"] = "allow";
    /**
     * Redirect the request to the IdP authorization endpoint.
     */
    UnauthenticatedAction["AUTHENTICATE"] = "authenticate";
})(UnauthenticatedAction = exports.UnauthenticatedAction || (exports.UnauthenticatedAction = {}));
/**
 * Listener Action that calls "registerListener" on TargetGroups
 */
class TargetGroupListenerAction extends ListenerAction {
    constructor(targetGroups, actionJson) {
        super(actionJson);
        this.targetGroups = targetGroups;
    }
    bind(_scope, listener, associatingConstruct) {
        for (const tg of this.targetGroups) {
            tg.registerListener(listener, associatingConstruct);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiYXBwbGljYXRpb24tbGlzdGVuZXItYWN0aW9uLmpzIiwic291cmNlUm9vdCI6IiIsInNvdXJjZXMiOlsiYXBwbGljYXRpb24tbGlzdGVuZXItYWN0aW9uLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7OztBQUFBLHdDQUEyRjtBQU0zRjs7Ozs7Ozs7Ozs7Ozs7O0dBZUc7QUFDSCxNQUFhLGNBQWM7SUFtSXpCOzs7Ozs7T0FNRztJQUNILFlBQXVDLFVBQXNDLEVBQXFCLElBQXFCO1FBQWhGLGVBQVUsR0FBVixVQUFVLENBQTRCO1FBQXFCLFNBQUksR0FBSixJQUFJLENBQWlCO0lBQ3ZILENBQUM7SUExSUQ7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxnQkFBZ0IsQ0FBQyxPQUFnQzs7UUFDN0QsT0FBTyxJQUFJLGNBQWMsQ0FBQztZQUN4QixJQUFJLEVBQUUsbUJBQW1CO1lBQ3pCLHNCQUFzQixFQUFFO2dCQUN0QixxQkFBcUIsRUFBRSxPQUFPLENBQUMscUJBQXFCO2dCQUNwRCxRQUFRLEVBQUUsT0FBTyxDQUFDLFFBQVE7Z0JBQzFCLFlBQVksRUFBRSxPQUFPLENBQUMsWUFBWSxDQUFDLFFBQVEsRUFBRTtnQkFDN0MsTUFBTSxFQUFFLE9BQU8sQ0FBQyxNQUFNO2dCQUN0QixhQUFhLEVBQUUsT0FBTyxDQUFDLGFBQWE7Z0JBQ3BDLGdCQUFnQixFQUFFLE9BQU8sQ0FBQyxnQkFBZ0I7Z0JBQzFDLGdDQUFnQyxFQUFFLE9BQU8sQ0FBQyxnQ0FBZ0M7Z0JBQzFFLHdCQUF3QixFQUFFLE9BQU8sQ0FBQyx3QkFBd0I7Z0JBQzFELEtBQUssRUFBRSxPQUFPLENBQUMsS0FBSztnQkFDcEIsaUJBQWlCLEVBQUUsT0FBTyxDQUFDLGlCQUFpQjtnQkFDNUMsY0FBYyxRQUFFLE9BQU8sQ0FBQyxjQUFjLDBDQUFFLFNBQVMsRUFBRTthQUNwRDtTQUNGLEVBQUUsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ25CLENBQUM7SUFFRDs7OztPQUlHO0lBQ0ksTUFBTSxDQUFDLE9BQU8sQ0FBQyxZQUF1QyxFQUFFLFVBQTBCLEVBQUU7UUFDekYsSUFBSSxZQUFZLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUM3QixNQUFNLElBQUksS0FBSyxDQUFDLDZEQUE2RCxDQUFDLENBQUM7U0FDaEY7UUFDRCxJQUFJLFlBQVksQ0FBQyxNQUFNLEtBQUssQ0FBQyxJQUFJLE9BQU8sQ0FBQyxrQkFBa0IsS0FBSyxTQUFTLEVBQUU7WUFDekUsMEVBQTBFO1lBQzFFLE9BQU8sSUFBSSx5QkFBeUIsQ0FBQyxZQUFZLEVBQUU7Z0JBQ2pELElBQUksRUFBRSxTQUFTO2dCQUNmLGNBQWMsRUFBRSxZQUFZLENBQUMsQ0FBQyxDQUFDLENBQUMsY0FBYzthQUMvQyxDQUFDLENBQUM7U0FDSjtRQUVELE9BQU8sSUFBSSx5QkFBeUIsQ0FBQyxZQUFZLEVBQUU7WUFDakQsSUFBSSxFQUFFLFNBQVM7WUFDZixhQUFhLEVBQUU7Z0JBQ2IsWUFBWSxFQUFFLFlBQVksQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLEVBQUUsY0FBYyxFQUFFLENBQUMsQ0FBQyxjQUFjLEVBQUUsQ0FBQyxDQUFDO2dCQUMzRSwyQkFBMkIsRUFBRSxPQUFPLENBQUMsa0JBQWtCLENBQUMsQ0FBQyxDQUFDO29CQUN4RCxlQUFlLEVBQUUsT0FBTyxDQUFDLGtCQUFrQixDQUFDLFNBQVMsRUFBRTtvQkFDdkQsT0FBTyxFQUFFLElBQUk7aUJBQ2QsQ0FBQyxDQUFDLENBQUMsU0FBUzthQUNkO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQUVEOzs7O09BSUc7SUFDSSxNQUFNLENBQUMsZUFBZSxDQUFDLFlBQW1DLEVBQUUsVUFBMEIsRUFBRTtRQUM3RixJQUFJLFlBQVksQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQzdCLE1BQU0sSUFBSSxLQUFLLENBQUMscUVBQXFFLENBQUMsQ0FBQztTQUN4RjtRQUVELE9BQU8sSUFBSSx5QkFBeUIsQ0FBQyxZQUFZLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxFQUFFO1lBQ3pFLElBQUksRUFBRSxTQUFTO1lBQ2YsYUFBYSxFQUFFO2dCQUNiLFlBQVksRUFBRSxZQUFZLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxFQUFFLGNBQWMsRUFBRSxDQUFDLENBQUMsV0FBVyxDQUFDLGNBQWMsRUFBRSxNQUFNLEVBQUUsQ0FBQyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7Z0JBQ3pHLDJCQUEyQixFQUFFLE9BQU8sQ0FBQyxrQkFBa0IsQ0FBQyxDQUFDLENBQUM7b0JBQ3hELGVBQWUsRUFBRSxPQUFPLENBQUMsa0JBQWtCLENBQUMsU0FBUyxFQUFFO29CQUN2RCxPQUFPLEVBQUUsSUFBSTtpQkFDZCxDQUFDLENBQUMsQ0FBQyxTQUFTO2FBQ2Q7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7T0FJRztJQUNJLE1BQU0sQ0FBQyxhQUFhLENBQUMsVUFBa0IsRUFBRSxVQUFnQyxFQUFFO1FBQ2hGLE9BQU8sSUFBSSxjQUFjLENBQUM7WUFDeEIsSUFBSSxFQUFFLGdCQUFnQjtZQUN0QixtQkFBbUIsRUFBRTtnQkFDbkIsVUFBVSxFQUFFLG1CQUFZLENBQUMsZUFBZSxDQUFDLFVBQVUsQ0FBQztnQkFDcEQsV0FBVyxFQUFFLE9BQU8sQ0FBQyxXQUFXO2dCQUNoQyxXQUFXLEVBQUUsT0FBTyxDQUFDLFdBQVc7YUFDakM7U0FDRixDQUFDLENBQUM7SUFDTCxDQUFDO0lBRUQ7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7O09Bb0JHO0lBQ0ksTUFBTSxDQUFDLFFBQVEsQ0FBQyxPQUF3QjtRQUM3QyxJQUFJLENBQUMsT0FBTyxDQUFDLElBQUksRUFBRSxPQUFPLENBQUMsSUFBSSxFQUFFLE9BQU8sQ0FBQyxJQUFJLEVBQUUsT0FBTyxDQUFDLFFBQVEsRUFBRSxPQUFPLENBQUMsS0FBSyxDQUFDLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsQ0FBQyxLQUFLLFNBQVMsQ0FBQyxLQUFLLENBQUMsQ0FBQyxFQUFFO1lBQ3RILE1BQU0sSUFBSSxLQUFLLENBQUMsMEdBQTBHLENBQUMsQ0FBQztTQUM3SDtRQUVELE9BQU8sSUFBSSxjQUFjLENBQUM7WUFDeEIsSUFBSSxFQUFFLFVBQVU7WUFDaEIsY0FBYyxFQUFFO2dCQUNkLFVBQVUsRUFBRSxPQUFPLENBQUMsU0FBUyxDQUFDLENBQUMsQ0FBQyxVQUFVLENBQUMsQ0FBQyxDQUFDLFVBQVU7Z0JBQ3ZELElBQUksRUFBRSxPQUFPLENBQUMsSUFBSTtnQkFDbEIsSUFBSSxFQUFFLE9BQU8sQ0FBQyxJQUFJO2dCQUNsQixJQUFJLEVBQUUsT0FBTyxDQUFDLElBQUk7Z0JBQ2xCLFFBQVEsRUFBRSxPQUFPLENBQUMsUUFBUTtnQkFDMUIsS0FBSyxFQUFFLE9BQU8sQ0FBQyxLQUFLO2FBQ3JCO1NBQ0YsQ0FBQyxDQUFDO0lBQ0wsQ0FBQztJQVlEOztPQUVHO0lBQ0ksYUFBYTs7UUFDbEIsT0FBTyxJQUFJLENBQUMsUUFBUSxDQUFDLENBQUMsSUFBSSxDQUFDLFVBQVUsRUFBRSxlQUFHLElBQUksQ0FBQyxJQUFJLDBDQUFFLGFBQWEscUNBQU0sRUFBRSxDQUFDLENBQUMsQ0FBQztJQUMvRSxDQUFDO0lBRUQ7O09BRUc7SUFDSSxJQUFJLENBQUMsS0FBZ0IsRUFBRSxRQUE4QixFQUFFLG9CQUFpQztRQUM3RixtQkFBbUI7UUFDbkIsS0FBSyxDQUFDLE9BQU8sQ0FBQyxLQUFLLENBQUMsQ0FBQztRQUNyQixLQUFLLENBQUMsT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDO1FBQ3hCLEtBQUssQ0FBQyxPQUFPLENBQUMsb0JBQW9CLENBQUMsQ0FBQztJQUN0QyxDQUFDO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDTyxRQUFRLENBQUMsT0FBcUM7UUFDdEQsSUFBSSxPQUFPLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUFFLE9BQU8sT0FBTyxDQUFDO1NBQUU7UUFFM0MsT0FBTyxPQUFPLENBQUMsR0FBRyxDQUFDLENBQUMsTUFBTSxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsQ0FBQyxFQUFFLEdBQUcsTUFBTSxFQUFFLEtBQUssRUFBRSxDQUFDLEdBQUcsQ0FBQyxFQUFFLENBQUMsQ0FBQyxDQUFDO0lBQ25FLENBQUM7Q0FDRjtBQTVLRCx3Q0E0S0M7QUF3TkQ7O0dBRUc7QUFDSCxJQUFZLHFCQWVYO0FBZkQsV0FBWSxxQkFBcUI7SUFDL0I7O09BRUc7SUFDSCxzQ0FBYSxDQUFBO0lBRWI7O09BRUc7SUFDSCx3Q0FBZSxDQUFBO0lBRWY7O09BRUc7SUFDSCxzREFBNkIsQ0FBQTtBQUMvQixDQUFDLEVBZlcscUJBQXFCLEdBQXJCLDZCQUFxQixLQUFyQiw2QkFBcUIsUUFlaEM7QUFFRDs7R0FFRztBQUNILE1BQU0seUJBQTBCLFNBQVEsY0FBYztJQUNwRCxZQUE2QixZQUF1QyxFQUFFLFVBQXNDO1FBQzFHLEtBQUssQ0FBQyxVQUFVLENBQUMsQ0FBQztRQURTLGlCQUFZLEdBQVosWUFBWSxDQUEyQjtJQUVwRSxDQUFDO0lBRU0sSUFBSSxDQUFDLE1BQWlCLEVBQUUsUUFBOEIsRUFBRSxvQkFBaUM7UUFDOUYsS0FBSyxNQUFNLEVBQUUsSUFBSSxJQUFJLENBQUMsWUFBWSxFQUFFO1lBQ2xDLEVBQUUsQ0FBQyxnQkFBZ0IsQ0FBQyxRQUFRLEVBQUUsb0JBQW9CLENBQUMsQ0FBQztTQUNyRDtJQUNILENBQUM7Q0FDRiIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCB7IENvbnN0cnVjdCwgRHVyYXRpb24sIElDb25zdHJ1Y3QsIFNlY3JldFZhbHVlLCBUb2tlbml6YXRpb24gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IENmbkxpc3RlbmVyIH0gZnJvbSAnLi4vZWxhc3RpY2xvYWRiYWxhbmNpbmd2Mi5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgSUxpc3RlbmVyQWN0aW9uIH0gZnJvbSAnLi4vc2hhcmVkL2xpc3RlbmVyLWFjdGlvbic7XG5pbXBvcnQgeyBJQXBwbGljYXRpb25MaXN0ZW5lciB9IGZyb20gJy4vYXBwbGljYXRpb24tbGlzdGVuZXInO1xuaW1wb3J0IHsgSUFwcGxpY2F0aW9uVGFyZ2V0R3JvdXAgfSBmcm9tICcuL2FwcGxpY2F0aW9uLXRhcmdldC1ncm91cCc7XG5cbi8qKlxuICogV2hhdCB0byBkbyB3aGVuIGEgY2xpZW50IG1ha2VzIGEgcmVxdWVzdCB0byBhIGxpc3RlbmVyXG4gKlxuICogU29tZSBhY3Rpb25zIGNhbiBiZSBjb21iaW5lZCB3aXRoIG90aGVyIG9uZXMgKHNwZWNpZmljYWxseSxcbiAqIHlvdSBjYW4gcGVyZm9ybSBhdXRoZW50aWNhdGlvbiBiZWZvcmUgc2VydmluZyB0aGUgcmVxdWVzdCkuXG4gKlxuICogTXVsdGlwbGUgYWN0aW9ucyBmb3JtIGEgbGlua2VkIGNoYWluOyB0aGUgY2hhaW4gbXVzdCBhbHdheXMgdGVybWluYXRlIGluIGFcbiAqICood2VpZ2h0ZWQpZm9yd2FyZCosICpmaXhlZFJlc3BvbnNlKiBvciAqcmVkaXJlY3QqIGFjdGlvbi5cbiAqXG4gKiBJZiBhbiBhY3Rpb24gc3VwcG9ydHMgY2hhaW5pbmcsIHRoZSBuZXh0IGFjdGlvbiBjYW4gYmUgaW5kaWNhdGVkXG4gKiBieSBwYXNzaW5nIGl0IGluIHRoZSBgbmV4dGAgcHJvcGVydHkuXG4gKlxuICogKENhbGxlZCBgTGlzdGVuZXJBY3Rpb25gIGluc3RlYWQgb2YgdGhlIG1vcmUgc3RyaWN0bHkgY29ycmVjdFxuICogYExpc3RlbmVyQWN0aW9uYCBiZWNhdXNlIHRoaXMgaXMgdGhlIGNsYXNzIG1vc3QgdXNlcnMgaW50ZXJhY3RcbiAqIHdpdGgsIGFuZCB3ZSB3YW50IHRvIG1ha2UgaXQgbm90IHRvbyB2aXN1YWxseSBvdmVyd2hlbG1pbmcpLlxuICovXG5leHBvcnQgY2xhc3MgTGlzdGVuZXJBY3Rpb24gaW1wbGVtZW50cyBJTGlzdGVuZXJBY3Rpb24ge1xuICAvKipcbiAgICogQXV0aGVudGljYXRlIHVzaW5nIGFuIGlkZW50aXR5IHByb3ZpZGVyIChJZFApIHRoYXQgaXMgY29tcGxpYW50IHdpdGggT3BlbklEIENvbm5lY3QgKE9JREMpXG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNsb2FkYmFsYW5jaW5nL2xhdGVzdC9hcHBsaWNhdGlvbi9saXN0ZW5lci1hdXRoZW50aWNhdGUtdXNlcnMuaHRtbCNvaWRjLXJlcXVpcmVtZW50c1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBhdXRoZW50aWNhdGVPaWRjKG9wdGlvbnM6IEF1dGhlbnRpY2F0ZU9pZGNPcHRpb25zKTogTGlzdGVuZXJBY3Rpb24ge1xuICAgIHJldHVybiBuZXcgTGlzdGVuZXJBY3Rpb24oe1xuICAgICAgdHlwZTogJ2F1dGhlbnRpY2F0ZS1vaWRjJyxcbiAgICAgIGF1dGhlbnRpY2F0ZU9pZGNDb25maWc6IHtcbiAgICAgICAgYXV0aG9yaXphdGlvbkVuZHBvaW50OiBvcHRpb25zLmF1dGhvcml6YXRpb25FbmRwb2ludCxcbiAgICAgICAgY2xpZW50SWQ6IG9wdGlvbnMuY2xpZW50SWQsXG4gICAgICAgIGNsaWVudFNlY3JldDogb3B0aW9ucy5jbGllbnRTZWNyZXQudG9TdHJpbmcoKSxcbiAgICAgICAgaXNzdWVyOiBvcHRpb25zLmlzc3VlcixcbiAgICAgICAgdG9rZW5FbmRwb2ludDogb3B0aW9ucy50b2tlbkVuZHBvaW50LFxuICAgICAgICB1c2VySW5mb0VuZHBvaW50OiBvcHRpb25zLnVzZXJJbmZvRW5kcG9pbnQsXG4gICAgICAgIGF1dGhlbnRpY2F0aW9uUmVxdWVzdEV4dHJhUGFyYW1zOiBvcHRpb25zLmF1dGhlbnRpY2F0aW9uUmVxdWVzdEV4dHJhUGFyYW1zLFxuICAgICAgICBvblVuYXV0aGVudGljYXRlZFJlcXVlc3Q6IG9wdGlvbnMub25VbmF1dGhlbnRpY2F0ZWRSZXF1ZXN0LFxuICAgICAgICBzY29wZTogb3B0aW9ucy5zY29wZSxcbiAgICAgICAgc2Vzc2lvbkNvb2tpZU5hbWU6IG9wdGlvbnMuc2Vzc2lvbkNvb2tpZU5hbWUsXG4gICAgICAgIHNlc3Npb25UaW1lb3V0OiBvcHRpb25zLnNlc3Npb25UaW1lb3V0Py50b1NlY29uZHMoKSxcbiAgICAgIH0sXG4gICAgfSwgb3B0aW9ucy5uZXh0KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGb3J3YXJkIHRvIG9uZSBvciBtb3JlIFRhcmdldCBHcm91cHNcbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2xvYWRiYWxhbmNpbmcvbGF0ZXN0L2FwcGxpY2F0aW9uL2xvYWQtYmFsYW5jZXItbGlzdGVuZXJzLmh0bWwjZm9yd2FyZC1hY3Rpb25zXG4gICAqL1xuICBwdWJsaWMgc3RhdGljIGZvcndhcmQodGFyZ2V0R3JvdXBzOiBJQXBwbGljYXRpb25UYXJnZXRHcm91cFtdLCBvcHRpb25zOiBGb3J3YXJkT3B0aW9ucyA9IHt9KTogTGlzdGVuZXJBY3Rpb24ge1xuICAgIGlmICh0YXJnZXRHcm91cHMubGVuZ3RoID09PSAwKSB7XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoJ05lZWQgYXQgbGVhc3Qgb25lIHRhcmdldEdyb3VwIGluIGEgTGlzdGVuZXJBY3Rpb24uZm9yd2FyZCgpJyk7XG4gICAgfVxuICAgIGlmICh0YXJnZXRHcm91cHMubGVuZ3RoID09PSAxICYmIG9wdGlvbnMuc3RpY2tpbmVzc0R1cmF0aW9uID09PSB1bmRlZmluZWQpIHtcbiAgICAgIC8vIFJlbmRlciBhIFwic2ltcGxlXCIgYWN0aW9uIGZvciBiYWNrd2FyZHMgY29tcGF0aWJpbGl0eSB3aXRoIG9sZCB0ZW1wbGF0ZXNcbiAgICAgIHJldHVybiBuZXcgVGFyZ2V0R3JvdXBMaXN0ZW5lckFjdGlvbih0YXJnZXRHcm91cHMsIHtcbiAgICAgICAgdHlwZTogJ2ZvcndhcmQnLFxuICAgICAgICB0YXJnZXRHcm91cEFybjogdGFyZ2V0R3JvdXBzWzBdLnRhcmdldEdyb3VwQXJuLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBUYXJnZXRHcm91cExpc3RlbmVyQWN0aW9uKHRhcmdldEdyb3Vwcywge1xuICAgICAgdHlwZTogJ2ZvcndhcmQnLFxuICAgICAgZm9yd2FyZENvbmZpZzoge1xuICAgICAgICB0YXJnZXRHcm91cHM6IHRhcmdldEdyb3Vwcy5tYXAoZyA9PiAoeyB0YXJnZXRHcm91cEFybjogZy50YXJnZXRHcm91cEFybiB9KSksXG4gICAgICAgIHRhcmdldEdyb3VwU3RpY2tpbmVzc0NvbmZpZzogb3B0aW9ucy5zdGlja2luZXNzRHVyYXRpb24gPyB7XG4gICAgICAgICAgZHVyYXRpb25TZWNvbmRzOiBvcHRpb25zLnN0aWNraW5lc3NEdXJhdGlvbi50b1NlY29uZHMoKSxcbiAgICAgICAgICBlbmFibGVkOiB0cnVlLFxuICAgICAgICB9IDogdW5kZWZpbmVkLFxuICAgICAgfSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBGb3J3YXJkIHRvIG9uZSBvciBtb3JlIFRhcmdldCBHcm91cHMgd2hpY2ggYXJlIHdlaWdodGVkIGRpZmZlcmVudGx5XG4gICAqXG4gICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2VsYXN0aWNsb2FkYmFsYW5jaW5nL2xhdGVzdC9hcHBsaWNhdGlvbi9sb2FkLWJhbGFuY2VyLWxpc3RlbmVycy5odG1sI2ZvcndhcmQtYWN0aW9uc1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyB3ZWlnaHRlZEZvcndhcmQodGFyZ2V0R3JvdXBzOiBXZWlnaHRlZFRhcmdldEdyb3VwW10sIG9wdGlvbnM6IEZvcndhcmRPcHRpb25zID0ge30pOiBMaXN0ZW5lckFjdGlvbiB7XG4gICAgaWYgKHRhcmdldEdyb3Vwcy5sZW5ndGggPT09IDApIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignTmVlZCBhdCBsZWFzdCBvbmUgdGFyZ2V0R3JvdXAgaW4gYSBMaXN0ZW5lckFjdGlvbi53ZWlnaHRlZEZvcndhcmQoKScpO1xuICAgIH1cblxuICAgIHJldHVybiBuZXcgVGFyZ2V0R3JvdXBMaXN0ZW5lckFjdGlvbih0YXJnZXRHcm91cHMubWFwKGcgPT4gZy50YXJnZXRHcm91cCksIHtcbiAgICAgIHR5cGU6ICdmb3J3YXJkJyxcbiAgICAgIGZvcndhcmRDb25maWc6IHtcbiAgICAgICAgdGFyZ2V0R3JvdXBzOiB0YXJnZXRHcm91cHMubWFwKGcgPT4gKHsgdGFyZ2V0R3JvdXBBcm46IGcudGFyZ2V0R3JvdXAudGFyZ2V0R3JvdXBBcm4sIHdlaWdodDogZy53ZWlnaHQgfSkpLFxuICAgICAgICB0YXJnZXRHcm91cFN0aWNraW5lc3NDb25maWc6IG9wdGlvbnMuc3RpY2tpbmVzc0R1cmF0aW9uID8ge1xuICAgICAgICAgIGR1cmF0aW9uU2Vjb25kczogb3B0aW9ucy5zdGlja2luZXNzRHVyYXRpb24udG9TZWNvbmRzKCksXG4gICAgICAgICAgZW5hYmxlZDogdHJ1ZSxcbiAgICAgICAgfSA6IHVuZGVmaW5lZCxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogUmV0dXJuIGEgZml4ZWQgcmVzcG9uc2VcbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2xvYWRiYWxhbmNpbmcvbGF0ZXN0L2FwcGxpY2F0aW9uL2xvYWQtYmFsYW5jZXItbGlzdGVuZXJzLmh0bWwjZml4ZWQtcmVzcG9uc2UtYWN0aW9uc1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyBmaXhlZFJlc3BvbnNlKHN0YXR1c0NvZGU6IG51bWJlciwgb3B0aW9uczogRml4ZWRSZXNwb25zZU9wdGlvbnMgPSB7fSk6IExpc3RlbmVyQWN0aW9uIHtcbiAgICByZXR1cm4gbmV3IExpc3RlbmVyQWN0aW9uKHtcbiAgICAgIHR5cGU6ICdmaXhlZC1yZXNwb25zZScsXG4gICAgICBmaXhlZFJlc3BvbnNlQ29uZmlnOiB7XG4gICAgICAgIHN0YXR1c0NvZGU6IFRva2VuaXphdGlvbi5zdHJpbmdpZnlOdW1iZXIoc3RhdHVzQ29kZSksXG4gICAgICAgIGNvbnRlbnRUeXBlOiBvcHRpb25zLmNvbnRlbnRUeXBlLFxuICAgICAgICBtZXNzYWdlQm9keTogb3B0aW9ucy5tZXNzYWdlQm9keSxcbiAgICAgIH0sXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogUmVkaXJlY3QgdG8gYSBkaWZmZXJlbnQgVVJJXG4gICAqXG4gICAqIEEgVVJJIGNvbnNpc3RzIG9mIHRoZSBmb2xsb3dpbmcgY29tcG9uZW50czpcbiAgICogcHJvdG9jb2w6Ly9ob3N0bmFtZTpwb3J0L3BhdGg/cXVlcnkuIFlvdSBtdXN0IG1vZGlmeSBhdCBsZWFzdCBvbmUgb2YgdGhlXG4gICAqIGZvbGxvd2luZyBjb21wb25lbnRzIHRvIGF2b2lkIGEgcmVkaXJlY3QgbG9vcDogcHJvdG9jb2wsIGhvc3RuYW1lLCBwb3J0LCBvclxuICAgKiBwYXRoLiBBbnkgY29tcG9uZW50cyB0aGF0IHlvdSBkbyBub3QgbW9kaWZ5IHJldGFpbiB0aGVpciBvcmlnaW5hbCB2YWx1ZXMuXG4gICAqXG4gICAqIFlvdSBjYW4gcmV1c2UgVVJJIGNvbXBvbmVudHMgdXNpbmcgdGhlIGZvbGxvd2luZyByZXNlcnZlZCBrZXl3b3JkczpcbiAgICpcbiAgICogLSBgI3twcm90b2NvbH1gXG4gICAqIC0gYCN7aG9zdH1gXG4gICAqIC0gYCN7cG9ydH1gXG4gICAqIC0gYCN7cGF0aH1gICh0aGUgbGVhZGluZyBcIi9cIiBpcyByZW1vdmVkKVxuICAgKiAtIGAje3F1ZXJ5fWBcbiAgICpcbiAgICogRm9yIGV4YW1wbGUsIHlvdSBjYW4gY2hhbmdlIHRoZSBwYXRoIHRvIFwiL25ldy8je3BhdGh9XCIsIHRoZSBob3N0bmFtZSB0b1xuICAgKiBcImV4YW1wbGUuI3tob3N0fVwiLCBvciB0aGUgcXVlcnkgdG8gXCIje3F1ZXJ5fSZ2YWx1ZT14eXpcIi5cbiAgICpcbiAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vZWxhc3RpY2xvYWRiYWxhbmNpbmcvbGF0ZXN0L2FwcGxpY2F0aW9uL2xvYWQtYmFsYW5jZXItbGlzdGVuZXJzLmh0bWwjcmVkaXJlY3QtYWN0aW9uc1xuICAgKi9cbiAgcHVibGljIHN0YXRpYyByZWRpcmVjdChvcHRpb25zOiBSZWRpcmVjdE9wdGlvbnMpOiBMaXN0ZW5lckFjdGlvbiB7XG4gICAgaWYgKFtvcHRpb25zLmhvc3QsIG9wdGlvbnMucGF0aCwgb3B0aW9ucy5wb3J0LCBvcHRpb25zLnByb3RvY29sLCBvcHRpb25zLnF1ZXJ5XS5maW5kSW5kZXgoeCA9PiB4ICE9PSB1bmRlZmluZWQpID09PSAtMSkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdUbyBwcmV2ZW50IHJlZGlyZWN0IGxvb3BzLCBzZXQgYXQgbGVhc3Qgb25lIG9mIFxcJ3Byb3RvY29sXFwnLCBcXCdob3N0XFwnLCBcXCdwb3J0XFwnLCBcXCdwYXRoXFwnLCBvciBcXCdxdWVyeVxcJy4nKTtcbiAgICB9XG5cbiAgICByZXR1cm4gbmV3IExpc3RlbmVyQWN0aW9uKHtcbiAgICAgIHR5cGU6ICdyZWRpcmVjdCcsXG4gICAgICByZWRpcmVjdENvbmZpZzoge1xuICAgICAgICBzdGF0dXNDb2RlOiBvcHRpb25zLnBlcm1hbmVudCA/ICdIVFRQXzMwMScgOiAnSFRUUF8zMDInLFxuICAgICAgICBob3N0OiBvcHRpb25zLmhvc3QsXG4gICAgICAgIHBhdGg6IG9wdGlvbnMucGF0aCxcbiAgICAgICAgcG9ydDogb3B0aW9ucy5wb3J0LFxuICAgICAgICBwcm90b2NvbDogb3B0aW9ucy5wcm90b2NvbCxcbiAgICAgICAgcXVlcnk6IG9wdGlvbnMucXVlcnksXG4gICAgICB9LFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENyZWF0ZSBhbiBpbnN0YW5jZSBvZiBMaXN0ZW5lckFjdGlvblxuICAgKlxuICAgKiBUaGUgZGVmYXVsdCBjbGFzcyBzaG91bGQgYmUgZ29vZCBlbm91Z2ggZm9yIG1vc3QgY2FzZXMgYW5kXG4gICAqIHNob3VsZCBiZSBjcmVhdGVkIGJ5IHVzaW5nIG9uZSBvZiB0aGUgc3RhdGljIGZhY3RvcnkgZnVuY3Rpb25zLFxuICAgKiBidXQgYWxsb3cgb3ZlcnJpZGluZyB0byBtYWtlIHN1cmUgd2UgYWxsb3cgZmxleGliaWxpdHkgZm9yIHRoZSBmdXR1cmUuXG4gICAqL1xuICBwcm90ZWN0ZWQgY29uc3RydWN0b3IocHJpdmF0ZSByZWFkb25seSBhY3Rpb25Kc29uOiBDZm5MaXN0ZW5lci5BY3Rpb25Qcm9wZXJ0eSwgcHJvdGVjdGVkIHJlYWRvbmx5IG5leHQ/OiBMaXN0ZW5lckFjdGlvbikge1xuICB9XG5cbiAgLyoqXG4gICAqIFJlbmRlciB0aGUgYWN0aW9ucyBpbiB0aGlzIGNoYWluXG4gICAqL1xuICBwdWJsaWMgcmVuZGVyQWN0aW9ucygpOiBDZm5MaXN0ZW5lci5BY3Rpb25Qcm9wZXJ0eVtdIHtcbiAgICByZXR1cm4gdGhpcy5yZW51bWJlcihbdGhpcy5hY3Rpb25Kc29uLCAuLi50aGlzLm5leHQ/LnJlbmRlckFjdGlvbnMoKSA/PyBbXV0pO1xuICB9XG5cbiAgLyoqXG4gICAqIENhbGxlZCB3aGVuIHRoZSBhY3Rpb24gaXMgYmVpbmcgdXNlZCBpbiBhIGxpc3RlbmVyXG4gICAqL1xuICBwdWJsaWMgYmluZChzY29wZTogQ29uc3RydWN0LCBsaXN0ZW5lcjogSUFwcGxpY2F0aW9uTGlzdGVuZXIsIGFzc29jaWF0aW5nQ29uc3RydWN0PzogSUNvbnN0cnVjdCkge1xuICAgIC8vIEVtcHR5IG9uIHB1cnBvc2VcbiAgICBBcnJheS5pc0FycmF5KHNjb3BlKTtcbiAgICBBcnJheS5pc0FycmF5KGxpc3RlbmVyKTtcbiAgICBBcnJheS5pc0FycmF5KGFzc29jaWF0aW5nQ29uc3RydWN0KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBSZW51bWJlciB0aGUgXCJvcmRlclwiIGZpZWxkcyBpbiB0aGUgYWN0aW9ucyBhcnJheS5cbiAgICpcbiAgICogV2UgZG9uJ3QgbnVtYmVyIGZvciAwIG9yIDEgZWxlbWVudHMsIGJ1dCBvdGhlcndpc2UgbnVtYmVyIHRoZW0gMS4uLiNhY3Rpb25zXG4gICAqIHNvIEVMQiBrbm93cyBhYm91dCB0aGUgcmlnaHQgb3JkZXIuXG4gICAqXG4gICAqIERvIHRoaXMgaW4gYExpc3RlbmVyQWN0aW9uYCBpbnN0ZWFkIG9mIGluIGBMaXN0ZW5lcmAgc28gdGhhdCB3ZSBnaXZlXG4gICAqIHVzZXJzIHRoZSBvcHBvcnR1bml0eSB0byBvdmVycmlkZSBieSBzdWJjbGFzc2luZyBhbmQgb3ZlcnJpZGluZyBgcmVuZGVyQWN0aW9uc2AuXG4gICAqL1xuICBwcm90ZWN0ZWQgcmVudW1iZXIoYWN0aW9uczogQ2ZuTGlzdGVuZXIuQWN0aW9uUHJvcGVydHlbXSk6IENmbkxpc3RlbmVyLkFjdGlvblByb3BlcnR5W10ge1xuICAgIGlmIChhY3Rpb25zLmxlbmd0aCA8IDIpIHsgcmV0dXJuIGFjdGlvbnM7IH1cblxuICAgIHJldHVybiBhY3Rpb25zLm1hcCgoYWN0aW9uLCBpKSA9PiAoeyAuLi5hY3Rpb24sIG9yZGVyOiBpICsgMSB9KSk7XG4gIH1cbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBgTGlzdGVuZXJBY3Rpb24uZm9yd2FyZCgpYFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEZvcndhcmRPcHRpb25zIHtcbiAgLyoqXG4gICAqIEZvciBob3cgbG9uZyBjbGllbnRzIHNob3VsZCBiZSBkaXJlY3RlZCB0byB0aGUgc2FtZSB0YXJnZXQgZ3JvdXBcbiAgICpcbiAgICogUmFuZ2UgYmV0d2VlbiAxIHNlY29uZCBhbmQgNyBkYXlzLlxuICAgKlxuICAgKiBAZGVmYXVsdCAtIE5vIHN0aWNraW5lc3NcbiAgICovXG4gIHJlYWRvbmx5IHN0aWNraW5lc3NEdXJhdGlvbj86IER1cmF0aW9uO1xufVxuXG4vKipcbiAqIEEgVGFyZ2V0IEdyb3VwIGFuZCB3ZWlnaHQgY29tYmluYXRpb25cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBXZWlnaHRlZFRhcmdldEdyb3VwIHtcbiAgLyoqXG4gICAqIFRoZSB0YXJnZXQgZ3JvdXBcbiAgICovXG4gIHJlYWRvbmx5IHRhcmdldEdyb3VwOiBJQXBwbGljYXRpb25UYXJnZXRHcm91cDtcblxuICAvKipcbiAgICogVGhlIHRhcmdldCBncm91cCdzIHdlaWdodFxuICAgKlxuICAgKiBSYW5nZSBpcyBbMC4uMTAwMCkuXG4gICAqXG4gICAqIEBkZWZhdWx0IDFcbiAgICovXG4gIHJlYWRvbmx5IHdlaWdodD86IG51bWJlcjtcbn1cblxuLyoqXG4gKiBPcHRpb25zIGZvciBgTGlzdGVuZXJBY3Rpb24uZml4ZWRSZXNwb25zZSgpYFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEZpeGVkUmVzcG9uc2VPcHRpb25zIHtcbiAgLyoqXG4gICAqIENvbnRlbnQgVHlwZSBvZiB0aGUgcmVzcG9uc2VcbiAgICpcbiAgICogVmFsaWQgVmFsdWVzOiB0ZXh0L3BsYWluIHwgdGV4dC9jc3MgfCB0ZXh0L2h0bWwgfCBhcHBsaWNhdGlvbi9qYXZhc2NyaXB0IHwgYXBwbGljYXRpb24vanNvblxuICAgKlxuICAgKiBAZGVmYXVsdCAtIEF1dG9tYXRpY2FsbHkgZGV0ZXJtaW5lZFxuICAgKi9cbiAgcmVhZG9ubHkgY29udGVudFR5cGU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSByZXNwb25zZSBib2R5XG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gYm9keVxuICAgKi9cbiAgcmVhZG9ubHkgbWVzc2FnZUJvZHk/OiBzdHJpbmc7XG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgYExpc3RlbmVyQWN0aW9uLnJlZGlyZWN0KClgXG4gKlxuICogQSBVUkkgY29uc2lzdHMgb2YgdGhlIGZvbGxvd2luZyBjb21wb25lbnRzOlxuICogcHJvdG9jb2w6Ly9ob3N0bmFtZTpwb3J0L3BhdGg/cXVlcnkuIFlvdSBtdXN0IG1vZGlmeSBhdCBsZWFzdCBvbmUgb2YgdGhlXG4gKiBmb2xsb3dpbmcgY29tcG9uZW50cyB0byBhdm9pZCBhIHJlZGlyZWN0IGxvb3A6IHByb3RvY29sLCBob3N0bmFtZSwgcG9ydCwgb3JcbiAqIHBhdGguIEFueSBjb21wb25lbnRzIHRoYXQgeW91IGRvIG5vdCBtb2RpZnkgcmV0YWluIHRoZWlyIG9yaWdpbmFsIHZhbHVlcy5cbiAqXG4gKiBZb3UgY2FuIHJldXNlIFVSSSBjb21wb25lbnRzIHVzaW5nIHRoZSBmb2xsb3dpbmcgcmVzZXJ2ZWQga2V5d29yZHM6XG4gKlxuICogLSBgI3twcm90b2NvbH1gXG4gKiAtIGAje2hvc3R9YFxuICogLSBgI3twb3J0fWBcbiAqIC0gYCN7cGF0aH1gICh0aGUgbGVhZGluZyBcIi9cIiBpcyByZW1vdmVkKVxuICogLSBgI3txdWVyeX1gXG4gKlxuICogRm9yIGV4YW1wbGUsIHlvdSBjYW4gY2hhbmdlIHRoZSBwYXRoIHRvIFwiL25ldy8je3BhdGh9XCIsIHRoZSBob3N0bmFtZSB0b1xuICogXCJleGFtcGxlLiN7aG9zdH1cIiwgb3IgdGhlIHF1ZXJ5IHRvIFwiI3txdWVyeX0mdmFsdWU9eHl6XCIuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgUmVkaXJlY3RPcHRpb25zIHtcbiAgLyoqXG4gICAqIFRoZSBob3N0bmFtZS5cbiAgICpcbiAgICogVGhpcyBjb21wb25lbnQgaXMgbm90IHBlcmNlbnQtZW5jb2RlZC4gVGhlIGhvc3RuYW1lIGNhbiBjb250YWluICN7aG9zdH0uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gY2hhbmdlXG4gICAqL1xuICByZWFkb25seSBob3N0Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgYWJzb2x1dGUgcGF0aCwgc3RhcnRpbmcgd2l0aCB0aGUgbGVhZGluZyBcIi9cIi5cbiAgICpcbiAgICogVGhpcyBjb21wb25lbnQgaXMgbm90IHBlcmNlbnQtZW5jb2RlZC4gVGhlIHBhdGggY2FuIGNvbnRhaW4gI3tob3N0fSwgI3twYXRofSwgYW5kICN7cG9ydH0uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gY2hhbmdlXG4gICAqL1xuICByZWFkb25seSBwYXRoPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcG9ydC5cbiAgICpcbiAgICogWW91IGNhbiBzcGVjaWZ5IGEgdmFsdWUgZnJvbSAxIHRvIDY1NTM1IG9yICN7cG9ydH0uXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gY2hhbmdlXG4gICAqL1xuICByZWFkb25seSBwb3J0Pzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcHJvdG9jb2wuXG4gICAqXG4gICAqIFlvdSBjYW4gc3BlY2lmeSBIVFRQLCBIVFRQUywgb3IgI3twcm90b2NvbH0uIFlvdSBjYW4gcmVkaXJlY3QgSFRUUCB0byBIVFRQLCBIVFRQIHRvIEhUVFBTLCBhbmQgSFRUUFMgdG8gSFRUUFMuIFlvdSBjYW5ub3QgcmVkaXJlY3QgSFRUUFMgdG8gSFRUUC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBjaGFuZ2VcbiAgICovXG4gIHJlYWRvbmx5IHByb3RvY29sPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgcXVlcnkgcGFyYW1ldGVycywgVVJMLWVuY29kZWQgd2hlbiBuZWNlc3NhcnksIGJ1dCBub3QgcGVyY2VudC1lbmNvZGVkLlxuICAgKlxuICAgKiBEbyBub3QgaW5jbHVkZSB0aGUgbGVhZGluZyBcIj9cIiwgYXMgaXQgaXMgYXV0b21hdGljYWxseSBhZGRlZC4gWW91IGNhbiBzcGVjaWZ5IGFueSBvZiB0aGUgcmVzZXJ2ZWQga2V5d29yZHMuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gTm8gY2hhbmdlXG4gICAqL1xuICByZWFkb25seSBxdWVyeT86IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIEhUVFAgcmVkaXJlY3QgY29kZS5cbiAgICpcbiAgICogVGhlIHJlZGlyZWN0IGlzIGVpdGhlciBwZXJtYW5lbnQgKEhUVFAgMzAxKSBvciB0ZW1wb3JhcnkgKEhUVFAgMzAyKS5cbiAgICpcbiAgICogQGRlZmF1bHQgZmFsc2VcbiAgICovXG4gIHJlYWRvbmx5IHBlcm1hbmVudD86IGJvb2xlYW47XG59XG5cbi8qKlxuICogT3B0aW9ucyBmb3IgYExpc3RlbmVyQWN0aW9uLmF1dGhlbmNpYXRlT2lkYygpYFxuICovXG5leHBvcnQgaW50ZXJmYWNlIEF1dGhlbnRpY2F0ZU9pZGNPcHRpb25zIHtcbiAgLyoqXG4gICAqIFdoYXQgYWN0aW9uIHRvIGV4ZWN1dGUgbmV4dFxuICAgKi9cbiAgcmVhZG9ubHkgbmV4dDogTGlzdGVuZXJBY3Rpb247XG5cbiAgLyoqXG4gICAqIFRoZSBxdWVyeSBwYXJhbWV0ZXJzICh1cCB0byAxMCkgdG8gaW5jbHVkZSBpbiB0aGUgcmVkaXJlY3QgcmVxdWVzdCB0byB0aGUgYXV0aG9yaXphdGlvbiBlbmRwb2ludC5cbiAgICpcbiAgICogQGRlZmF1bHQgLSBObyBleHRyYSBwYXJhbWV0ZXJzXG4gICAqL1xuICByZWFkb25seSBhdXRoZW50aWNhdGlvblJlcXVlc3RFeHRyYVBhcmFtcz86IFJlY29yZDxzdHJpbmcsIHN0cmluZz47XG5cbiAgLyoqXG4gICAqIFRoZSBhdXRob3JpemF0aW9uIGVuZHBvaW50IG9mIHRoZSBJZFAuXG4gICAqXG4gICAqIFRoaXMgbXVzdCBiZSBhIGZ1bGwgVVJMLCBpbmNsdWRpbmcgdGhlIEhUVFBTIHByb3RvY29sLCB0aGUgZG9tYWluLCBhbmQgdGhlIHBhdGguXG4gICAqL1xuICByZWFkb25seSBhdXRob3JpemF0aW9uRW5kcG9pbnQ6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIE9BdXRoIDIuMCBjbGllbnQgaWRlbnRpZmllci5cbiAgICovXG4gIHJlYWRvbmx5IGNsaWVudElkOiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBPQXV0aCAyLjAgY2xpZW50IHNlY3JldC5cbiAgICovXG4gIHJlYWRvbmx5IGNsaWVudFNlY3JldDogU2VjcmV0VmFsdWU7XG5cbiAgLyoqXG4gICAqIFRoZSBPSURDIGlzc3VlciBpZGVudGlmaWVyIG9mIHRoZSBJZFAuXG4gICAqXG4gICAqIFRoaXMgbXVzdCBiZSBhIGZ1bGwgVVJMLCBpbmNsdWRpbmcgdGhlIEhUVFBTIHByb3RvY29sLCB0aGUgZG9tYWluLCBhbmQgdGhlIHBhdGguXG4gICAqL1xuICByZWFkb25seSBpc3N1ZXI6IHN0cmluZztcblxuICAvKipcbiAgICogVGhlIGJlaGF2aW9yIGlmIHRoZSB1c2VyIGlzIG5vdCBhdXRoZW50aWNhdGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBVbmF1dGhlbnRpY2F0ZWRBY3Rpb24uQVVUSEVOVElDQVRFXG4gICAqL1xuICByZWFkb25seSBvblVuYXV0aGVudGljYXRlZFJlcXVlc3Q/OiBVbmF1dGhlbnRpY2F0ZWRBY3Rpb247XG5cbiAgLyoqXG4gICAqIFRoZSBzZXQgb2YgdXNlciBjbGFpbXMgdG8gYmUgcmVxdWVzdGVkIGZyb20gdGhlIElkUC5cbiAgICpcbiAgICogVG8gdmVyaWZ5IHdoaWNoIHNjb3BlIHZhbHVlcyB5b3VyIElkUCBzdXBwb3J0cyBhbmQgaG93IHRvIHNlcGFyYXRlIG11bHRpcGxlIHZhbHVlcywgc2VlIHRoZSBkb2N1bWVudGF0aW9uIGZvciB5b3VyIElkUC5cbiAgICpcbiAgICogQGRlZmF1bHQgXCJvcGVuaWRcIlxuICAgKi9cbiAgcmVhZG9ubHkgc2NvcGU/OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSBuYW1lIG9mIHRoZSBjb29raWUgdXNlZCB0byBtYWludGFpbiBzZXNzaW9uIGluZm9ybWF0aW9uLlxuICAgKlxuICAgKiBAZGVmYXVsdCBcIkFXU0VMQkF1dGhTZXNzaW9uQ29va2llXCJcbiAgICovXG4gIHJlYWRvbmx5IHNlc3Npb25Db29raWVOYW1lPzogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgbWF4aW11bSBkdXJhdGlvbiBvZiB0aGUgYXV0aGVudGljYXRpb24gc2Vzc2lvbi5cbiAgICpcbiAgICogQGRlZmF1bHQgRHVyYXRpb24uZGF5cyg3KVxuICAgKi9cbiAgcmVhZG9ubHkgc2Vzc2lvblRpbWVvdXQ/OiBEdXJhdGlvbjtcblxuICAvKipcbiAgICogVGhlIHRva2VuIGVuZHBvaW50IG9mIHRoZSBJZFAuXG4gICAqXG4gICAqIFRoaXMgbXVzdCBiZSBhIGZ1bGwgVVJMLCBpbmNsdWRpbmcgdGhlIEhUVFBTIHByb3RvY29sLCB0aGUgZG9tYWluLCBhbmQgdGhlIHBhdGguXG4gICAqL1xuICByZWFkb25seSB0b2tlbkVuZHBvaW50OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIFRoZSB1c2VyIGluZm8gZW5kcG9pbnQgb2YgdGhlIElkUC5cbiAgICpcbiAgICogVGhpcyBtdXN0IGJlIGEgZnVsbCBVUkwsIGluY2x1ZGluZyB0aGUgSFRUUFMgcHJvdG9jb2wsIHRoZSBkb21haW4sIGFuZCB0aGUgcGF0aC5cbiAgICovXG4gIHJlYWRvbmx5IHVzZXJJbmZvRW5kcG9pbnQ6IHN0cmluZztcbn1cblxuLyoqXG4gKiBXaGF0IHRvIGRvIHdpdGggdW5hdXRoZW50aWNhdGVkIHJlcXVlc3RzXG4gKi9cbmV4cG9ydCBlbnVtIFVuYXV0aGVudGljYXRlZEFjdGlvbiB7XG4gIC8qKlxuICAgKiBSZXR1cm4gYW4gSFRUUCA0MDEgVW5hdXRob3JpemVkIGVycm9yLlxuICAgKi9cbiAgREVOWSA9ICdkZW55JyxcblxuICAvKipcbiAgICogQWxsb3cgdGhlIHJlcXVlc3QgdG8gYmUgZm9yd2FyZGVkIHRvIHRoZSB0YXJnZXQuXG4gICAqL1xuICBBTExPVyA9ICdhbGxvdycsXG5cbiAgLyoqXG4gICAqIFJlZGlyZWN0IHRoZSByZXF1ZXN0IHRvIHRoZSBJZFAgYXV0aG9yaXphdGlvbiBlbmRwb2ludC5cbiAgICovXG4gIEFVVEhFTlRJQ0FURSA9ICdhdXRoZW50aWNhdGUnLFxufVxuXG4vKipcbiAqIExpc3RlbmVyIEFjdGlvbiB0aGF0IGNhbGxzIFwicmVnaXN0ZXJMaXN0ZW5lclwiIG9uIFRhcmdldEdyb3Vwc1xuICovXG5jbGFzcyBUYXJnZXRHcm91cExpc3RlbmVyQWN0aW9uIGV4dGVuZHMgTGlzdGVuZXJBY3Rpb24ge1xuICBjb25zdHJ1Y3Rvcihwcml2YXRlIHJlYWRvbmx5IHRhcmdldEdyb3VwczogSUFwcGxpY2F0aW9uVGFyZ2V0R3JvdXBbXSwgYWN0aW9uSnNvbjogQ2ZuTGlzdGVuZXIuQWN0aW9uUHJvcGVydHkpIHtcbiAgICBzdXBlcihhY3Rpb25Kc29uKTtcbiAgfVxuXG4gIHB1YmxpYyBiaW5kKF9zY29wZTogQ29uc3RydWN0LCBsaXN0ZW5lcjogSUFwcGxpY2F0aW9uTGlzdGVuZXIsIGFzc29jaWF0aW5nQ29uc3RydWN0PzogSUNvbnN0cnVjdCkge1xuICAgIGZvciAoY29uc3QgdGcgb2YgdGhpcy50YXJnZXRHcm91cHMpIHtcbiAgICAgIHRnLnJlZ2lzdGVyTGlzdGVuZXIobGlzdGVuZXIsIGFzc29jaWF0aW5nQ29uc3RydWN0KTtcbiAgICB9XG4gIH1cbn0iXX0=