"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpCodeTarget = exports.HttpCodeElb = exports.ApplicationLoadBalancer = void 0;
const cloudwatch = require("@aws-cdk/aws-cloudwatch");
const ec2 = require("@aws-cdk/aws-ec2");
const core_1 = require("@aws-cdk/core");
const base_load_balancer_1 = require("../shared/base-load-balancer");
const application_listener_1 = require("./application-listener");
/**
 * Define an Application Load Balancer
 *
 * @resource AWS::ElasticLoadBalancingV2::LoadBalancer
 */
class ApplicationLoadBalancer extends base_load_balancer_1.BaseLoadBalancer {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'application',
            securityGroups: core_1.Lazy.listValue({ produce: () => [this.securityGroup.securityGroupId] }),
            ipAddressType: props.ipAddressType,
        });
        this.securityGroup = props.securityGroup || new ec2.SecurityGroup(this, 'SecurityGroup', {
            vpc: props.vpc,
            description: `Automatically created Security Group for ELB ${this.node.uniqueId}`,
            allowAllOutbound: false,
        });
        this.connections = new ec2.Connections({ securityGroups: [this.securityGroup] });
        if (props.http2Enabled === false) {
            this.setAttribute('routing.http2.enabled', 'false');
        }
        if (props.idleTimeout !== undefined) {
            this.setAttribute('idle_timeout.timeout_seconds', props.idleTimeout.toSeconds().toString());
        }
    }
    /**
     * Import an existing Application Load Balancer
     */
    static fromApplicationLoadBalancerAttributes(scope, id, attrs) {
        return new ImportedApplicationLoadBalancer(scope, id, attrs);
    }
    /**
     * Add a new listener to this load balancer
     */
    addListener(id, props) {
        return new application_listener_1.ApplicationListener(this, id, {
            loadBalancer: this,
            ...props,
        });
    }
    /**
     * Return the given named metric for this Application Load Balancer
     *
     * @default Average over 5 minutes
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/ApplicationELB',
            metricName,
            dimensions: { LoadBalancer: this.loadBalancerFullName },
            ...props,
        });
    }
    /**
     * The total number of concurrent TCP connections active from clients to the
     * load balancer and from the load balancer to targets.
     *
     * @default Sum over 5 minutes
     */
    metricActiveConnectionCount(props) {
        return this.metric('ActiveConnectionCount', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * The number of TLS connections initiated by the client that did not
     * establish a session with the load balancer. Possible causes include a
     * mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     */
    metricClientTlsNegotiationErrorCount(props) {
        return this.metric('ClientTLSNegotiationErrorCount', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * The number of load balancer capacity units (LCU) used by your load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricConsumedLCUs(props) {
        return this.metric('ConsumedLCUs', {
            statistic: 'sum',
            ...props,
        });
    }
    /**
     * The number of fixed-response actions that were successful.
     *
     * @default Sum over 5 minutes
     */
    metricHttpFixedResponseCount(props) {
        return this.metric('HTTP_Fixed_Response_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of redirect actions that were successful.
     *
     * @default Sum over 5 minutes
     */
    metricHttpRedirectCount(props) {
        return this.metric('HTTP_Redirect_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of redirect actions that couldn't be completed because the URL
     * in the response location header is larger than 8K.
     *
     * @default Sum over 5 minutes
     */
    metricHttpRedirectUrlLimitExceededCount(props) {
        return this.metric('HTTP_Redirect_Url_Limit_Exceeded_Count', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of HTTP 3xx/4xx/5xx codes that originate from the load balancer.
     *
     * This does not include any response codes generated by the targets.
     *
     * @default Sum over 5 minutes
     */
    metricHttpCodeElb(code, props) {
        return this.metric(code, {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of HTTP 2xx/3xx/4xx/5xx response codes generated by all targets
     * in the load balancer.
     *
     * This does not include any response codes generated by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricHttpCodeTarget(code, props) {
        return this.metric(code, {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of bytes processed by the load balancer over IPv6.
     *
     * @default Sum over 5 minutes
     */
    metricIpv6ProcessedBytes(props) {
        return this.metric('IPv6ProcessedBytes', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of IPv6 requests received by the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricIpv6RequestCount(props) {
        return this.metric('IPv6RequestCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of new TCP connections established from clients to the
     * load balancer and from the load balancer to targets.
     *
     * @default Sum over 5 minutes
     */
    metricNewConnectionCount(props) {
        return this.metric('NewConnectionCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The total number of bytes processed by the load balancer over IPv4 and IPv6.
     *
     * @default Sum over 5 minutes
     */
    metricProcessedBytes(props) {
        return this.metric('ProcessedBytes', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of connections that were rejected because the load balancer had
     * reached its maximum number of connections.
     *
     * @default Sum over 5 minutes
     */
    metricRejectedConnectionCount(props) {
        return this.metric('RejectedConnectionCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of requests processed over IPv4 and IPv6.
     *
     * This count includes only the requests with a response generated by a target of the load balancer.
     *
     * @default Sum over 5 minutes
     */
    metricRequestCount(props) {
        return this.metric('RequestCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of rules processed by the load balancer given a request rate averaged over an hour.
     *
     * @default Sum over 5 minutes
     */
    metricRuleEvaluations(props) {
        return this.metric('RuleEvaluations', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of connections that were not successfully established between the load balancer and target.
     *
     * @default Sum over 5 minutes
     */
    metricTargetConnectionErrorCount(props) {
        return this.metric('TargetConnectionErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The time elapsed, in seconds, after the request leaves the load balancer until a response from the target is received.
     *
     * @default Average over 5 minutes
     */
    metricTargetResponseTime(props) {
        return this.metric('TargetResponseTime', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of TLS connections initiated by the load balancer that did not establish a session with the target.
     *
     * Possible causes include a mismatch of ciphers or protocols.
     *
     * @default Sum over 5 minutes
     */
    metricTargetTLSNegotiationErrorCount(props) {
        return this.metric('TargetTLSNegotiationErrorCount', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of user authentications that could not be completed
     *
     * Because an authenticate action was misconfigured, the load balancer
     * couldn't establish a connection with the IdP, or the load balancer
     * couldn't complete the authentication flow due to an internal error.
     *
     * @default Sum over 5 minutes
     */
    metricElbAuthError(props) {
        return this.metric('ELBAuthError', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The number of user authentications that could not be completed because the
     * IdP denied access to the user or an authorization code was used more than
     * once.
     *
     * @default Sum over 5 minutes
     */
    metricElbAuthFailure(props) {
        return this.metric('ELBAuthFailure', {
            statistic: 'Sum',
            ...props,
        });
    }
    /**
     * The time elapsed, in milliseconds, to query the IdP for the ID token and user info.
     *
     * If one or more of these operations fail, this is the time to failure.
     *
     * @default Average over 5 minutes
     */
    metricElbAuthLatency(props) {
        return this.metric('ELBAuthLatency', {
            statistic: 'Average',
            ...props,
        });
    }
    /**
     * The number of authenticate actions that were successful.
     *
     * This metric is incremented at the end of the authentication workflow,
     * after the load balancer has retrieved the user claims from the IdP.
     *
     * @default Sum over 5 minutes
     */
    metricElbAuthSuccess(props) {
        return this.metric('ELBAuthSuccess', {
            statistic: 'Sum',
            ...props,
        });
    }
}
exports.ApplicationLoadBalancer = ApplicationLoadBalancer;
/**
 * Count of HTTP status originating from the load balancer
 *
 * This count does not include any response codes generated by the targets.
 */
var HttpCodeElb;
(function (HttpCodeElb) {
    /**
     * The number of HTTP 3XX redirection codes that originate from the load balancer.
     */
    HttpCodeElb["ELB_3XX_COUNT"] = "HTTPCode_ELB_3XX_Count";
    /**
     * The number of HTTP 4XX client error codes that originate from the load balancer.
     *
     * Client errors are generated when requests are malformed or incomplete.
     * These requests have not been received by the target. This count does not
     * include any response codes generated by the targets.
     */
    HttpCodeElb["ELB_4XX_COUNT"] = "HTTPCode_ELB_4XX_Count";
    /**
     * The number of HTTP 5XX server error codes that originate from the load balancer.
     */
    HttpCodeElb["ELB_5XX_COUNT"] = "HTTPCode_ELB_5XX_Count";
})(HttpCodeElb = exports.HttpCodeElb || (exports.HttpCodeElb = {}));
/**
 * Count of HTTP status originating from the targets
 */
var HttpCodeTarget;
(function (HttpCodeTarget) {
    /**
     * The number of 2xx response codes from targets
     */
    HttpCodeTarget["TARGET_2XX_COUNT"] = "HTTPCode_Target_2XX_Count";
    /**
     * The number of 3xx response codes from targets
     */
    HttpCodeTarget["TARGET_3XX_COUNT"] = "HTTPCode_Target_3XX_Count";
    /**
     * The number of 4xx response codes from targets
     */
    HttpCodeTarget["TARGET_4XX_COUNT"] = "HTTPCode_Target_4XX_Count";
    /**
     * The number of 5xx response codes from targets
     */
    HttpCodeTarget["TARGET_5XX_COUNT"] = "HTTPCode_Target_5XX_Count";
})(HttpCodeTarget = exports.HttpCodeTarget || (exports.HttpCodeTarget = {}));
/**
 * An ApplicationLoadBalancer that has been defined elsewhere
 */
class ImportedApplicationLoadBalancer extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        this.props = props;
        this.vpc = props.vpc;
        this.loadBalancerArn = props.loadBalancerArn;
        this.connections = new ec2.Connections({
            securityGroups: [ec2.SecurityGroup.fromSecurityGroupId(this, 'SecurityGroup', props.securityGroupId, {
                    allowAllOutbound: props.securityGroupAllowsAllOutbound,
                })],
        });
    }
    addListener(id, props) {
        return new application_listener_1.ApplicationListener(this, id, {
            loadBalancer: this,
            ...props,
        });
    }
    get loadBalancerCanonicalHostedZoneId() {
        if (this.props.loadBalancerCanonicalHostedZoneId) {
            return this.props.loadBalancerCanonicalHostedZoneId;
        }
        // tslint:disable-next-line:max-line-length
        throw new Error(`'loadBalancerCanonicalHostedZoneId' was not provided when constructing Application Load Balancer ${this.node.path} from attributes`);
    }
    get loadBalancerDnsName() {
        if (this.props.loadBalancerDnsName) {
            return this.props.loadBalancerDnsName;
        }
        // tslint:disable-next-line:max-line-length
        throw new Error(`'loadBalancerDnsName' was not provided when constructing Application Load Balancer ${this.node.path} from attributes`);
    }
}
//# sourceMappingURL=data:application/json;base64,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