# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['pangea', 'pangea.services', 'pangea.services.audit']

package_data = \
{'': ['*']}

install_requires = \
['alive-progress>=2.4.1,<3.0.0',
 'cryptography==37.0.2',
 'pydantic>=1.10.2,<2.0.0',
 'python-dateutil>=2.8.2,<3.0.0',
 'requests>=2.27.1,<3.0.0',
 'schema>=0.7.5,<0.8.0']

setup_kwargs = {
    'name': 'pangea-sdk',
    'version': '0.4.0',
    'description': 'Pangea API SDK',
    'long_description': '<p>\n  <br />\n  <a href="https://pangea.cloud?utm_source=github&utm_medium=node-sdk" target="_blank" rel="noopener noreferrer">\n    <img src="https://pangea-marketing.s3.us-west-2.amazonaws.com/pangea-color.svg" alt="Pangea Logo" height="40">\n  </a>\n  <br />\n</p>\n\n<p>\n<br />\n\n[![documentation](https://img.shields.io/badge/documentation-pangea-blue?style=for-the-badge&labelColor=551B76)](https://pangea.cloud/docs/sdk/python/)\n[![Discord](https://img.shields.io/discord/1017567751818182786?color=%23551b76&label=Discord&logo=discord&logoColor=%23FFFFFF&style=for-the-badge)](https://discord.gg/z7yXhC7cQr)\n\n<br />\n</p>\n\n# Pangea Python SDK\n\nA Python SDK for integrating with Pangea Services.\n\n## Setup\n\n```\npip3 install python-pangea\n# or\npoetry add python-pangea\n```\n\n## Usage\n\n### Secure Audit Service - Log Data\n\n```\nimport os\nfrom pangea.config import PangeaConfig\nfrom pangea.services import Audit\n\n# Read your project domain from an env variable\ndomain = os.getenv("PANGEA_DOMAIN)\n\n# Read your access token from an env variable\ntoken = os.getenv("PANGEA_TOKEN")\n\n# Read the Audit Config ID from an env variable,\n# required for tokens enabled for all services\nconfig_id = os.getenv("AUDIT_CONFIG_ID")\n\n# Create a Config object contain the Audit Config ID\nconfig = PangeaConfig(domain=domain, config_id=config_id)\n\n# Initialize an Audit instance using the config object\naudit = Audit(token, config=config)\n\n# Create test data\n# All input fields are listed, only `message` is required\nevent = {\n    "action": "reboot",\n    "actor": "villan",\n    "target": "world",\n    "status": "error",\n    "source": "test",\n    "old" : "on",\n    "new" : "restart",\n    "message": "despicable act prevented",\n}\n\nresponse = audit.log(event)\n\nprint(response.result)\n```\n\n### Secure Audit Service - Search Data\n\n```\nimport os\nfrom pangea.config import PangeaConfig\nfrom pangea.services import Audit\n\n# Read your access token from an env variable\ntoken = os.getenv("PANGEA_TOKEN")\n\n# Read the Audit Config ID from an env variable\nconfig_id = os.getenv("AUDIT_CONFIG_ID")\n\n# Create a Config object contain the Audit Config ID\nconfig = PangeaConfig(config_id=config_id)\n\n# Initialize an Audit instance using the config object\naudit = Audit(token, config=config)\n\n# Search for \'message\' containing \'reboot\'\n# filtered on \'source=test\', with 5 results per-page\nresponse = audit.search(\n        query="message:prevented",\n        limit=5\n    )\n\nif response.success:\n    print("Search Request ID:", response.request_id, "\\n")\n\n    print(\n        f"Found {response.result.count} event(s)",\n    )\n    for row in response.result.events:\n        print(f"{row.event.received_at}\\taction: {row.event.actor}\\taction: {row.event.action}\\ttarget: {row.event.target}\\tstatus: {row.event.status}\\tmessage: {row.event.message}")\n\nelse:\n    print("Search Failed:", response.code, response.status)\n```\n\n### Secure Audit Service - Integrity Tools\n\n#### Verify audit data\n\nVerify that an event or a list of events has not been tampered with. Usage:\n\n```\nusage: python -m pangea.verify_audit [-h] [--file PATH]\n\nPangea Audit Verifier\n\noptions:\n  -h, --help            show this help message and exit\n  --file PATH, -f PATH  Input file (default: standard input).\n```\n\nIt accepts multiple file formats:\n- a Verification Artifact from the Pangea User Console\n- a search response from the REST API:\n\n```\ncurl -H "Authorization: Bearer ${PANGEA_TOKEN}" -X POST -H \'Content-Type: application/json\'  --data \'{"verbose": true}\' https://audit.dev.aws.pangea.cloud/v1/search\n```\n\n\n\n#### Bulk Download Audit Data\n\nDownload all audit logs for a given time range. Start and end date should be provided,\na variety of formats is supported, including ISO-8601. The result is stored in a\njsonl file (one json per line)\n\n```\nusage: python -m pangea.dump_audit [-h] [--token TOKEN] [--domain DOMAIN] [--config-id CONFIG_ID] [--output OUTPUT] start end\n\nPangea Audit Dump Tool\n\npositional arguments:\n  start                 Start timestamp. Supports a variety of formats, including ISO-8601\n  end                   End timestamp. Supports a variety of formats, including ISO-8601\n\noptions:\n  -h, --help            show this help message and exit\n  --token TOKEN, -t TOKEN\n                        Pangea token (default: env PANGEA_TOKEN)\n  --domain DOMAIN, -d DOMAIN\n                        Pangea base domain (default: env PANGEA_DOMAIN)\n  --config-id CONFIG_ID, -c CONFIG_ID\n                        Audit config id (default: env PANGEA_AUDIT_CONFIG_ID)\n  --output OUTPUT, -o OUTPUT\n                        Output file name. Default: dump-<timestamp>\n```\n\n#### Perform Exhaustive Verification of Audit Data\n\nThis script performs extensive verification on a range of events of the log stream. Appart from verifying the hash\nand the membership proof, it checks that there is no omissions in the stream, i.e. all the events are present and properly located.\n\n```\nusage: python -m pangea.deep_verify [-h] [--token TOKEN] [--domain DOMAIN] [--config-id CONFIG_ID] --file FILE\n\nPangea Audit Event Deep Verifier\n\noptions:\n  -h, --help            show this help message and exit\n  --token TOKEN, -t TOKEN\n                        Pangea token (default: env PANGEA_TOKEN)\n  --domain DOMAIN, -d DOMAIN\n                        Pangea base domain (default: env PANGEA_DOMAIN)\n  --config-id CONFIG_ID, -c CONFIG_ID\n                        Audit config id (default: env PANGEA_AUDIT_CONFIG_ID)\n  --file FILE, -f FILE  Event input file. Must be a collection of JSON Objects separated by newlines\n```\n\nIt accepts multiple file formats:\n- a Verification Artifact from the Pangea User Console\n- a file generated by the `dump_audit` command\n- a search response from the REST API (see `verify_audit`)\n\n\n## Contributing\n\nCurrently, the setup scripts only have support for Mac/ZSH environments.\nFuture support is incoming.\n\nTo install our linters, simply run `./dev/setup_repo.sh`\nThese linters will run on every `git commit` operation.\n\n## Generate SDK Documentation\n\n### Overview\n\nThroughout the SDK, there are Python doc strings that serve as the source of our SDK docs.\n\nThe documentation pipeline here looks like:\n\n1. Write doc strings throughout your Python code. Please refer to existing doc strings as an example of what and how to document.\n1. Make your pull request.\n1. After the pull request is merged, go ahead and run the `parse_module.py` script to generate the JSON docs uses for rendering.\n1. Copy the output from `parse_module.py` and overwrite the existing `python_sdk.json` file in the docs repo. File is located in `platform/docs/openapi/python_sdk.json` in the Pangea monorepo. Save this and make a merge request to update the Python SDK docs in the Pangea monorepo.\n\n### Running the autogen sdk doc script\n\nMake sure you have all the dependencies installed. From the root of the `python-pangea` repo run:\n\n```shell\npoetry install\n```\n\nNow run the script\n\n```shell\npoetry run python parse_module.py\n```\n\nThat will output the script in the terminal. If you\'re on a mac, you can do\n\n```shell\npoetry run python parse_module.py | pbcopy\n```\n\nto copy the output from the script into your clipboard. At the moment, a bunch of stuff will be printed to the terminal if you pipe it to `pbcopy`, but the script still works and copies the output to your clipboard.\n',
    'author': 'Glenn Gallien',
    'author_email': 'glenn.gallien@pangea.cloud',
    'maintainer': 'None',
    'maintainer_email': 'None',
    'url': '',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'python_requires': '>=3.7.2,<4.0.0',
}


setup(**setup_kwargs)
