import struct

import pytest

from bronzebeard import asm


@pytest.mark.parametrize(
    'rd, rs1, aq, rl, code', [
    (0,  0,   0,  0,  0b00010000000000000010000000101111),
    (0,  0,   1,  0,  0b00010100000000000010000000101111),
    (0,  0,   0,  1,  0b00010010000000000010000000101111),
    (0,  0,   1,  1,  0b00010110000000000010000000101111),
    (31, 0,   0,  0,  0b00010000000000000010111110101111),
    (0,  31,  0,  0,  0b00010000000011111010000000101111),
    (31, 31,  0,  0,  0b00010000000011111010111110101111),
    (0,  0,   0,  0,  0b00010000000000000010000000101111),
    (31, 0,   0,  0,  0b00010000000000000010111110101111),
    (0,  31,  0,  0,  0b00010000000011111010000000101111),
    (31, 31,  0,  0,  0b00010000000011111010111110101111),
])
def test_lr_w(rd, rs1, aq, rl, code):
    assert asm.LR_W(rd, rs1, aq=aq, rl=rl) == code


@pytest.mark.parametrize(
    'rd, rs1, rs2, aq, rl, code', [
    (0,  0,   0,   0,  0,  0b00011000000000000010000000101111),
    (0,  0,   0,   1,  0,  0b00011100000000000010000000101111),
    (0,  0,   0,   0,  1,  0b00011010000000000010000000101111),
    (0,  0,   0,   1,  1,  0b00011110000000000010000000101111),
    (31, 0,   0,   0,  0,  0b00011000000000000010111110101111),
    (0,  31,  0,   0,  0,  0b00011000000011111010000000101111),
    (31, 31,  0,   0,  0,  0b00011000000011111010111110101111),
    (0,  0,   31,  0,  0,  0b00011001111100000010000000101111),
    (31, 0,   31,  0,  0,  0b00011001111100000010111110101111),
    (0,  31,  31,  0,  0,  0b00011001111111111010000000101111),
    (31, 31,  31,  0,  0,  0b00011001111111111010111110101111),
])
def test_sc_w(rd, rs1, rs2, aq, rl, code):
    assert asm.SC_W(rd, rs1, rs2, aq=aq, rl=rl) == code


@pytest.mark.parametrize(
    'rd, rs1, rs2, aq, rl, code', [
    (0,  0,   0,   0,  0,  0b00001000000000000010000000101111),
    (0,  0,   0,   1,  0,  0b00001100000000000010000000101111),
    (0,  0,   0,   0,  1,  0b00001010000000000010000000101111),
    (0,  0,   0,   1,  1,  0b00001110000000000010000000101111),
    (31, 0,   0,   0,  0,  0b00001000000000000010111110101111),
    (0,  31,  0,   0,  0,  0b00001000000011111010000000101111),
    (31, 31,  0,   0,  0,  0b00001000000011111010111110101111),
    (0,  0,   31,  0,  0,  0b00001001111100000010000000101111),
    (31, 0,   31,  0,  0,  0b00001001111100000010111110101111),
    (0,  31,  31,  0,  0,  0b00001001111111111010000000101111),
    (31, 31,  31,  0,  0,  0b00001001111111111010111110101111),
])
def test_amoswap_w(rd, rs1, rs2, aq, rl, code):
    assert asm.AMOSWAP_W(rd, rs1, rs2, aq=aq, rl=rl) == code


@pytest.mark.parametrize(
    'rd, rs1, rs2, aq, rl, code', [
    (0,  0,   0,   0,  0,  0b00000000000000000010000000101111),
    (0,  0,   0,   1,  0,  0b00000100000000000010000000101111),
    (0,  0,   0,   0,  1,  0b00000010000000000010000000101111),
    (0,  0,   0,   1,  1,  0b00000110000000000010000000101111),
    (31, 0,   0,   0,  0,  0b00000000000000000010111110101111),
    (0,  31,  0,   0,  0,  0b00000000000011111010000000101111),
    (31, 31,  0,   0,  0,  0b00000000000011111010111110101111),
    (0,  0,   31,  0,  0,  0b00000001111100000010000000101111),
    (31, 0,   31,  0,  0,  0b00000001111100000010111110101111),
    (0,  31,  31,  0,  0,  0b00000001111111111010000000101111),
    (31, 31,  31,  0,  0,  0b00000001111111111010111110101111),
])
def test_amoadd_w(rd, rs1, rs2, aq, rl, code):
    assert asm.AMOADD_W(rd, rs1, rs2, aq=aq, rl=rl) == code


@pytest.mark.parametrize(
    'rd, rs1, rs2, aq, rl, code', [
    (0,  0,   0,   0,  0,  0b00100000000000000010000000101111),
    (0,  0,   0,   1,  0,  0b00100100000000000010000000101111),
    (0,  0,   0,   0,  1,  0b00100010000000000010000000101111),
    (0,  0,   0,   1,  1,  0b00100110000000000010000000101111),
    (31, 0,   0,   0,  0,  0b00100000000000000010111110101111),
    (0,  31,  0,   0,  0,  0b00100000000011111010000000101111),
    (31, 31,  0,   0,  0,  0b00100000000011111010111110101111),
    (0,  0,   31,  0,  0,  0b00100001111100000010000000101111),
    (31, 0,   31,  0,  0,  0b00100001111100000010111110101111),
    (0,  31,  31,  0,  0,  0b00100001111111111010000000101111),
    (31, 31,  31,  0,  0,  0b00100001111111111010111110101111),
])
def test_amoxor_w(rd, rs1, rs2, aq, rl, code):
    assert asm.AMOXOR_W(rd, rs1, rs2, aq=aq, rl=rl) == code


@pytest.mark.parametrize(
    'rd, rs1, rs2, aq, rl, code', [
    (0,  0,   0,   0,  0,  0b01100000000000000010000000101111),
    (0,  0,   0,   1,  0,  0b01100100000000000010000000101111),
    (0,  0,   0,   0,  1,  0b01100010000000000010000000101111),
    (0,  0,   0,   1,  1,  0b01100110000000000010000000101111),
    (31, 0,   0,   0,  0,  0b01100000000000000010111110101111),
    (0,  31,  0,   0,  0,  0b01100000000011111010000000101111),
    (31, 31,  0,   0,  0,  0b01100000000011111010111110101111),
    (0,  0,   31,  0,  0,  0b01100001111100000010000000101111),
    (31, 0,   31,  0,  0,  0b01100001111100000010111110101111),
    (0,  31,  31,  0,  0,  0b01100001111111111010000000101111),
    (31, 31,  31,  0,  0,  0b01100001111111111010111110101111),
])
def test_amoand_w(rd, rs1, rs2, aq, rl, code):
    assert asm.AMOAND_W(rd, rs1, rs2, aq=aq, rl=rl) == code


@pytest.mark.parametrize(
    'rd, rs1, rs2, aq, rl, code', [
    (0,  0,   0,   0,  0,  0b01000000000000000010000000101111),
    (0,  0,   0,   1,  0,  0b01000100000000000010000000101111),
    (0,  0,   0,   0,  1,  0b01000010000000000010000000101111),
    (0,  0,   0,   1,  1,  0b01000110000000000010000000101111),
    (31, 0,   0,   0,  0,  0b01000000000000000010111110101111),
    (0,  31,  0,   0,  0,  0b01000000000011111010000000101111),
    (31, 31,  0,   0,  0,  0b01000000000011111010111110101111),
    (0,  0,   31,  0,  0,  0b01000001111100000010000000101111),
    (31, 0,   31,  0,  0,  0b01000001111100000010111110101111),
    (0,  31,  31,  0,  0,  0b01000001111111111010000000101111),
    (31, 31,  31,  0,  0,  0b01000001111111111010111110101111),
])
def test_amoor_w(rd, rs1, rs2, aq, rl, code):
    assert asm.AMOOR_W(rd, rs1, rs2, aq=aq, rl=rl) == code


@pytest.mark.parametrize(
    'rd, rs1, rs2, aq, rl, code', [
    (0,  0,   0,   0,  0,  0b10000000000000000010000000101111),
    (0,  0,   0,   1,  0,  0b10000100000000000010000000101111),
    (0,  0,   0,   0,  1,  0b10000010000000000010000000101111),
    (0,  0,   0,   1,  1,  0b10000110000000000010000000101111),
    (31, 0,   0,   0,  0,  0b10000000000000000010111110101111),
    (0,  31,  0,   0,  0,  0b10000000000011111010000000101111),
    (31, 31,  0,   0,  0,  0b10000000000011111010111110101111),
    (0,  0,   31,  0,  0,  0b10000001111100000010000000101111),
    (31, 0,   31,  0,  0,  0b10000001111100000010111110101111),
    (0,  31,  31,  0,  0,  0b10000001111111111010000000101111),
    (31, 31,  31,  0,  0,  0b10000001111111111010111110101111),
])
def test_amomin_w(rd, rs1, rs2, aq, rl, code):
    assert asm.AMOMIN_W(rd, rs1, rs2, aq=aq, rl=rl) == code


@pytest.mark.parametrize(
    'rd, rs1, rs2, aq, rl, code', [
    (0,  0,   0,   0,  0,  0b10100000000000000010000000101111),
    (0,  0,   0,   1,  0,  0b10100100000000000010000000101111),
    (0,  0,   0,   0,  1,  0b10100010000000000010000000101111),
    (0,  0,   0,   1,  1,  0b10100110000000000010000000101111),
    (31, 0,   0,   0,  0,  0b10100000000000000010111110101111),
    (0,  31,  0,   0,  0,  0b10100000000011111010000000101111),
    (31, 31,  0,   0,  0,  0b10100000000011111010111110101111),
    (0,  0,   31,  0,  0,  0b10100001111100000010000000101111),
    (31, 0,   31,  0,  0,  0b10100001111100000010111110101111),
    (0,  31,  31,  0,  0,  0b10100001111111111010000000101111),
    (31, 31,  31,  0,  0,  0b10100001111111111010111110101111),
])
def test_amomax_w(rd, rs1, rs2, aq, rl, code):
    assert asm.AMOMAX_W(rd, rs1, rs2, aq=aq, rl=rl) == code


@pytest.mark.parametrize(
    'rd, rs1, rs2, aq, rl, code', [
    (0,  0,   0,   0,  0,  0b11000000000000000010000000101111),
    (0,  0,   0,   1,  0,  0b11000100000000000010000000101111),
    (0,  0,   0,   0,  1,  0b11000010000000000010000000101111),
    (0,  0,   0,   1,  1,  0b11000110000000000010000000101111),
    (31, 0,   0,   0,  0,  0b11000000000000000010111110101111),
    (0,  31,  0,   0,  0,  0b11000000000011111010000000101111),
    (31, 31,  0,   0,  0,  0b11000000000011111010111110101111),
    (0,  0,   31,  0,  0,  0b11000001111100000010000000101111),
    (31, 0,   31,  0,  0,  0b11000001111100000010111110101111),
    (0,  31,  31,  0,  0,  0b11000001111111111010000000101111),
    (31, 31,  31,  0,  0,  0b11000001111111111010111110101111),
])
def test_amominu_w(rd, rs1, rs2, aq, rl, code):
    assert asm.AMOMINU_W(rd, rs1, rs2, aq=aq, rl=rl) == code


@pytest.mark.parametrize(
    'rd, rs1, rs2, aq, rl, code', [
    (0,  0,   0,   0,  0,  0b11100000000000000010000000101111),
    (0,  0,   0,   1,  0,  0b11100100000000000010000000101111),
    (0,  0,   0,   0,  1,  0b11100010000000000010000000101111),
    (0,  0,   0,   1,  1,  0b11100110000000000010000000101111),
    (31, 0,   0,   0,  0,  0b11100000000000000010111110101111),
    (0,  31,  0,   0,  0,  0b11100000000011111010000000101111),
    (31, 31,  0,   0,  0,  0b11100000000011111010111110101111),
    (0,  0,   31,  0,  0,  0b11100001111100000010000000101111),
    (31, 0,   31,  0,  0,  0b11100001111100000010111110101111),
    (0,  31,  31,  0,  0,  0b11100001111111111010000000101111),
    (31, 31,  31,  0,  0,  0b11100001111111111010111110101111),
])
def test_amomaxu_w(rd, rs1, rs2, aq, rl, code):
    assert asm.AMOMAXU_W(rd, rs1, rs2, aq=aq, rl=rl) == code


@pytest.mark.parametrize(
    'source,               expected', [
    ('lr.w      x0 x1',    asm.LR_W('x0', 'x1')),
    ('sc.w      x0 x1 x2', asm.SC_W('x0', 'x1', 'x2')),
    ('amoswap.w x0 x1 x2', asm.AMOSWAP_W('x0', 'x1', 'x2')),
    ('amoadd.w  x0 x1 x2', asm.AMOADD_W('x0', 'x1', 'x2')),
    ('amoxor.w  x0 x1 x2', asm.AMOXOR_W('x0', 'x1', 'x2')),
    ('amoand.w  x0 x1 x2', asm.AMOAND_W('x0', 'x1', 'x2')),
    ('amoor.w   x0 x1 x2', asm.AMOOR_W('x0', 'x1', 'x2')),
    ('amomin.w  x0 x1 x2', asm.AMOMIN_W('x0', 'x1', 'x2')),
    ('amomax.w  x0 x1 x2', asm.AMOMAX_W('x0', 'x1', 'x2')),
    ('amominu.w x0 x1 x2', asm.AMOMINU_W('x0', 'x1', 'x2')),
    ('amomaxu.w x0 x1 x2', asm.AMOMAXU_W('x0', 'x1', 'x2')),
])
def test_assemble_ext_a(source, expected):
    binary = asm.assemble(source)
    target = struct.pack('<I', expected)
    assert binary == target


def test_assemble_aq_rl():
    source = r"""
    lr.w zero zero
    sc.w zero zero zero 0 0
    sc.w zero zero zero 1 0
    sc.w zero zero zero 0 1
    sc.w zero zero zero 1 1
    amomaxu.w t0 t1 t2
    """
    binary = asm.assemble(source)
    target = b''.join(struct.pack('<I', inst) for inst in [
        asm.LR_W(0, 0),
        asm.SC_W(0, 0, 0, aq=0, rl=0),
        asm.SC_W(0, 0, 0, aq=1, rl=0),
        asm.SC_W(0, 0, 0, aq=0, rl=1),
        asm.SC_W(0, 0, 0, aq=1, rl=1),
        asm.AMOMAXU_W('t0', 't1', 't2'),
    ])
    assert binary == target
