"use strict";
var _a, _b, _c, _d, _e, _f;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ConfigurationSource = exports.ConfigurationContent = exports.LambdaValidator = exports.JsonSchemaValidator = exports.ConfigurationSourceType = exports.ValidatorType = exports.ConfigurationType = exports.SourcedConfiguration = exports.HostedConfiguration = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const fs = require("fs");
const aws_cdk_lib_1 = require("aws-cdk-lib");
const aws_appconfig_1 = require("aws-cdk-lib/aws-appconfig");
const iam = require("aws-cdk-lib/aws-iam");
const constructs_1 = require("constructs");
const deployment_strategy_1 = require("./deployment-strategy");
const extension_1 = require("./extension");
const hash_1 = require("./private/hash");
class ConfigurationBase extends constructs_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        this.name = props.name || aws_cdk_lib_1.Names.uniqueResourceName(this, {
            maxLength: 128,
            separator: '-',
        });
        this.application = props.application;
        this.applicationId = this.application.applicationId;
        this.type = props.type;
        this.validators = props.validators;
        this.description = props.description;
        this.deployTo = props.deployTo;
        this.deploymentKey = props.deploymentKey;
        this.deploymentStrategy = props.deploymentStrategy || new deployment_strategy_1.DeploymentStrategy(this, 'DeploymentStrategy', {
            rolloutStrategy: deployment_strategy_1.RolloutStrategy.CANARY_10_PERCENT_20_MINUTES,
        });
    }
    /**
     * Adds an extension defined by the action point and event destination
     * and also creates an extension association to the configuration profile.
     *
     * @param actionPoint The action point which triggers the event
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    on(actionPoint, eventDestination, options) {
        this.extensible.on(actionPoint, eventDestination, options);
    }
    /**
     * Adds a PRE_CREATE_HOSTED_CONFIGURATION_VERSION extension with the
     * provided event destination and also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    preCreateHostedConfigurationVersion(eventDestination, options) {
        this.extensible.preCreateHostedConfigurationVersion(eventDestination, options);
    }
    /**
     * Adds a PRE_START_DEPLOYMENT extension with the provided event destination
     * and also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    preStartDeployment(eventDestination, options) {
        this.extensible.preStartDeployment(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_START extension with the provided event destination
     * and also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentStart(eventDestination, options) {
        this.extensible.onDeploymentStart(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_STEP extension with the provided event destination
     * and also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentStep(eventDestination, options) {
        this.extensible.onDeploymentStep(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_BAKING extension with the provided event destination and
     * also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentBaking(eventDestination, options) {
        this.extensible.onDeploymentBaking(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_COMPLETE extension with the provided event destination
     * and also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentComplete(eventDestination, options) {
        this.extensible.onDeploymentComplete(eventDestination, options);
    }
    /**
     * Adds an ON_DEPLOYMENT_ROLLED_BACK extension with the provided event destination
     * and also creates an extension association to the configuration profile.
     *
     * @param eventDestination The event that occurs during the extension
     * @param options Options for the extension
     */
    onDeploymentRolledBack(eventDestination, options) {
        this.extensible.onDeploymentRolledBack(eventDestination, options);
    }
    /**
     * Adds an extension association to the configuration profile.
     *
     * @param extension The extension to create an association for
     */
    addExtension(extension) {
        this.extensible.addExtension(extension);
    }
    addExistingEnvironmentsToApplication() {
        this.deployTo?.forEach((environment) => {
            if (!this.application.environments.includes(environment)) {
                this.application.addExistingEnvironment(environment);
            }
        });
    }
    deployConfigToEnvironments() {
        if (this.application.environments.length == 0) {
            this.application.addEnvironment('Environment', {
                description: this.description,
            });
        }
        if ((!this.deployTo && this.application.environments.length > 1) || !this.versionNumber) {
            return;
        }
        this.application.environments.forEach((environment) => {
            if ((this.deployTo && !this.deployTo.includes(environment))) {
                return;
            }
            const logicalId = `Deployment${this.getDeploymentHash(environment)}`;
            if (this.node.tryFindChild(logicalId)) {
                return;
            }
            new aws_appconfig_1.CfnDeployment(this, logicalId, {
                applicationId: this.application.applicationId,
                configurationProfileId: this.configurationProfileId,
                deploymentStrategyId: this.deploymentStrategy.deploymentStrategyId,
                environmentId: environment.environmentId,
                configurationVersion: this.versionNumber,
                description: this.description,
                kmsKeyIdentifier: this.deploymentKey?.keyArn,
            });
        });
    }
}
class HostedConfiguration extends ConfigurationBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_HostedConfigurationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, HostedConfiguration);
            }
            throw error;
        }
        this._cfnConfigurationProfile = new aws_appconfig_1.CfnConfigurationProfile(this, 'ConfigurationProfile', {
            applicationId: this.applicationId,
            locationUri: 'hosted',
            name: this.name,
            description: this.description,
            type: this.type,
            validators: this.validators,
        });
        this.configurationProfileId = this._cfnConfigurationProfile.ref;
        this.configurationProfileArn = aws_cdk_lib_1.Stack.of(this).formatArn({
            service: 'appconfig',
            resource: 'application',
            resourceName: `${this.applicationId}/configurationprofile/${this.configurationProfileId}`,
        });
        this.extensible = new extension_1.ExtensibleBase(scope, this.configurationProfileArn, this.name);
        this.content = props.content.content;
        this.contentType = props.contentType || 'application/json';
        this.latestVersionNumber = props.latestVersionNumber;
        this.versionLabel = props.versionLabel;
        this._cfnHostedConfigurationVersion = new aws_appconfig_1.CfnHostedConfigurationVersion(this, 'Resource', {
            applicationId: this.applicationId,
            configurationProfileId: this.configurationProfileId,
            content: this.content,
            contentType: this.contentType,
            description: this.description,
            latestVersionNumber: this.latestVersionNumber,
            versionLabel: this.versionLabel,
        });
        this._cfnHostedConfigurationVersion.applyRemovalPolicy(aws_cdk_lib_1.RemovalPolicy.RETAIN);
        this.versionNumber = this._cfnHostedConfigurationVersion.ref;
        this.hostedConfigurationVersionArn = aws_cdk_lib_1.Stack.of(this).formatArn({
            service: 'appconfig',
            resource: 'application',
            resourceName: `${this.applicationId}/configurationprofile/${this.configurationProfileId}/hostedconfigurationversion/${this.versionNumber}`,
        });
        this.addExistingEnvironmentsToApplication();
        this.deployConfigToEnvironments();
    }
    getDeploymentHash(environment) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_IEnvironment(environment);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getDeploymentHash);
            }
            throw error;
        }
        const combinedString = `
      ${this.application.name}
      ${this.name}
      ${environment.name}
      ${this.content}
    `;
        return (0, hash_1.getHash)(combinedString);
    }
}
exports.HostedConfiguration = HostedConfiguration;
_a = JSII_RTTI_SYMBOL_1;
HostedConfiguration[_a] = { fqn: "@aws-cdk/aws-appconfig-alpha.HostedConfiguration", version: "2.91.0-alpha.0" };
class SourcedConfiguration extends ConfigurationBase {
    constructor(scope, id, props) {
        super(scope, id, props);
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_SourcedConfigurationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, SourcedConfiguration);
            }
            throw error;
        }
        this.location = props.location;
        this.locationUri = this.location.locationUri;
        this.versionNumber = props.versionNumber;
        this.sourceKey = this.location.key;
        this.retrievalRole = props.retrievalRole || this.location.type != ConfigurationSourceType.CODE_PIPELINE
            ? new iam.Role(this, 'Role', {
                roleName: aws_cdk_lib_1.PhysicalName.GENERATE_IF_NEEDED,
                assumedBy: new iam.ServicePrincipal('appconfig.amazonaws.com'),
                inlinePolicies: {
                    ['AllowAppConfigReadFromSourcePolicy']: this.getPolicyForRole(),
                },
            })
            : undefined;
        this._cfnConfigurationProfile = new aws_appconfig_1.CfnConfigurationProfile(this, 'Resource', {
            applicationId: this.applicationId,
            locationUri: this.locationUri,
            name: this.name,
            description: this.description,
            retrievalRoleArn: this.retrievalRole?.roleArn,
            type: this.type,
            validators: this.validators,
        });
        this.configurationProfileId = this._cfnConfigurationProfile.ref;
        this.configurationProfileArn = aws_cdk_lib_1.Stack.of(this).formatArn({
            service: 'appconfig',
            resource: 'application',
            resourceName: `${this.applicationId}/configurationprofile/${this.configurationProfileId}`,
        });
        this.extensible = new extension_1.ExtensibleBase(scope, this.configurationProfileArn, this.name);
        this.addExistingEnvironmentsToApplication();
        this.deployConfigToEnvironments();
    }
    getDeploymentHash(environment) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_appconfig_alpha_IEnvironment(environment);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.getDeploymentHash);
            }
            throw error;
        }
        const combinedString = `
      ${this.application.name}
      ${this.name}
      ${environment.name}
      ${this.versionNumber}
      ${this.location.type}
    `;
        return (0, hash_1.getHash)(combinedString);
    }
    getPolicyForRole() {
        const policy = new iam.PolicyStatement({
            effect: iam.Effect.ALLOW,
        });
        const document = new iam.PolicyDocument({
            statements: [policy],
        });
        if (this.location.type == ConfigurationSourceType.SSM_PARAMETER) {
            policy.addActions('ssm:GetParameter');
            policy.addResources(this.locationUri);
        }
        else if (this.location.type == ConfigurationSourceType.SSM_DOCUMENT) {
            policy.addActions('ssm:GetDocument');
            policy.addResources(aws_cdk_lib_1.Stack.of(this).formatArn({
                service: 'ssm',
                resource: 'document',
                resourceName: this.locationUri.split('://')[1],
            }));
        }
        else if (this.location.type == ConfigurationSourceType.S3) {
            const bucketAndObjectKey = this.locationUri.split('://')[1];
            const sep = bucketAndObjectKey.search('/');
            const bucketName = bucketAndObjectKey.substring(0, sep);
            const objectKey = bucketAndObjectKey.substring(sep + 1);
            policy.addActions('s3:GetObject', 's3:GetObjectMetadata', 's3:GetObjectVersion');
            policy.addResources(aws_cdk_lib_1.Stack.of(this).formatArn({
                region: '',
                account: '',
                service: 's3',
                arnFormat: aws_cdk_lib_1.ArnFormat.NO_RESOURCE_NAME,
                resource: `${bucketName}/${objectKey}`,
            }));
            const bucketPolicy = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: [
                    's3:GetBucketLocation',
                    's3:GetBucketVersioning',
                    's3:ListBucket',
                    's3:ListBucketVersions',
                ],
                resources: [
                    aws_cdk_lib_1.Stack.of(this).formatArn({
                        region: '',
                        account: '',
                        service: 's3',
                        arnFormat: aws_cdk_lib_1.ArnFormat.NO_RESOURCE_NAME,
                        resource: bucketName,
                    }),
                ],
            });
            const listBucketsPolicy = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['s3:ListAllMyBuckets'],
                resources: ['*'],
            });
            document.addStatements(bucketPolicy, listBucketsPolicy);
        }
        else {
            policy.addActions('secretsmanager:GetSecretValue');
            policy.addResources(this.locationUri);
        }
        if (this.sourceKey) {
            const keyPolicy = new iam.PolicyStatement({
                effect: iam.Effect.ALLOW,
                actions: ['kms:Decrypt'],
                resources: [this.sourceKey.keyArn],
            });
            document.addStatements(keyPolicy);
        }
        return document;
    }
}
exports.SourcedConfiguration = SourcedConfiguration;
_b = JSII_RTTI_SYMBOL_1;
SourcedConfiguration[_b] = { fqn: "@aws-cdk/aws-appconfig-alpha.SourcedConfiguration", version: "2.91.0-alpha.0" };
/**
 * The configuration type.
 */
var ConfigurationType;
(function (ConfigurationType) {
    ConfigurationType["FREEFORM"] = "AWS.Freeform";
    ConfigurationType["FEATURE_FLAGS"] = "AWS.AppConfig.FeatureFlags";
})(ConfigurationType || (exports.ConfigurationType = ConfigurationType = {}));
/**
 * The validator type.
 */
var ValidatorType;
(function (ValidatorType) {
    ValidatorType["JSON_SCHEMA"] = "JSON_SCHEMA";
    ValidatorType["LAMBDA"] = "LAMBDA";
})(ValidatorType || (exports.ValidatorType = ValidatorType = {}));
/**
 * The configuration source type.
 */
var ConfigurationSourceType;
(function (ConfigurationSourceType) {
    ConfigurationSourceType["S3"] = "S3";
    ConfigurationSourceType["SECRETS_MANAGER"] = "SECRETS_MANAGER";
    ConfigurationSourceType["SSM_PARAMETER"] = "SSM_PARAMETER";
    ConfigurationSourceType["SSM_DOCUMENT"] = "SSM_DOCUMENT";
    ConfigurationSourceType["CODE_PIPELINE"] = "CODE_PIPELINE";
})(ConfigurationSourceType || (exports.ConfigurationSourceType = ConfigurationSourceType = {}));
/**
 * Defines a JSON Schema validator.
 */
class JsonSchemaValidator {
    /**
     * Defines a JSON Schema validator from a file.
     *
     * @param path The path to the file that defines the validator
     */
    static fromFile(path) {
        return {
            content: fs.readFileSync(path).toString(),
            type: ValidatorType.JSON_SCHEMA,
        };
    }
    /**
     * Defines a JSON Schema validator from inline code.
     *
     * @param code The inline code that defines the validator
     */
    static fromInline(code) {
        return {
            content: code,
            type: ValidatorType.JSON_SCHEMA,
        };
    }
}
exports.JsonSchemaValidator = JsonSchemaValidator;
_c = JSII_RTTI_SYMBOL_1;
JsonSchemaValidator[_c] = { fqn: "@aws-cdk/aws-appconfig-alpha.JsonSchemaValidator", version: "2.91.0-alpha.0" };
/**
 * Defines an AWS Lambda validator.
 */
class LambdaValidator {
    /**
     *  Defines an AWS Lambda validator from a Lambda function. This will call
     * `addPermission` to your function to grant AWS AppConfig permissions.
     *
     * @param func The function that defines the validator
     */
    static fromFunction(func) {
        if (!func.permissionsNode.tryFindChild('AppConfigPermission')) {
            func.addPermission('AppConfigPermission', {
                principal: new iam.ServicePrincipal('appconfig.amazonaws.com'),
            });
        }
        return {
            content: func.functionArn,
            type: ValidatorType.LAMBDA,
        };
    }
}
exports.LambdaValidator = LambdaValidator;
_d = JSII_RTTI_SYMBOL_1;
LambdaValidator[_d] = { fqn: "@aws-cdk/aws-appconfig-alpha.LambdaValidator", version: "2.91.0-alpha.0" };
/**
 * Defines the hosted configuration content.
 */
class ConfigurationContent {
    /**
     * Defines the hosted configuration content from a file.
     *
     * @param path The path to the file that defines configuration content
     */
    static fromFile(path) {
        return {
            content: fs.readFileSync(path).toString(),
        };
    }
    /**
     * Defines the hosted configuration content from inline code.
     *
     * @param content The inline code that defines the configuration content
     */
    static fromInline(content) {
        return {
            content,
        };
    }
}
exports.ConfigurationContent = ConfigurationContent;
_e = JSII_RTTI_SYMBOL_1;
ConfigurationContent[_e] = { fqn: "@aws-cdk/aws-appconfig-alpha.ConfigurationContent", version: "2.91.0-alpha.0" };
/**
 * Defines the integrated configuration sources.
 */
class ConfigurationSource {
    /**
     * Defines configuration content from an Amazon S3 bucket.
     *
     * @param bucket The S3 bucket where the configuration is stored
     * @param objectKey The path to the configuration
     * @param key The KMS Key that the bucket is encrypted with
     */
    static fromBucket(bucket, objectKey, key) {
        return {
            locationUri: bucket.s3UrlForObject(objectKey),
            type: ConfigurationSourceType.S3,
            key,
        };
    }
    /**
     * Defines configuration content from an AWS Secrets Manager secret.
     *
     * @param secret The secret where the configuration is stored
     */
    static fromSecret(secret) {
        return {
            locationUri: secret.secretArn,
            type: ConfigurationSourceType.SECRETS_MANAGER,
            key: secret.encryptionKey,
        };
    }
    /**
     * Defines configuration content from a Systems Manager (SSM) Parameter Store parameter.
     *
     * @param parameter The parameter where the configuration is stored
     * @param key The KMS Key that the secure string is encrypted with
     */
    static fromParameter(parameter, key) {
        return {
            locationUri: parameter.parameterArn,
            type: ConfigurationSourceType.SSM_PARAMETER,
            key,
        };
    }
    /**
     * Defines configuration content from a Systems Manager (SSM) document.
     *
     * @param document The SSM document where the configuration is stored
     */
    static fromCfnDocument(document) {
        return {
            locationUri: `ssm-document://${document.ref}`,
            type: ConfigurationSourceType.SSM_DOCUMENT,
        };
    }
    /**
     * Defines configuration content from AWS CodePipeline.
     *
     * @param pipeline The pipeline where the configuration is stored
     * @returns
     */
    static fromPipeline(pipeline) {
        return {
            locationUri: `codepipeline://${pipeline.pipelineName}`,
            type: ConfigurationSourceType.CODE_PIPELINE,
        };
    }
}
exports.ConfigurationSource = ConfigurationSource;
_f = JSII_RTTI_SYMBOL_1;
ConfigurationSource[_f] = { fqn: "@aws-cdk/aws-appconfig-alpha.ConfigurationSource", version: "2.91.0-alpha.0" };
//# sourceMappingURL=data:application/json;base64,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