import { Resource, IResource, Duration } from 'aws-cdk-lib';
import { Construct } from 'constructs';
export interface DeploymentStrategyProps {
    /**
     * The rollout strategy for the deployment strategy. You can use predefined deployment
     * strategies, such as RolloutStrategy.ALL_AT_ONCE, RolloutStrategy.LINEAR_50_PERCENT_EVERY_30_SECONDS,
     * or RolloutStrategy.CANARY_10_PERCENT_20_MINUTES.
     */
    readonly rolloutStrategy: RolloutStrategy;
    /**
     * A name for the deployment strategy.
     *
     * @default - A name is generated.
     */
    readonly name?: string;
    /**
     * A description of the deployment strategy.
     *
     * @default - No description.
     */
    readonly description?: string;
}
/**
 * An AWS AppConfig deployment strategy.
 *
 * @resource AWS::AppConfig::DeploymentStrategy
 * @see https://docs.aws.amazon.com/appconfig/latest/userguide/appconfig-creating-deployment-strategy.html
 */
export declare class DeploymentStrategy extends Resource implements IDeploymentStrategy {
    /**
     * Imports a deployment strategy into the CDK using its Amazon Resource Name (ARN).
     *
     * @param scope The parent construct
     * @param id The name of the deployment strategy construct
     * @param deploymentStrategyArn The Amazon Resource Name (ARN) of the deployment strategy
     */
    static fromDeploymentStrategyArn(scope: Construct, id: string, deploymentStrategyArn: string): IDeploymentStrategy;
    /**
     * Imports a deployment strategy into the CDK using its ID.
     *
     * @param scope The parent construct
     * @param id The name of the deployment strategy construct
     * @param deploymentStrategyId The ID of the deployment strategy
     */
    static fromDeploymentStrategyId(scope: Construct, id: string, deploymentStrategyId: string): IDeploymentStrategy;
    /**
     * The name of the deployment strategy.
     */
    readonly name?: string;
    /**
     * The deployment duration in minutes of the deployment strategy.
     */
    readonly deploymentDurationInMinutes?: number;
    /**
     * The growth factor of the deployment strategy.
     */
    readonly growthFactor?: number;
    /**
     * The description of the deployment strategy.
     */
    readonly description?: string;
    /**
     * The final bake time in minutes of the deployment strategy.
     */
    readonly finalBakeTimeInMinutes?: number;
    /**
     * The growth type of the deployment strategy.
     */
    readonly growthType?: GrowthType;
    /**
     * The ID of the deployment strategy.
     */
    readonly deploymentStrategyId: string;
    /**
     * The Amazon Resource Name (ARN) of the deployment strategy.
     */
    readonly deploymentStrategyArn: string;
    private readonly _cfnDeploymentStrategy;
    constructor(scope: Construct, id: string, props: DeploymentStrategyProps);
}
/**
 * Defines the growth type of the deployment strategy.
 */
export declare enum GrowthType {
    LINEAR = "LINEAR",
    EXPONENTIAL = "EXPONENTIAL"
}
/**
 * Defines the deployment strategy ID's of AWS AppConfig predefined strategies.
 */
export declare enum PredefinedDeploymentStrategyId {
    CANARY_10_PERCENT_20_MINUTES = "AppConfig.Canary10Percent20Minutes",
    LINEAR_50_PERCENT_EVERY_30_SECONDS = "AppConfig.Linear50PercentEvery30Seconds",
    LINEAR_20_PERCENT_EVERY_6_MINUTES = "AppConfig.Linear20PercentEvery6Minutes",
    ALL_AT_ONCE = "AppConfig.AllAtOnce"
}
export interface RolloutStrategyProps {
    /**
     * The growth factor of the deployment strategy. This defines
     * the percentage of targets to receive a deployed configuration
     * during each interval.
     */
    readonly growthFactor: number;
    /**
     * The deployment duration of the deployment strategy. This defines
     * the total amount of time for a deployment to last.
     */
    readonly deploymentDuration: Duration;
    /**
     * The final bake time of the deployment strategy.
     *
     * This setting specifies the amount of time AWS AppConfig monitors for Amazon
     * CloudWatch alarms after the configuration has been deployed to
     * 100% of its targets, before considering the deployment to be complete.
     * If an alarm is triggered during this time, AWS AppConfig rolls back
     * the deployment.
     *
     * @default Duration.minutes(0)
     */
    readonly finalBakeTime?: Duration;
}
/**
 * Defines the rollout strategy for a deployment strategy and includes the growth factor,
 * deployment duration, growth type, and optionally final bake time.
 */
export declare abstract class RolloutStrategy {
    static readonly CANARY_10_PERCENT_20_MINUTES: RolloutStrategy;
    static readonly LINEAR_50_PERCENT_EVERY_30_SECONDS: RolloutStrategy;
    static readonly LINEAR_20_PERCENT_EVERY_6_MINUTES: RolloutStrategy;
    static readonly ALL_AT_ONCE: RolloutStrategy;
    /**
     * @returns A linear rollout strategy.
     */
    static linear(props: RolloutStrategyProps): RolloutStrategy;
    /**
     * @returns An exponential rollout strategy.
     */
    static exponential(props: RolloutStrategyProps): RolloutStrategy;
    /**
     * The growth factor of the rollout strategy.
     */
    abstract readonly growthFactor: number;
    /**
     * The deployment duration of the rollout strategy.
     */
    abstract readonly deploymentDuration: Duration;
    /**
     * The growth type of the rollout strategy.
     */
    abstract readonly growthType?: GrowthType;
    /**
     * The final bake time of the deployment strategy.
     */
    abstract readonly finalBakeTime?: Duration;
}
export interface IDeploymentStrategy extends IResource {
    /**
     * The name of the deployment strategy.
     */
    readonly name?: string;
    /**
     * The deployment duration in minutes.
     */
    readonly deploymentDurationInMinutes?: number;
    /**
     * The growth factor of the deployment strategy.
     */
    readonly growthFactor?: number;
    /**
     * The description of the deployment strategy.
     */
    readonly description?: string;
    /**
     * The final bake time in minutes.
     */
    readonly finalBakeTimeInMinutes?: number;
    /**
     * The growth type of the deployment strategy.
     */
    readonly growthType?: GrowthType;
    /**
     * The ID of the deployment strategy.
     */
    readonly deploymentStrategyId: string;
    /**
     * The Amazon Resource Name (ARN) of the deployment strategy.
     */
    readonly deploymentStrategyArn: string;
}
