# class DocTemplater
import os
import kclvm.tools.docs.model_pb2 as model


def md_schema_section_render(schema_name: str):
    return f"Schema {schema_name}"


def md_module_doc_templater(module: model.ModuleDoc) -> str:
    def table(records, headings) -> str:
        """
        Generate a Markdown table from records.
        records -- Iterable.  Rows will be generated from this.
        headings -- List of column headings.
        """
        table_heading = (
            f"|{'|'.join(headings)}|\n|{'|'.join(['-' * len(h) for h in headings])}|\n"
        )
        table_rows = "\n".join(
            [f"|{record[0]}|{record[1]}|{record[2]}|{record[3]}|" for record in records]
        )
        return f"{table_heading}{table_rows}\n"

    # module header
    header = f"# {module.name}\n\n"
    # source code link
    file_name = module.relative_path.replace(f".{os.sep}", "")
    source_code = (
        f"Source: [{file_name}]({module.source_code_url}{os.sep}{file_name})\n\n"
        if module.source_code_url
        else ""
    )
    # module doc string
    module_doc = f"{module.doc}\n\n" if module.doc else ""
    # schemas header
    schemas_section = ""
    if module.schemas:
        for schema in module.schemas:
            _schema_header = f"## Schema {schema.name}\n\n"
            _schema_docstring = f"{schema.doc}\n\n" if schema.doc else ""

            _base_schema_section = ""
            if schema.base_schema and schema.base_schema.type_str:
                _base_schema_section = (
                    f"### Base Schema\n{schema.base_schema.type_str}\n\n"
                )

            attributes_section = ""
            if schema.attributes:
                attributes_header = "### Attributes\n\n"
                attributes_table = table(
                    [
                        [
                            f"**{a.name}**<br />{a.doc}" if a.doc else f"**{a.name}**",
                            a.type.type_str,
                            a.default_value if a.default_value else "Undefined",
                            "optional" if a.is_optional else "**required**",
                        ]
                        for a in schema.attributes
                    ],
                    [
                        "Name and Description",
                        "Type",
                        "Default Value",
                        "Required",
                    ],
                )
                attributes_section = f"{attributes_header}{attributes_table}"
            # Examples
            _examples = (
                f"### Examples\n```python\n{schema.examples}\n```\n\n"
                if schema.examples
                else ""
            )
            schemas_section = f"{schemas_section}{_schema_header}{_schema_docstring}{_base_schema_section}{attributes_section}{_examples}"
    # not editable footer
    not_editable = "<!-- Auto generated by kcl-doc tool, please do not edit. -->"
    return f"{header}{source_code}{module_doc}{schemas_section}{not_editable}"
