"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const kms = require("@aws-cdk/aws-kms");
const core_1 = require("@aws-cdk/core");
const queue_base_1 = require("./queue-base");
const sqs_generated_1 = require("./sqs.generated");
const validate_props_1 = require("./validate-props");
/**
 * What kind of encryption to apply to this queue
 */
var QueueEncryption;
(function (QueueEncryption) {
    /**
     * Messages in the queue are not encrypted
     */
    QueueEncryption["UNENCRYPTED"] = "NONE";
    /**
     * Server-side KMS encryption with a master key managed by SQS.
     */
    QueueEncryption["KMS_MANAGED"] = "MANAGED";
    /**
     * Server-side encryption with a KMS key managed by the user.
     *
     * If `encryptionKey` is specified, this key will be used, otherwise, one will be defined.
     */
    QueueEncryption["KMS"] = "KMS";
})(QueueEncryption = exports.QueueEncryption || (exports.QueueEncryption = {}));
/**
 * A new Amazon SQS queue
 */
class Queue extends queue_base_1.QueueBase {
    constructor(scope, id, props = {}) {
        super(scope, id, {
            physicalName: props.queueName,
        });
        this.autoCreatePolicy = true;
        validate_props_1.validateProps(props);
        const redrivePolicy = props.deadLetterQueue
            ? {
                deadLetterTargetArn: props.deadLetterQueue.queue.queueArn,
                maxReceiveCount: props.deadLetterQueue.maxReceiveCount,
            }
            : undefined;
        const { encryptionMasterKey, encryptionProps } = _determineEncryptionProps.call(this);
        const fifoProps = this.determineFifoProps(props);
        this.fifo = fifoProps.fifoQueue || false;
        const queue = new sqs_generated_1.CfnQueue(this, 'Resource', {
            queueName: this.physicalName,
            ...fifoProps,
            ...encryptionProps,
            redrivePolicy,
            delaySeconds: props.deliveryDelay && props.deliveryDelay.toSeconds(),
            maximumMessageSize: props.maxMessageSizeBytes,
            messageRetentionPeriod: props.retentionPeriod && props.retentionPeriod.toSeconds(),
            receiveMessageWaitTimeSeconds: props.receiveMessageWaitTime && props.receiveMessageWaitTime.toSeconds(),
            visibilityTimeout: props.visibilityTimeout && props.visibilityTimeout.toSeconds(),
        });
        this.queueArn = this.getResourceArnAttribute(queue.attrArn, {
            service: 'sqs',
            resource: this.physicalName,
        });
        this.queueName = this.getResourceNameAttribute(queue.attrQueueName);
        this.encryptionMasterKey = encryptionMasterKey;
        this.queueUrl = queue.ref;
        function _determineEncryptionProps() {
            let encryption = props.encryption || QueueEncryption.UNENCRYPTED;
            if (encryption !== QueueEncryption.KMS && props.encryptionMasterKey) {
                encryption = QueueEncryption.KMS; // KMS is implied by specifying an encryption key
            }
            if (encryption === QueueEncryption.UNENCRYPTED) {
                return { encryptionProps: {} };
            }
            if (encryption === QueueEncryption.KMS_MANAGED) {
                return {
                    encryptionProps: {
                        kmsMasterKeyId: 'alias/aws/sqs',
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            if (encryption === QueueEncryption.KMS) {
                const masterKey = props.encryptionMasterKey || new kms.Key(this, 'Key', {
                    description: `Created by ${this.node.path}`,
                });
                return {
                    encryptionMasterKey: masterKey,
                    encryptionProps: {
                        kmsMasterKeyId: masterKey.keyArn,
                        kmsDataKeyReusePeriodSeconds: props.dataKeyReuse && props.dataKeyReuse.toSeconds(),
                    },
                };
            }
            throw new Error(`Unexpected 'encryptionType': ${encryption}`);
        }
    }
    static fromQueueArn(scope, id, queueArn) {
        return Queue.fromQueueAttributes(scope, id, { queueArn });
    }
    /**
     * Import an existing queue
     */
    static fromQueueAttributes(scope, id, attrs) {
        const stack = core_1.Stack.of(scope);
        const queueName = attrs.queueName || stack.parseArn(attrs.queueArn).resource;
        const queueUrl = attrs.queueUrl || `https://sqs.${stack.region}.${stack.urlSuffix}/${stack.account}/${queueName}`;
        class Import extends queue_base_1.QueueBase {
            constructor() {
                super(...arguments);
                this.queueArn = attrs.queueArn; // arn:aws:sqs:us-east-1:123456789012:queue1
                this.queueUrl = queueUrl;
                this.queueName = queueName;
                this.encryptionMasterKey = attrs.keyArn
                    ? kms.Key.fromKeyArn(this, 'Key', attrs.keyArn)
                    : undefined;
                this.fifo = queueName.endsWith('.fifo') ? true : false;
                this.autoCreatePolicy = false;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Look at the props, see if the FIFO props agree, and return the correct subset of props
     */
    determineFifoProps(props) {
        // Check if any of the signals that we have say that this is a FIFO queue.
        let fifoQueue = props.fifo;
        const queueName = props.queueName;
        if (typeof fifoQueue === 'undefined' && queueName && !core_1.Token.isUnresolved(queueName) && queueName.endsWith('.fifo')) {
            fifoQueue = true;
        }
        if (typeof fifoQueue === 'undefined' && props.contentBasedDeduplication) {
            fifoQueue = true;
        }
        // If we have a name, see that it agrees with the FIFO setting
        if (typeof queueName === 'string') {
            if (fifoQueue && !queueName.endsWith('.fifo')) {
                throw new Error("FIFO queue names must end in '.fifo'");
            }
            if (!fifoQueue && queueName.endsWith('.fifo')) {
                throw new Error("Non-FIFO queue name may not end in '.fifo'");
            }
        }
        if (props.contentBasedDeduplication && !fifoQueue) {
            throw new Error('Content-based deduplication can only be defined for FIFO queues');
        }
        return {
            contentBasedDeduplication: props.contentBasedDeduplication,
            fifoQueue,
        };
    }
}
exports.Queue = Queue;
//# sourceMappingURL=data:application/json;base64,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