# -------------------------------------------------------------------------
# Copyright (c) Microsoft Corporation. All rights reserved.
# Licensed under the MIT License. See License.txt in the project root for
# license information.
# --------------------------------------------------------------------------
"""Malware Entity class."""
from typing import Any, List, Mapping, Optional

from ..._version import VERSION
from ...common.utility import export
from .entity import Entity
from .file import File
from .process import Process

__version__ = VERSION
__author__ = "Ian Hellen"


# pylint: disable=invalid-name


@export
class Malware(Entity):
    """
    Malware Entity class.

    Attributes
    ----------
    Name : str
        Malware Name
    Category : str
        Malware Category
    File : File
        Malware File
    Files : List[File]
        Malware Files
    Processes : List[Process]
        Malware Processes

    """

    ID_PROPERTIES = ["Name"]

    def __init__(self, src_entity: Mapping[str, Any] = None, **kwargs):
        """
        Create a new instance of the entity type.

        Parameters
        ----------
        src_entity : Mapping[str, Any], optional
            Create entity from existing entity or
            other mapping object that implements entity properties.
            (the default is None)

        Other Parameters
        ----------------
        kwargs : Dict[str, Any]
            Supply the entity properties as a set of
            kw arguments.

        """
        self.Name: str = ""
        self.Category: str = ""
        self.File: Optional[File] = None
        self.Files: List[File] = []
        self.Processes: List[Process] = []
        super().__init__(src_entity=src_entity, **kwargs)

    @property
    def description_str(self) -> str:
        """Return Entity Description."""
        return f"{self.Name}: {self.Category}"

    @property
    def name_str(self) -> str:
        """Return Entity Name."""
        return self.Name or self.__class__.__name__

    _entity_schema = {
        # Name (type System.String)
        "Name": None,
        # Category (type System.String)
        "Category": None,
        # File (type Microsoft.Azure.Security.Detection.AlertContracts.V3.Entities.File)
        "File": "File",
        "Files": (list, "File"),
        "Processes": (list, "Process"),
        "TimeGenerated": None,
        "StartTime": None,
        "EndTime": None,
    }
