import json
import os
import subprocess


def prepare_env(platform: str, config: json, select_config: str = None):
    if platform != "gha" and platform != "azp":
        raise ValueError("Only GitHub Actions and Azure Pipelines is supported at this point.")

    if platform != "azp" and select_config is not None:
        raise ValueError("The --select-config parameter can only be used with Azure Pipelines.")

    if select_config:
        config = config[select_config]

    def _set_env_variable(var_name: str, value: str):
        print("{} = {}".format(var_name, value))
        os.environ[var_name] = value
        if platform == "gha":
            subprocess.run(
                'echo "{}={}" >> $GITHUB_ENV'.format(var_name, value),
                shell=True
            )
        if platform == "azp":
            if compiler == "VISUAL":
                subprocess.run(
                    'echo ##vso[task.setvariable variable={}]{}'.format(var_name, value),
                    shell=True
                )
            else:
                subprocess.run(
                    'echo "##vso[task.setvariable variable={}]{}"'.format(var_name, value),
                    shell=True
                )

    compiler = config["compiler"]
    compiler_version = config["version"]
    docker_image = config.get("dockerImage", "")
    build_type = config.get("buildType", "")

    _set_env_variable("BPT_CWD", config["cwd"])
    _set_env_variable("CONAN_VERSION", config["recipe_version"])

    if compiler == "APPLE_CLANG":
        if "." not in compiler_version:
            compiler_version = "{}.0".format(compiler_version)

    _set_env_variable("CONAN_{}_VERSIONS".format(compiler), compiler_version)

    if compiler == "GCC" or compiler == "CLANG":
        if docker_image == "":
            compiler_lower = compiler.lower()
            version_without_dot = compiler_version.replace(".", "")
            docker_image = "conanio/{}{}".format(compiler_lower, version_without_dot)
        _set_env_variable("CONAN_DOCKER_IMAGE", docker_image)

    if build_type != "":
        _set_env_variable("CONAN_BUILD_TYPES", build_type)

    if platform == "gha" or platform == "azp":
        if compiler == "APPLE_CLANG":
            xcode_mapping = {
                "9.1": "/Applications/Xcode_9.4.1.app",
                "10.0": "/Applications/Xcode_10.3.app",
                "11.0": "/Applications/Xcode_11.3.1.app",
                "12.0": "/Applications/Xcode_12.2.app",
            }
            if compiler_version in xcode_mapping:
                subprocess.run(
                    'sudo xcode-select -switch "{}"'.format(xcode_mapping[compiler_version]),
                    shell=True
                )
                print('executing: xcode-select -switch "{}"'.format(xcode_mapping[compiler_version]))

            subprocess.run(
                'clang++ --version',
                shell=True
            )

    if platform == "azp" and compiler == "VISUAL":
        with open(os.path.join(os.path.dirname(__file__), "prepare_env_azp_windows.ps1"), "r") as file:
            content = file.read()
            file.close()

        with open("execute.ps1", "w", encoding="utf-8") as file:
            file.write(content)
            file.close()

        subprocess.run("powershell -file {}".format(os.path.join(os.getcwd(), "execute.ps1")), shell=True, check=True)

    subprocess.run("conan user", shell=True)
