# AUTOGENERATED! DO NOT EDIT! File to edit: ../nbs/04_vanity.ipynb.

# %% auto 0
__all__ = ['hex_chars', 'npub_chars', 'vanity_notifyr', 'guess_bytes', 'guess_bech32', 'guess_hex', 'time_guess',
           'get_guess_rate', 'expected_guesses_by_char', 'expected_chars_by_time', 'expected_time',
           'average_char_by_time', 'average_time_by_char', 'expected_performance', 'gen_vanity_pubkey']

# %% ../nbs/04_vanity.ipynb 4
from .nostr import PrivateKey
from .notifyr import notifyr

# %% ../nbs/04_vanity.ipynb 7
import time
import secrets
import secp256k1
from typing import Union
from .nostr import PrivateKey
from .notifyr import notifyr
from nostr import bech32

# %% ../nbs/04_vanity.ipynb 9
def guess_bytes():
    privkey_bytes = secrets.token_bytes(32) 
    sk = secp256k1.PrivateKey(privkey_bytes)
    pubkey_bytes = sk.pubkey.serialize()[1:]
    return privkey_bytes, pubkey_bytes

def guess_bech32(startswith=''):
    privkey_bytes, pubkey_bytes = guess_bytes()
    converted_bits = bech32.convertbits(pubkey_bytes, 8, 5)
    npub = bech32.bech32_encode("npub", converted_bits, bech32.Encoding.BECH32)
    if npub.startswith(startswith):
        return privkey_bytes.hex(), npub
    else:
        return None, None

def guess_hex(startswith=''):
    privkey_bytes, pubkey_bytes = guess_bytes()
    privkey_hex = privkey_bytes.hex()
    pubkey_hex = pubkey_bytes.hex()
    if pubkey_hex.startswith(startswith):
        return privkey_bytes.hex(), pubkey_bytes.hex()
    else:
        return None, None

# %% ../nbs/04_vanity.ipynb 13
def time_guess(guesser):
    start = time.perf_counter()
    pub = guesser(startswith=' ')
    if pub is None: # this is just replicating what we will actually be doing
        pass
    end = time.perf_counter()
    interval = end - start
    return interval

def get_guess_rate(guesser, n_guesses=1e4):
    n_guesses = int(n_guesses)
    t = sum([time_guess(guesser) for _ in range(n_guesses)]) / n_guesses
    guesses_per_second = 1 / t
    return guesses_per_second

# %% ../nbs/04_vanity.ipynb 17
import math

# %% ../nbs/04_vanity.ipynb 18
def expected_guesses_by_char(options: Union[str,list], num_char: int) -> float:
    """return an average number of guesses it would take to guess
    a pattern based on the number of characters in the pattern and
    the number of character options in the random output

    Parameters
    ----------
    options : list or str
        a set of characters as a str or list that are options for
        guessing
    num_char : int
        the number of characters in the pattern

    Returns
    -------
    float
        the expected number of guesses required to match the pattern
    """
    p = 1 / len(options)
    return (p ** -num_char - 1)/ (1 - p)

def expected_chars_by_time(options: Union[str,list], num_guesses: int) -> float:
    """the length of pattern you might expect to be able to guess given a
    certain number of guesses.

    Parameters
    ----------
    options : list or str
        a set of characters as a str or list that are options for
        guessing
    num_guesses : int
        the total number of guesses at a pattern

    Returns
    -------
    float
        th
    """
    p = 1 / len(options)
    n = - math.log(1 + (num_guesses * (1 - p))) / math.log(p)
    return n

def expected_time(options: Union[str,list], num_char: int, time_per_guess: float) -> float:
    """the expected amount of time it would take to guess a pattern with a certain
    length based on the average time per guess and the character options

    Parameters
    ----------
    options : list or str
        a set of characters as a str or list that are options for
        guessing
    num_char : int
        the number of characters in the pattern
    time_per_guess : float
        averge time per guess in seconds

    Returns
    -------
    float
        the expected amount of time needed to guess the pattern
    """
    n_guess = expected_guesses_by_char(options, num_char)
    time_seconds = n_guess * time_per_guess
    return time_seconds


# %% ../nbs/04_vanity.ipynb 19
hex_chars = 'abcdef0123456789'
npub_chars = '023456789acdefghjklmnpqrstuvwxyz'

def average_char_by_time(options: Union[str,list], time_per_guess: float) -> None:
    """print an average number of characters you would expect to be
    able to guess for certain time periods based on character options
    and the average time per guess

    Parameters
    ----------
    options : list or str
        a set of characters as a str or list that are options for
        guessing
    time_per_guess : float
        the average time elapsed per guess
    """
    seconds_in_month = 60 * 60 * 24 * 30.5
    seconds_in_day = 60 * 60 * 24
    seconds_in_hour = 60 * 60
    seconds_in_minute = 60

    guesses_per_month = seconds_in_month / time_per_guess
    guesses_per_day = seconds_in_day / time_per_guess
    guesses_per_hour = seconds_in_hour / time_per_guess
    guesses_per_minute = seconds_in_minute / time_per_guess
    guesses_per_second = 1 / time_per_guess

    guesses = [guesses_per_second, guesses_per_minute,\
               guesses_per_hour, guesses_per_day, guesses_per_month]

    expected_chars = [expected_chars_by_time(options, g) for g in guesses]
    results = zip(['one second', 'one minute', 'one hour', 'one day', 'one month'],
                   expected_chars)
    for t, c in results:
        print(f'In {t} you can expect to get {c} characters on average')

def average_time_by_char(options: Union[str,list], time_per_guess: float) -> None:
    """print an average elapsed time for a range of pattern lengths

    Parameters
    ----------
    options : Union[str,list]
        a set of characters as a str or list that are options for
        guessing
    time_per_guess : float
        the average time elapsed per guess
    """
    for n in range(20):
        n += 1
        t = expected_time(options, n, time_per_guess)
        print(f'{n} characters: it might take {t} seconds')



# %% ../nbs/04_vanity.ipynb 23
def expected_performance():
    print(
        '''This is a random guessing process - estimations are an average, but the actual
        time it takes to find a key could be significantly more or less than the estimate!
        Please keep that in mind when choosing an option.
        ''')
    print('hex:')
    time_per_guess_hex = 1 / get_guess_rate(guesser=guess_hex)
    average_char_by_time(hex_chars, time_per_guess_hex)
    print('\n')
    average_time_by_char(hex_chars, time_per_guess_hex)
    print('\n')

    print('npub:')
    time_per_guess_bech32 = 1 / get_guess_rate(guesser=guess_bech32)
    average_char_by_time(npub_chars, time_per_guess_bech32)
    print('\n')
    average_time_by_char(npub_chars, time_per_guess_bech32)
    print('\n')

    

# %% ../nbs/04_vanity.ipynb 28
def gen_vanity_pubkey(startswith: str, style='hex') -> PrivateKey:
    """randomly generate private keys until one matches the desire
    startswith for an npub or hex

    Parameters
    ----------
    startswith : str
        characters that the public key should start with. More chars
        means longer run time
    style : str, optional
        'npub' or 'hex' - npub is more commonly displayed on apps
        while hex is the true base private key with no encoding,
        by default 'hex'

    Returns
    -------
    PrivateKey
        returns a private key object
    """
    pubkey = ''
    if style == 'npub':
        npub_chars = '023456789acdefghjklmnpqrstuvwxyz'
        if not all(c in npub_chars for c in startswith):
            raise ValueError(f'character of selection not '
                              'in npub pattern ({npub_chars})')
        startswith = f'npub1{startswith}'
    else:
        hex_chars = 'abcdef0123456789'
        if not all(c in hex_chars for c in startswith):
            raise ValueError(f'character of selection not in '
                              'hex pattern ({hex_chars})')
    while not pubkey.startswith(startswith):
        privkey = PrivateKey()
        if style == 'npub':
            pubkey = privkey.public_key.bech32()
        else:
            pubkey = privkey.public_key.hex()
    return privkey

# %% ../nbs/04_vanity.ipynb 30
def gen_vanity_pubkey(startswith: str, style='hex') -> PrivateKey:
    """randomly generate private keys until one matches the desire
    startswith for an npub or hex

    Parameters
    ----------
    startswith : str
        characters that the public key should start with. More chars
        means longer run time
    style : str, optional
        'npub' or 'hex' - npub is more commonly displayed on apps
        while hex is the true base private key with no encoding,
        by default 'hex'

    Returns
    -------
    PrivateKey
        returns a private key object
    """
    pubkey = None
    if style == 'npub':
        if not all(c in npub_chars for c in startswith):
            raise ValueError(f'character of selection not '
                              'in npub pattern ({npub_chars})')
        time_per_guess = 1 / get_guess_rate(guess_bech32)
        t = expected_time(npub_chars, len(startswith), time_per_guess)
        startswith = f'npub1{startswith}'
    else:
        if not all(c in hex_chars for c in startswith):
            raise ValueError(f'character of selection not in '
                              'hex pattern ({hex_chars})')
        time_per_guess = 1 / get_guess_rate(guess_hex)
        t = expected_time(hex_chars, len(startswith), time_per_guess)
    print(f'It might take {t} seconds to find a {style} pubkey that starts with '
          f'{startswith}. Note that this is a very rough estimate and due '
          'to the random nature of finding vanity keys it could take MUCH '
          'longer.')
    while pubkey is None:
        if style == 'npub':
            privkey_hex, pubkey = guess_bech32(startswith=startswith)
        else:
            privkey_hex, pubkey = guess_hex(startswith=startswith)
    return PrivateKey.from_hex(privkey_hex)

# %% ../nbs/04_vanity.ipynb 40
vanity_notifyr = notifyr(gen_vanity_pubkey)
