"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3SourceAction = exports.S3Trigger = void 0;
const codepipeline = require("@aws-cdk/aws-codepipeline");
const targets = require("@aws-cdk/aws-events-targets");
const core_1 = require("@aws-cdk/core");
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the S3 Action detect changes.
 * This is the type of the {@link S3SourceAction.trigger} property.
 */
var S3Trigger;
(function (S3Trigger) {
    /**
     * The Action will never detect changes -
     * the Pipeline it's part of will only begin a run when explicitly started.
     */
    S3Trigger["NONE"] = "None";
    /**
     * CodePipeline will poll S3 to detect changes.
     * This is the default method of detecting changes.
     */
    S3Trigger["POLL"] = "Poll";
    /**
     * CodePipeline will use CloudWatch Events to be notified of changes.
     * Note that the Bucket that the Action uses needs to be part of a CloudTrail Trail
     * for the events to be delivered.
     */
    S3Trigger["EVENTS"] = "Events";
})(S3Trigger = exports.S3Trigger || (exports.S3Trigger = {}));
/**
 * Source that is provided by a specific Amazon S3 object.
 *
 * Will trigger the pipeline as soon as the S3 object changes, but only if there is
 * a CloudTrail Trail in the account that captures the S3 event.
 */
class S3SourceAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.bucket,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'S3',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        if (props.bucketKey.length === 0) {
            throw new Error('Property bucketKey cannot be an empty string');
        }
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            versionId: this.variableExpression('VersionId'),
            eTag: this.variableExpression('ETag'),
        };
    }
    bound(_scope, stage, options) {
        if (this.props.trigger === S3Trigger.EVENTS) {
            const id = this.generateEventId(stage);
            this.props.bucket.onCloudTrailWriteObject(id, {
                target: new targets.CodePipeline(stage.pipeline),
                paths: [this.props.bucketKey],
            });
        }
        // we need to read from the source bucket...
        this.props.bucket.grantRead(options.role);
        // ...and write to the Pipeline bucket
        options.bucket.grantWrite(options.role);
        return {
            configuration: {
                S3Bucket: this.props.bucket.bucketName,
                S3ObjectKey: this.props.bucketKey,
                PollForSourceChanges: this.props.trigger && this.props.trigger === S3Trigger.POLL,
            },
        };
    }
    generateEventId(stage) {
        let ret;
        const baseId = stage.pipeline.node.uniqueId + 'SourceEventRule';
        if (core_1.Token.isUnresolved(this.props.bucketKey)) {
            // If bucketKey is a Token, don't include it in the ID.
            // Instead, use numbers to differentiate if multiple actions observe the same bucket
            let candidate = baseId;
            let counter = 0;
            while (this.props.bucket.node.tryFindChild(candidate) !== undefined) {
                counter += 1;
                candidate = baseId + counter;
            }
            ret = candidate;
        }
        else {
            // we can't use Tokens in construct IDs,
            // however, if bucketKey is not a Token,
            // we want it to differentiate between multiple actions
            // observing the same Bucket with different keys
            ret = baseId + this.props.bucketKey;
            if (this.props.bucket.node.tryFindChild(ret)) {
                // this means a duplicate path for the same bucket - error out
                throw new Error(`S3 source action with path '${this.props.bucketKey}' is already present in the pipeline for this source bucket`);
            }
        }
        return ret;
    }
}
exports.S3SourceAction = S3SourceAction;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoic291cmNlLWFjdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbInNvdXJjZS1hY3Rpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsMERBQTBEO0FBQzFELHVEQUF1RDtBQUV2RCx3Q0FBaUQ7QUFDakQsc0NBQW1DO0FBQ25DLHNDQUFpRDtBQUVqRDs7O0dBR0c7QUFDSCxJQUFZLFNBbUJYO0FBbkJELFdBQVksU0FBUztJQUNuQjs7O09BR0c7SUFDSCwwQkFBYSxDQUFBO0lBRWI7OztPQUdHO0lBQ0gsMEJBQWEsQ0FBQTtJQUViOzs7O09BSUc7SUFDSCw4QkFBaUIsQ0FBQTtBQUNuQixDQUFDLEVBbkJXLFNBQVMsR0FBVCxpQkFBUyxLQUFULGlCQUFTLFFBbUJwQjtBQTZDRDs7Ozs7R0FLRztBQUNILE1BQWEsY0FBZSxTQUFRLGVBQU07SUFHeEMsWUFBWSxLQUEwQjtRQUNwQyxLQUFLLENBQUM7WUFDSixHQUFHLEtBQUs7WUFDUixRQUFRLEVBQUUsS0FBSyxDQUFDLE1BQU07WUFDdEIsUUFBUSxFQUFFLFlBQVksQ0FBQyxjQUFjLENBQUMsTUFBTTtZQUM1QyxRQUFRLEVBQUUsSUFBSTtZQUNkLGNBQWMsRUFBRSw2QkFBb0IsRUFBRTtZQUN0QyxPQUFPLEVBQUUsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDO1NBQ3hCLENBQUMsQ0FBQztRQUVILElBQUksS0FBSyxDQUFDLFNBQVMsQ0FBQyxNQUFNLEtBQUssQ0FBQyxFQUFFO1lBQ2hDLE1BQU0sSUFBSSxLQUFLLENBQUMsOENBQThDLENBQUMsQ0FBQztTQUNqRTtRQUVELElBQUksQ0FBQyxLQUFLLEdBQUcsS0FBSyxDQUFDO0lBQ3JCLENBQUM7SUFFRCw0Q0FBNEM7SUFDNUMsSUFBVyxTQUFTO1FBQ2xCLE9BQVE7WUFDTixTQUFTLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLFdBQVcsQ0FBQztZQUMvQyxJQUFJLEVBQUUsSUFBSSxDQUFDLGtCQUFrQixDQUFDLE1BQU0sQ0FBQztTQUN0QyxDQUFDO0lBQ0osQ0FBQztJQUVTLEtBQUssQ0FBQyxNQUFpQixFQUFFLEtBQTBCLEVBQUUsT0FBdUM7UUFFcEcsSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sS0FBSyxTQUFTLENBQUMsTUFBTSxFQUFFO1lBQzNDLE1BQU0sRUFBRSxHQUFHLElBQUksQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLENBQUM7WUFDdkMsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsdUJBQXVCLENBQUMsRUFBRSxFQUFFO2dCQUM1QyxNQUFNLEVBQUUsSUFBSSxPQUFPLENBQUMsWUFBWSxDQUFDLEtBQUssQ0FBQyxRQUFRLENBQUM7Z0JBQ2hELEtBQUssRUFBRSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDO2FBQzlCLENBQUMsQ0FBQztTQUNKO1FBRUQsNENBQTRDO1FBQzVDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLFNBQVMsQ0FBQyxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUM7UUFFMUMsc0NBQXNDO1FBQ3RDLE9BQU8sQ0FBQyxNQUFNLENBQUMsVUFBVSxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztRQUV4QyxPQUFPO1lBQ0wsYUFBYSxFQUFFO2dCQUNiLFFBQVEsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxVQUFVO2dCQUN0QyxXQUFXLEVBQUUsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTO2dCQUNqQyxvQkFBb0IsRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sSUFBSSxJQUFJLENBQUMsS0FBSyxDQUFDLE9BQU8sS0FBSyxTQUFTLENBQUMsSUFBSTthQUNsRjtTQUNGLENBQUM7SUFDSixDQUFDO0lBRU8sZUFBZSxDQUFDLEtBQTBCO1FBQ2hELElBQUksR0FBVyxDQUFDO1FBQ2hCLE1BQU0sTUFBTSxHQUFHLEtBQUssQ0FBQyxRQUFRLENBQUMsSUFBSSxDQUFDLFFBQVEsR0FBRyxpQkFBaUIsQ0FBQztRQUVoRSxJQUFJLFlBQUssQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxTQUFTLENBQUMsRUFBRTtZQUM1Qyx1REFBdUQ7WUFDdkQsb0ZBQW9GO1lBQ3BGLElBQUksU0FBUyxHQUFHLE1BQU0sQ0FBQztZQUN2QixJQUFJLE9BQU8sR0FBRyxDQUFDLENBQUM7WUFDaEIsT0FBTyxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxJQUFJLENBQUMsWUFBWSxDQUFDLFNBQVMsQ0FBQyxLQUFLLFNBQVMsRUFBRTtnQkFDbkUsT0FBTyxJQUFJLENBQUMsQ0FBQztnQkFDYixTQUFTLEdBQUcsTUFBTSxHQUFHLE9BQU8sQ0FBQzthQUM5QjtZQUNELEdBQUcsR0FBRyxTQUFTLENBQUM7U0FDakI7YUFBTTtZQUNMLHdDQUF3QztZQUN4Qyx3Q0FBd0M7WUFDeEMsdURBQXVEO1lBQ3ZELGdEQUFnRDtZQUNoRCxHQUFHLEdBQUcsTUFBTSxHQUFHLElBQUksQ0FBQyxLQUFLLENBQUMsU0FBUyxDQUFDO1lBQ3BDLElBQUksSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsSUFBSSxDQUFDLFlBQVksQ0FBQyxHQUFHLENBQUMsRUFBRTtnQkFDNUMsOERBQThEO2dCQUM5RCxNQUFNLElBQUksS0FBSyxDQUFDLCtCQUErQixJQUFJLENBQUMsS0FBSyxDQUFDLFNBQVMsNkRBQTZELENBQUMsQ0FBQzthQUNuSTtTQUNGO1FBRUQsT0FBTyxHQUFHLENBQUM7SUFDYixDQUFDO0NBQ0Y7QUFqRkQsd0NBaUZDIiwic291cmNlc0NvbnRlbnQiOlsiaW1wb3J0ICogYXMgY29kZXBpcGVsaW5lIGZyb20gJ0Bhd3MtY2RrL2F3cy1jb2RlcGlwZWxpbmUnO1xuaW1wb3J0ICogYXMgdGFyZ2V0cyBmcm9tICdAYXdzLWNkay9hd3MtZXZlbnRzLXRhcmdldHMnO1xuaW1wb3J0ICogYXMgczMgZnJvbSAnQGF3cy1jZGsvYXdzLXMzJztcbmltcG9ydCB7IENvbnN0cnVjdCwgVG9rZW4gfSBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IEFjdGlvbiB9IGZyb20gJy4uL2FjdGlvbic7XG5pbXBvcnQgeyBzb3VyY2VBcnRpZmFjdEJvdW5kcyB9IGZyb20gJy4uL2NvbW1vbic7XG5cbi8qKlxuICogSG93IHNob3VsZCB0aGUgUzMgQWN0aW9uIGRldGVjdCBjaGFuZ2VzLlxuICogVGhpcyBpcyB0aGUgdHlwZSBvZiB0aGUge0BsaW5rIFMzU291cmNlQWN0aW9uLnRyaWdnZXJ9IHByb3BlcnR5LlxuICovXG5leHBvcnQgZW51bSBTM1RyaWdnZXIge1xuICAvKipcbiAgICogVGhlIEFjdGlvbiB3aWxsIG5ldmVyIGRldGVjdCBjaGFuZ2VzIC1cbiAgICogdGhlIFBpcGVsaW5lIGl0J3MgcGFydCBvZiB3aWxsIG9ubHkgYmVnaW4gYSBydW4gd2hlbiBleHBsaWNpdGx5IHN0YXJ0ZWQuXG4gICAqL1xuICBOT05FID0gJ05vbmUnLFxuXG4gIC8qKlxuICAgKiBDb2RlUGlwZWxpbmUgd2lsbCBwb2xsIFMzIHRvIGRldGVjdCBjaGFuZ2VzLlxuICAgKiBUaGlzIGlzIHRoZSBkZWZhdWx0IG1ldGhvZCBvZiBkZXRlY3RpbmcgY2hhbmdlcy5cbiAgICovXG4gIFBPTEwgPSAnUG9sbCcsXG5cbiAgLyoqXG4gICAqIENvZGVQaXBlbGluZSB3aWxsIHVzZSBDbG91ZFdhdGNoIEV2ZW50cyB0byBiZSBub3RpZmllZCBvZiBjaGFuZ2VzLlxuICAgKiBOb3RlIHRoYXQgdGhlIEJ1Y2tldCB0aGF0IHRoZSBBY3Rpb24gdXNlcyBuZWVkcyB0byBiZSBwYXJ0IG9mIGEgQ2xvdWRUcmFpbCBUcmFpbFxuICAgKiBmb3IgdGhlIGV2ZW50cyB0byBiZSBkZWxpdmVyZWQuXG4gICAqL1xuICBFVkVOVFMgPSAnRXZlbnRzJyxcbn1cblxuLyoqXG4gKiBUaGUgQ29kZVBpcGVsaW5lIHZhcmlhYmxlcyBlbWl0dGVkIGJ5IHRoZSBTMyBzb3VyY2UgQWN0aW9uLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFMzU291cmNlVmFyaWFibGVzIHtcbiAgLyoqIFRoZSBpZGVudGlmaWVyIG9mIHRoZSBTMyB2ZXJzaW9uIG9mIHRoZSBvYmplY3QgdGhhdCB0cmlnZ2VyZWQgdGhlIGJ1aWxkLiAqL1xuICByZWFkb25seSB2ZXJzaW9uSWQ6IHN0cmluZztcblxuICAvKiogVGhlIGUtdGFnIG9mIHRoZSBTMyB2ZXJzaW9uIG9mIHRoZSBvYmplY3QgdGhhdCB0cmlnZ2VyZWQgdGhlIGJ1aWxkLiAqL1xuICByZWFkb25seSBlVGFnOiBzdHJpbmc7XG59XG5cbi8qKlxuICogQ29uc3RydWN0aW9uIHByb3BlcnRpZXMgb2YgdGhlIHtAbGluayBTM1NvdXJjZUFjdGlvbiBTMyBzb3VyY2UgQWN0aW9ufS5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBTM1NvdXJjZUFjdGlvblByb3BzIGV4dGVuZHMgY29kZXBpcGVsaW5lLkNvbW1vbkF3c0FjdGlvblByb3BzIHtcbiAgLyoqXG4gICAqXG4gICAqL1xuICByZWFkb25seSBvdXRwdXQ6IGNvZGVwaXBlbGluZS5BcnRpZmFjdDtcblxuICAvKipcbiAgICogVGhlIGtleSB3aXRoaW4gdGhlIFMzIGJ1Y2tldCB0aGF0IHN0b3JlcyB0aGUgc291cmNlIGNvZGUuXG4gICAqXG4gICAqIEBleGFtcGxlICdwYXRoL3RvL2ZpbGUuemlwJ1xuICAgKi9cbiAgcmVhZG9ubHkgYnVja2V0S2V5OiBzdHJpbmc7XG5cbiAgLyoqXG4gICAqIEhvdyBzaG91bGQgQ29kZVBpcGVsaW5lIGRldGVjdCBzb3VyY2UgY2hhbmdlcyBmb3IgdGhpcyBBY3Rpb24uXG4gICAqIE5vdGUgdGhhdCBpZiB0aGlzIGlzIFMzVHJpZ2dlci5FVkVOVFMsIHlvdSBuZWVkIHRvIG1ha2Ugc3VyZSB0byBpbmNsdWRlIHRoZSBzb3VyY2UgQnVja2V0IGluIGEgQ2xvdWRUcmFpbCBUcmFpbCxcbiAgICogYXMgb3RoZXJ3aXNlIHRoZSBDbG91ZFdhdGNoIEV2ZW50cyB3aWxsIG5vdCBiZSBlbWl0dGVkLlxuICAgKlxuICAgKiBAZGVmYXVsdCBTM1RyaWdnZXIuUE9MTFxuICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9BbWF6b25DbG91ZFdhdGNoL2xhdGVzdC9ldmVudHMvbG9nLXMzLWRhdGEtZXZlbnRzLmh0bWxcbiAgICovXG4gIHJlYWRvbmx5IHRyaWdnZXI/OiBTM1RyaWdnZXI7XG5cbiAgLyoqXG4gICAqIFRoZSBBbWF6b24gUzMgYnVja2V0IHRoYXQgc3RvcmVzIHRoZSBzb3VyY2UgY29kZVxuICAgKi9cbiAgcmVhZG9ubHkgYnVja2V0OiBzMy5JQnVja2V0O1xufVxuXG4vKipcbiAqIFNvdXJjZSB0aGF0IGlzIHByb3ZpZGVkIGJ5IGEgc3BlY2lmaWMgQW1hem9uIFMzIG9iamVjdC5cbiAqXG4gKiBXaWxsIHRyaWdnZXIgdGhlIHBpcGVsaW5lIGFzIHNvb24gYXMgdGhlIFMzIG9iamVjdCBjaGFuZ2VzLCBidXQgb25seSBpZiB0aGVyZSBpc1xuICogYSBDbG91ZFRyYWlsIFRyYWlsIGluIHRoZSBhY2NvdW50IHRoYXQgY2FwdHVyZXMgdGhlIFMzIGV2ZW50LlxuICovXG5leHBvcnQgY2xhc3MgUzNTb3VyY2VBY3Rpb24gZXh0ZW5kcyBBY3Rpb24ge1xuICBwcml2YXRlIHJlYWRvbmx5IHByb3BzOiBTM1NvdXJjZUFjdGlvblByb3BzO1xuXG4gIGNvbnN0cnVjdG9yKHByb3BzOiBTM1NvdXJjZUFjdGlvblByb3BzKSB7XG4gICAgc3VwZXIoe1xuICAgICAgLi4ucHJvcHMsXG4gICAgICByZXNvdXJjZTogcHJvcHMuYnVja2V0LFxuICAgICAgY2F0ZWdvcnk6IGNvZGVwaXBlbGluZS5BY3Rpb25DYXRlZ29yeS5TT1VSQ0UsXG4gICAgICBwcm92aWRlcjogJ1MzJyxcbiAgICAgIGFydGlmYWN0Qm91bmRzOiBzb3VyY2VBcnRpZmFjdEJvdW5kcygpLFxuICAgICAgb3V0cHV0czogW3Byb3BzLm91dHB1dF0sXG4gICAgfSk7XG5cbiAgICBpZiAocHJvcHMuYnVja2V0S2V5Lmxlbmd0aCA9PT0gMCkge1xuICAgICAgdGhyb3cgbmV3IEVycm9yKCdQcm9wZXJ0eSBidWNrZXRLZXkgY2Fubm90IGJlIGFuIGVtcHR5IHN0cmluZycpO1xuICAgIH1cblxuICAgIHRoaXMucHJvcHMgPSBwcm9wcztcbiAgfVxuXG4gIC8qKiBUaGUgdmFyaWFibGVzIGVtaXR0ZWQgYnkgdGhpcyBhY3Rpb24uICovXG4gIHB1YmxpYyBnZXQgdmFyaWFibGVzKCk6IFMzU291cmNlVmFyaWFibGVzIHtcbiAgICByZXR1cm4gIHtcbiAgICAgIHZlcnNpb25JZDogdGhpcy52YXJpYWJsZUV4cHJlc3Npb24oJ1ZlcnNpb25JZCcpLFxuICAgICAgZVRhZzogdGhpcy52YXJpYWJsZUV4cHJlc3Npb24oJ0VUYWcnKSxcbiAgICB9O1xuICB9XG5cbiAgcHJvdGVjdGVkIGJvdW5kKF9zY29wZTogQ29uc3RydWN0LCBzdGFnZTogY29kZXBpcGVsaW5lLklTdGFnZSwgb3B0aW9uczogY29kZXBpcGVsaW5lLkFjdGlvbkJpbmRPcHRpb25zKTpcbiAgY29kZXBpcGVsaW5lLkFjdGlvbkNvbmZpZyB7XG4gICAgaWYgKHRoaXMucHJvcHMudHJpZ2dlciA9PT0gUzNUcmlnZ2VyLkVWRU5UUykge1xuICAgICAgY29uc3QgaWQgPSB0aGlzLmdlbmVyYXRlRXZlbnRJZChzdGFnZSk7XG4gICAgICB0aGlzLnByb3BzLmJ1Y2tldC5vbkNsb3VkVHJhaWxXcml0ZU9iamVjdChpZCwge1xuICAgICAgICB0YXJnZXQ6IG5ldyB0YXJnZXRzLkNvZGVQaXBlbGluZShzdGFnZS5waXBlbGluZSksXG4gICAgICAgIHBhdGhzOiBbdGhpcy5wcm9wcy5idWNrZXRLZXldLFxuICAgICAgfSk7XG4gICAgfVxuXG4gICAgLy8gd2UgbmVlZCB0byByZWFkIGZyb20gdGhlIHNvdXJjZSBidWNrZXQuLi5cbiAgICB0aGlzLnByb3BzLmJ1Y2tldC5ncmFudFJlYWQob3B0aW9ucy5yb2xlKTtcblxuICAgIC8vIC4uLmFuZCB3cml0ZSB0byB0aGUgUGlwZWxpbmUgYnVja2V0XG4gICAgb3B0aW9ucy5idWNrZXQuZ3JhbnRXcml0ZShvcHRpb25zLnJvbGUpO1xuXG4gICAgcmV0dXJuIHtcbiAgICAgIGNvbmZpZ3VyYXRpb246IHtcbiAgICAgICAgUzNCdWNrZXQ6IHRoaXMucHJvcHMuYnVja2V0LmJ1Y2tldE5hbWUsXG4gICAgICAgIFMzT2JqZWN0S2V5OiB0aGlzLnByb3BzLmJ1Y2tldEtleSxcbiAgICAgICAgUG9sbEZvclNvdXJjZUNoYW5nZXM6IHRoaXMucHJvcHMudHJpZ2dlciAmJiB0aGlzLnByb3BzLnRyaWdnZXIgPT09IFMzVHJpZ2dlci5QT0xMLFxuICAgICAgfSxcbiAgICB9O1xuICB9XG5cbiAgcHJpdmF0ZSBnZW5lcmF0ZUV2ZW50SWQoc3RhZ2U6IGNvZGVwaXBlbGluZS5JU3RhZ2UpOiBzdHJpbmcge1xuICAgIGxldCByZXQ6IHN0cmluZztcbiAgICBjb25zdCBiYXNlSWQgPSBzdGFnZS5waXBlbGluZS5ub2RlLnVuaXF1ZUlkICsgJ1NvdXJjZUV2ZW50UnVsZSc7XG5cbiAgICBpZiAoVG9rZW4uaXNVbnJlc29sdmVkKHRoaXMucHJvcHMuYnVja2V0S2V5KSkge1xuICAgICAgLy8gSWYgYnVja2V0S2V5IGlzIGEgVG9rZW4sIGRvbid0IGluY2x1ZGUgaXQgaW4gdGhlIElELlxuICAgICAgLy8gSW5zdGVhZCwgdXNlIG51bWJlcnMgdG8gZGlmZmVyZW50aWF0ZSBpZiBtdWx0aXBsZSBhY3Rpb25zIG9ic2VydmUgdGhlIHNhbWUgYnVja2V0XG4gICAgICBsZXQgY2FuZGlkYXRlID0gYmFzZUlkO1xuICAgICAgbGV0IGNvdW50ZXIgPSAwO1xuICAgICAgd2hpbGUgKHRoaXMucHJvcHMuYnVja2V0Lm5vZGUudHJ5RmluZENoaWxkKGNhbmRpZGF0ZSkgIT09IHVuZGVmaW5lZCkge1xuICAgICAgICBjb3VudGVyICs9IDE7XG4gICAgICAgIGNhbmRpZGF0ZSA9IGJhc2VJZCArIGNvdW50ZXI7XG4gICAgICB9XG4gICAgICByZXQgPSBjYW5kaWRhdGU7XG4gICAgfSBlbHNlIHtcbiAgICAgIC8vIHdlIGNhbid0IHVzZSBUb2tlbnMgaW4gY29uc3RydWN0IElEcyxcbiAgICAgIC8vIGhvd2V2ZXIsIGlmIGJ1Y2tldEtleSBpcyBub3QgYSBUb2tlbixcbiAgICAgIC8vIHdlIHdhbnQgaXQgdG8gZGlmZmVyZW50aWF0ZSBiZXR3ZWVuIG11bHRpcGxlIGFjdGlvbnNcbiAgICAgIC8vIG9ic2VydmluZyB0aGUgc2FtZSBCdWNrZXQgd2l0aCBkaWZmZXJlbnQga2V5c1xuICAgICAgcmV0ID0gYmFzZUlkICsgdGhpcy5wcm9wcy5idWNrZXRLZXk7XG4gICAgICBpZiAodGhpcy5wcm9wcy5idWNrZXQubm9kZS50cnlGaW5kQ2hpbGQocmV0KSkge1xuICAgICAgICAvLyB0aGlzIG1lYW5zIGEgZHVwbGljYXRlIHBhdGggZm9yIHRoZSBzYW1lIGJ1Y2tldCAtIGVycm9yIG91dFxuICAgICAgICB0aHJvdyBuZXcgRXJyb3IoYFMzIHNvdXJjZSBhY3Rpb24gd2l0aCBwYXRoICcke3RoaXMucHJvcHMuYnVja2V0S2V5fScgaXMgYWxyZWFkeSBwcmVzZW50IGluIHRoZSBwaXBlbGluZSBmb3IgdGhpcyBzb3VyY2UgYnVja2V0YCk7XG4gICAgICB9XG4gICAgfVxuXG4gICAgcmV0dXJuIHJldDtcbiAgfVxufVxuIl19