"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProwlerAudit = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// import { Stack, Duration, RemovalPolicy, CustomResource } from 'aws-cdk-lib';
// eslint-disable-next-line no-duplicate-imports
// import { aws_iam as iam, aws_logs as logs, aws_s3 as s3, aws_codebuild as codebuild, aws_lambda as lambda, custom_resources as cr } from 'aws-cdk-lib';
const aws_cdk_lib_1 = require("aws-cdk-lib");
const codebuild = require("aws-cdk-lib/aws-codebuild");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
// eslint-disable-next-line import/no-extraneous-dependencies
const s3 = require("aws-cdk-lib/aws-s3");
const cr = require("aws-cdk-lib/custom-resources");
const statement = require("cdk-iam-floyd");
const constructs_1 = require("constructs");
/**
 * Creates a CodeBuild project to audit an AWS account with Prowler and stores the html report in a S3 bucket. This will run onece at the beginning and on a schedule afterwards. Partial contribution from https://github.com/stevecjones
 */
class ProwlerAudit extends constructs_1.Construct {
    constructor(parent, id, props) {
        var _b, _c, _d, _e, _f;
        super(parent, id);
        // defaults
        this.serviceName = (props === null || props === void 0 ? void 0 : props.serviceName) ? props.serviceName : 'prowler';
        this.logsRetentionInDays = (props === null || props === void 0 ? void 0 : props.logsRetentionInDays) ? props.logsRetentionInDays : logs.RetentionDays.THREE_DAYS;
        this.enableScheduler = (props === null || props === void 0 ? void 0 : props.enableScheduler) ? props.enableScheduler : false;
        this.prowlerScheduler = (props === null || props === void 0 ? void 0 : props.prowlerScheduler) ? props.prowlerScheduler : 'cron(0 22 * * ? *)';
        this.prowlerOptions = (props === null || props === void 0 ? void 0 : props.prowlerOptions) ? props.prowlerOptions : '-M text,junit-xml,html,csv,json';
        this.prowlerVersion = (props === null || props === void 0 ? void 0 : props.prowlerVersion) ? props.prowlerVersion : '2.7.0';
        const reportBucket = (_b = props === null || props === void 0 ? void 0 : props.reportBucket) !== null && _b !== void 0 ? _b : new s3.Bucket(this, 'ReportBucket', {
            //bucketName: `${'123456'}-prowler-reports`,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        const reportGroup = new codebuild.ReportGroup(this, 'reportGroup', { /**reportGroupName: 'testReportGroup', */ removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY });
        const preBuildCommands = [];
        if (!!(props === null || props === void 0 ? void 0 : props.allowlist)) {
            const prowlerFilename = 'allowlist.txt';
            if (props.allowlist.isZipArchive) {
                preBuildCommands.push(`aws s3 cp ${props.allowlist.s3ObjectUrl} .`);
                preBuildCommands.push(`unzip ${props.allowlist.s3ObjectKey} -d prowler`);
            }
            else {
                preBuildCommands.push(`aws s3 cp ${props.allowlist.s3ObjectUrl} prowler/${prowlerFilename}`);
            }
            this.prowlerOptions = this.prowlerOptions + ` -w ${prowlerFilename}`;
        }
        const prowlerBuild = this.codebuildProject = new codebuild.Project(this, 'prowlerBuild', {
            description: 'Run Prowler assessment',
            timeout: aws_cdk_lib_1.Duration.hours(5),
            environment: {
                environmentVariables: {
                    BUCKET_REPORT: { value: reportBucket.bucketName || '' },
                    BUCKET_PREFIX: { value: (_c = props === null || props === void 0 ? void 0 : props.reportBucketPrefix) !== null && _c !== void 0 ? _c : '' },
                    ADDITIONAL_S3_ARGS: { value: (_d = props === null || props === void 0 ? void 0 : props.additionalS3CopyArgs) !== null && _d !== void 0 ? _d : '' },
                    PROWLER_OPTIONS: { value: this.prowlerOptions || '' },
                },
                buildImage: codebuild.LinuxBuildImage.fromCodeBuildImageId('aws/codebuild/amazonlinux2-x86_64-standard:3.0'),
            },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        'runtime-versions': {
                            python: 3.8,
                        },
                        'commands': [
                            'echo "Installing Prowler and dependencies..."',
                            'pip3 install detect-secrets',
                            'yum -y install jq',
                            'curl "https://awscli.amazonaws.com/awscli-exe-linux-x86_64.zip" -o "awscliv2.zip"',
                            'unzip awscliv2.zip',
                            './aws/install',
                            `git clone -b ${this.prowlerVersion} https://github.com/prowler-cloud/prowler`,
                        ],
                    },
                    pre_build: {
                        commands: preBuildCommands,
                    },
                    build: {
                        commands: [
                            `echo "Running Prowler as ./prowler ${this.prowlerOptions} && echo OK || echo FAILED"`,
                            'cd prowler',
                            `./prowler ${this.prowlerOptions} && echo OK || echo FAILED`,
                        ],
                    },
                    post_build: {
                        commands: [
                            'echo "Uploading reports to S3..." ',
                            'aws s3 cp --sse AES256 output/ s3://$BUCKET_REPORT/$BUCKET_PREFIX --recursive $ADDITIONAL_S3_ARGS',
                            'echo "Done!"',
                        ],
                    },
                },
                reports: {
                    [reportGroup.reportGroupName]: {
                        'files': ['**/*'],
                        'base-directory': 'prowler/junit-reports',
                        'file-format': 'JunitXml',
                    },
                },
            }),
        });
        if (!!(props === null || props === void 0 ? void 0 : props.allowlist)) {
            props.allowlist.bucket.grantRead(prowlerBuild);
        }
        (_e = prowlerBuild.role) === null || _e === void 0 ? void 0 : _e.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('SecurityAudit'));
        (_f = prowlerBuild.role) === null || _f === void 0 ? void 0 : _f.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('job-function/ViewOnlyAccess'));
        // prowlerBuild.addToRolePolicy(new statement.Dax().allow().to());
        prowlerBuild.addToRolePolicy(new statement.Ds().allow().toListAuthorizedApplications());
        prowlerBuild.addToRolePolicy(new statement.Ec2().allow().toGetEbsEncryptionByDefault());
        prowlerBuild.addToRolePolicy(new statement.Ecr().allow().toDescribeImageScanFindings().toDescribeImages().toDescribeRegistry());
        prowlerBuild.addToRolePolicy(new statement.Tag().allow().toGetTagKeys());
        prowlerBuild.addToRolePolicy(new statement.Lambda().allow().toGetFunction());
        prowlerBuild.addToRolePolicy(new statement.Glue().allow().toSearchTables().toGetConnections());
        prowlerBuild.addToRolePolicy(new statement.Apigateway().allow().toGET());
        prowlerBuild.addToRolePolicy(new iam.PolicyStatement({ actions: ['support:Describe*'], resources: ['*'] }));
        reportBucket.grantPut(prowlerBuild);
        const myRole = new iam.Role(this, 'MyRole', { assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com') });
        const prowlerStartBuildLambda = new lambda.Function(this, 'Lambda', {
            runtime: lambda.Runtime.PYTHON_3_6,
            timeout: aws_cdk_lib_1.Duration.seconds(120),
            handler: 'index.lambda_handler',
            code: lambda.Code.fromInline(`import boto3
import cfnresponse
from botocore.exceptions import ClientError
def lambda_handler(event,context):
  props = event['ResourceProperties']
  codebuild_client = boto3.client('codebuild')
  if (event['RequestType'] == 'Create' or event['RequestType'] == 'Update'):
    try:
        response = codebuild_client.start_build(projectName=props['Build'])
        print(response)
        print("Respond: SUCCESS")
        cfnresponse.send(event, context, cfnresponse.SUCCESS, {})
    except Exception as ex:
        print(ex.response['Error']['Message'])
        cfnresponse.send(event, context, cfnresponse.FAILED, ex.response)
      `),
        });
        prowlerStartBuildLambda.addToRolePolicy(new statement.Codebuild().allow().toStartBuild()); // onResource project ...
        const myProvider = new cr.Provider(this, 'MyProvider', {
            onEventHandler: prowlerStartBuildLambda,
            logRetention: this.logsRetentionInDays,
            role: myRole,
        });
        new aws_cdk_lib_1.CustomResource(this, 'Resource1', {
            serviceToken: myProvider.serviceToken,
            properties: {
                Build: prowlerBuild.projectName,
                RERUN_PROWLER: Boolean(this.node.tryGetContext('reRunProwler')) ? Date.now().toString() : '',
            },
        });
        if (this.enableScheduler) {
            const prowlerSchedulerLambda = new lambda.Function(this, 'ScheduleLambda', {
                runtime: lambda.Runtime.PYTHON_3_6,
                timeout: aws_cdk_lib_1.Duration.seconds(120),
                handler: 'index.lambda_handler',
                environment: {
                    buildName: prowlerBuild.projectName,
                },
                code: lambda.Code.fromInline(`import boto3
        import os
        def lambda_handler(event,context):
          codebuild_client = boto3.client('codebuild')
          print("Running Prowler scheduled!: " + os.environ['buildName'])
          project_name = os.environ['buildName']
          response = codebuild_client.start_build(projectName=project_name)
          print(response)
          print("Respond: SUCCESS")
        `),
            });
            new events.Rule(this, 'Rule', {
                description: 'A schedule for the Lambda function that triggers Prowler in CodeBuild.',
                targets: [new targets.LambdaFunction(prowlerSchedulerLambda)],
                schedule: events.Schedule.expression(this.prowlerScheduler || ''),
            });
        }
    }
}
exports.ProwlerAudit = ProwlerAudit;
_a = JSII_RTTI_SYMBOL_1;
ProwlerAudit[_a] = { fqn: "cdk-prowler.ProwlerAudit", version: "2.16.0" };
//# sourceMappingURL=data:application/json;base64,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