#!/usr/bin/env python3
"""
安全修复验证测试
验证CSRF保护、内存安全、配置管理等修复效果
"""

import sys
import os
import time

# 移除src目录路径添加

def test_config_management():
    """测试统一配置管理"""
    print("🧪 测试统一配置管理...")
    
    try:
        from backend.config import get_config, get_security_config, get_server_config
        
        print("1. 获取配置管理器...")
        config = get_config()
        print(f"✅ 配置管理器创建成功: {type(config)}")
        
        print("2. 测试配置验证...")
        is_valid = config.validate_config()
        print(f"✅ 配置验证结果: {is_valid}")
        
        print("3. 测试安全配置...")
        security_config = get_security_config()
        print(f"✅ CSRF令牌字节数: {security_config.csrf_token_bytes}")
        print(f"✅ 最大内容长度: {security_config.max_content_length}")
        print(f"✅ 允许的文件扩展名: {security_config.allowed_extensions}")
        
        print("4. 测试服务器配置...")
        server_config = get_server_config()
        print(f"✅ 端口范围: {server_config.port_range_start}-{server_config.port_range_end}")
        print(f"✅ 默认超时: {server_config.default_timeout}秒")
        
        print("5. 测试配置字典转换...")
        config_dict = config.to_dict()
        print(f"✅ 配置项数量: {len(config_dict)}个主要类别")
        
        print("✅ 配置管理测试完成")
        
    except Exception as e:
        print(f"❌ 配置管理测试失败: {e}")
        import traceback
        traceback.print_exc()

def test_csrf_protection():
    """测试CSRF保护"""
    print("\n🧪 测试CSRF保护...")
    
    try:
        from backend.app import CSRFProtection
        
        print("1. 创建CSRF保护实例...")
        csrf = CSRFProtection()
        print(f"✅ CSRF保护创建成功: {type(csrf)}")
        
        print("2. 测试令牌生成...")
        token1 = csrf.generate_token()
        token2 = csrf.generate_token()
        print(f"✅ 令牌1: {token1[:16]}...")
        print(f"✅ 令牌2: {token2[:16]}...")
        print(f"✅ 令牌唯一性: {token1 != token2}")
        
        print("3. 测试令牌验证...")
        is_valid = csrf.validate_token(token1)
        print(f"✅ 令牌验证结果: {is_valid}")
        
        print("4. 测试一次性令牌...")
        is_valid_again = csrf.validate_token(token1)
        print(f"✅ 重复验证结果（应为False）: {is_valid_again}")
        
        print("5. 测试无效令牌...")
        is_invalid = csrf.validate_token("invalid_token")
        print(f"✅ 无效令牌验证结果（应为False）: {is_invalid}")
        
        print("✅ CSRF保护测试完成")
        
    except Exception as e:
        print(f"❌ CSRF保护测试失败: {e}")
        import traceback
        traceback.print_exc()

def test_memory_safety():
    """测试内存安全检查"""
    print("\n🧪 测试内存安全检查...")
    
    try:
        from backend.app import FeedbackApp
        from backend.feedback_handler import FeedbackHandler
        
        print("1. 创建应用实例...")
        handler = FeedbackHandler()
        app_instance = FeedbackApp(handler, "测试", "")
        print(f"✅ 应用实例创建成功: {type(app_instance)}")
        
        print("2. 测试小数据内存检查...")
        small_data = {
            'text': 'Hello World',
            'images': [],
            'timestamp': time.time()
        }
        is_safe = app_instance._check_memory_safety(small_data)
        print(f"✅ 小数据安全检查: {is_safe}")
        
        print("3. 测试大数据内存检查...")
        large_data = {
            'text': 'x' * (10 * 1024 * 1024),  # 10MB文本
            'images': ['x' * (5 * 1024 * 1024)] * 10,  # 10个5MB图片
            'timestamp': time.time()
        }
        is_unsafe = app_instance._check_memory_safety(large_data)
        print(f"✅ 大数据安全检查（应为False）: {is_unsafe}")
        
        print("4. 测试图片格式验证...")
        # PNG文件头
        png_header = b'\x89PNG\r\n\x1a\n' + b'fake_png_data'
        is_png_valid = app_instance._validate_image_data(png_header)
        print(f"✅ PNG格式验证: {is_png_valid}")
        
        # 无效数据
        invalid_data = b'invalid_image_data'
        is_invalid = app_instance._validate_image_data(invalid_data)
        print(f"✅ 无效格式验证（应为False）: {is_invalid}")
        
        print("✅ 内存安全测试完成")
        
    except Exception as e:
        print(f"❌ 内存安全测试失败: {e}")
        import traceback
        traceback.print_exc()

def test_queue_limits():
    """测试队列限制"""
    print("\n🧪 测试队列限制...")
    
    try:
        from backend.feedback_handler import FeedbackHandler
        
        print("1. 创建有限制的反馈处理器...")
        handler = FeedbackHandler(max_queue_size=3)
        print(f"✅ 处理器创建成功，队列大小限制: {handler.max_queue_size}")
        
        print("2. 测试队列填充...")
        for i in range(3):
            handler.put_result({'test': f'data_{i}'})
            print(f"   添加数据 {i+1}/3")
        
        print("3. 测试队列满时的行为...")
        try:
            # 这应该会阻塞或抛出异常，取决于Queue的实现
            import queue
            handler.result_queue.put({'test': 'overflow'}, block=False)
            print("⚠️ 队列溢出未被阻止")
        except queue.Full:
            print("✅ 队列满时正确阻止添加")
        
        print("4. 清理队列...")
        handler.clear_queue()
        print("✅ 队列清理完成")
        
        print("✅ 队列限制测试完成")
        
    except Exception as e:
        print(f"❌ 队列限制测试失败: {e}")
        import traceback
        traceback.print_exc()

def test_version_history():
    """测试版本历史修复"""
    print("\n🧪 测试版本历史修复...")
    
    try:
        from backend.version import VERSION_HISTORY, get_version_history
        
        print("1. 检查版本历史...")
        history = get_version_history()
        print(f"✅ 版本历史条目数: {len(history)}")
        
        print("2. 检查重复版本...")
        versions = list(history.keys())
        unique_versions = set(versions)
        has_duplicates = len(versions) != len(unique_versions)
        print(f"✅ 是否有重复版本（应为False）: {has_duplicates}")
        
        print("3. 显示版本历史...")
        for version, description in history.items():
            print(f"   {version}: {description}")
        
        print("✅ 版本历史测试完成")
        
    except Exception as e:
        print(f"❌ 版本历史测试失败: {e}")
        import traceback
        traceback.print_exc()

if __name__ == "__main__":
    print("🔒 开始安全修复验证测试...\n")
    
    test_config_management()
    test_csrf_protection()
    test_memory_safety()
    test_queue_limits()
    test_version_history()
    
    print("\n🎉 所有安全测试完成！")
    print("\n📋 修复总结:")
    print("✅ CSRF保护 - 防止跨站请求伪造攻击")
    print("✅ 内存安全 - 精确计算内存使用，防止溢出")
    print("✅ 队列限制 - 防止内存泄漏")
    print("✅ 配置管理 - 统一配置，避免硬编码")
    print("✅ 版本历史 - 清理重复条目")
    print("✅ 图片验证 - 文件头魔数验证")
    print("✅ 文件安全 - 安全文件名处理")
