"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CustomRule = exports.RedirectStatus = exports.App = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const amplify_generated_1 = require("./amplify.generated");
const branch_1 = require("./branch");
const domain_1 = require("./domain");
const utils_1 = require("./utils");
/**
 * An Amplify Console application
 */
class App extends core_1.Resource {
    constructor(scope, id, props) {
        var _a, _b, _c;
        super(scope, id);
        this.customRules = props.customRules || [];
        this.environmentVariables = props.environmentVariables || {};
        this.autoBranchEnvironmentVariables = props.autoBranchCreation && props.autoBranchCreation.environmentVariables || {};
        const role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('amplify.amazonaws.com'),
        });
        this.grantPrincipal = role;
        const sourceCodeProviderOptions = (_a = props.sourceCodeProvider) === null || _a === void 0 ? void 0 : _a.bind(this);
        const app = new amplify_generated_1.CfnApp(this, 'Resource', {
            accessToken: (_b = sourceCodeProviderOptions === null || sourceCodeProviderOptions === void 0 ? void 0 : sourceCodeProviderOptions.accessToken) === null || _b === void 0 ? void 0 : _b.toString(),
            autoBranchCreationConfig: props.autoBranchCreation && {
                autoBranchCreationPatterns: props.autoBranchCreation.patterns,
                basicAuthConfig: props.autoBranchCreation.basicAuth && props.autoBranchCreation.basicAuth.bind(this, 'BranchBasicAuth'),
                buildSpec: props.autoBranchCreation.buildSpec && props.autoBranchCreation.buildSpec.toBuildSpec(),
                enableAutoBranchCreation: true,
                enableAutoBuild: props.autoBranchCreation.autoBuild === undefined ? true : props.autoBranchCreation.autoBuild,
                environmentVariables: core_1.Lazy.anyValue({ produce: () => utils_1.renderEnvironmentVariables(this.autoBranchEnvironmentVariables) }, { omitEmptyArray: true }),
                enablePullRequestPreview: props.autoBranchCreation.pullRequestPreview === undefined ? true : props.autoBranchCreation.pullRequestPreview,
                pullRequestEnvironmentName: props.autoBranchCreation.pullRequestEnvironmentName,
                stage: props.autoBranchCreation.stage,
            },
            enableBranchAutoDeletion: props.autoBranchDeletion,
            basicAuthConfig: props.basicAuth && props.basicAuth.bind(this, 'AppBasicAuth'),
            buildSpec: props.buildSpec && props.buildSpec.toBuildSpec(),
            customRules: core_1.Lazy.anyValue({ produce: () => this.customRules }, { omitEmptyArray: true }),
            description: props.description,
            environmentVariables: core_1.Lazy.anyValue({ produce: () => utils_1.renderEnvironmentVariables(this.environmentVariables) }, { omitEmptyArray: true }),
            iamServiceRole: role.roleArn,
            name: props.appName || this.node.id,
            oauthToken: (_c = sourceCodeProviderOptions === null || sourceCodeProviderOptions === void 0 ? void 0 : sourceCodeProviderOptions.oauthToken) === null || _c === void 0 ? void 0 : _c.toString(),
            repository: sourceCodeProviderOptions === null || sourceCodeProviderOptions === void 0 ? void 0 : sourceCodeProviderOptions.repository,
        });
        this.appId = app.attrAppId;
        this.appName = app.attrAppName;
        this.arn = app.attrArn;
        this.defaultDomain = app.attrDefaultDomain;
    }
    /**
     * Import an existing application
     */
    static fromAppId(scope, id, appId) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.appId = appId;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds a custom rewrite/redirect rule to this application
     */
    addCustomRule(rule) {
        this.customRules.push(rule);
        return this;
    }
    /**
     * Adds an environment variable to this application.
     *
     * All environment variables that you add are encrypted to prevent rogue
     * access so you can use them to store secret information.
     */
    addEnvironment(name, value) {
        this.environmentVariables[name] = value;
        return this;
    }
    /**
     * Adds an environment variable to the auto created branch.
     *
     * All environment variables that you add are encrypted to prevent rogue
     * access so you can use them to store secret information.
     */
    addAutoBranchEnvironment(name, value) {
        this.autoBranchEnvironmentVariables[name] = value;
        return this;
    }
    /**
     * Adds a branch to this application
     */
    addBranch(id, options = {}) {
        return new branch_1.Branch(this, id, {
            ...options,
            app: this,
        });
    }
    /**
     * Adds a domain to this application
     */
    addDomain(id, options = {}) {
        return new domain_1.Domain(this, id, {
            ...options,
            app: this,
        });
    }
}
exports.App = App;
/**
 * The status code for a URL rewrite or redirect rule.
 */
var RedirectStatus;
(function (RedirectStatus) {
    /**
     * Rewrite (200)
     */
    RedirectStatus["REWRITE"] = "200";
    /**
     * Permanent redirect (301)
     */
    RedirectStatus["PERMANENT_REDIRECT"] = "301";
    /**
     * Temporary redirect (302)
     */
    RedirectStatus["TEMPORARY_REDIRECT"] = "302";
    /**
     * Not found (404)
     */
    RedirectStatus["NOT_FOUND"] = "404";
    /**
     * Not found rewrite (404)
     */
    RedirectStatus["NOT_FOUND_REWRITE"] = "404-200";
})(RedirectStatus = exports.RedirectStatus || (exports.RedirectStatus = {}));
/**
 * Custom rewrite/redirect rule for an Amplify App.
 *
 * @see https://docs.aws.amazon.com/amplify/latest/userguide/redirects.html
 */
class CustomRule {
    constructor(options) {
        this.source = options.source;
        this.target = options.target;
        this.status = options.status;
        this.condition = options.condition;
    }
}
exports.CustomRule = CustomRule;
/**
 * Sets up a 200 rewrite for all paths to `index.html` except for path
 * containing a file extension.
 */
CustomRule.SINGLE_PAGE_APPLICATION_REDIRECT = new CustomRule({
    source: '</^[^.]+$/>',
    target: '/index.html',
    status: RedirectStatus.REWRITE,
});
//# sourceMappingURL=data:application/json;base64,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