"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.HttpStage = void 0;
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const apigatewayv2_generated_1 = require("../apigatewayv2.generated");
const api_mapping_1 = require("./api-mapping");
const DEFAULT_STAGE_NAME = '$default';
/**
 * Represents a stage where an instance of the API is deployed.
 * @resource AWS::ApiGatewayV2::Stage
 */
class HttpStage extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.stageName ? props.stageName : DEFAULT_STAGE_NAME,
        });
        new apigatewayv2_generated_1.CfnStage(this, 'Resource', {
            apiId: props.httpApi.httpApiId,
            stageName: this.physicalName,
            autoDeploy: props.autoDeploy,
        });
        this.stageName = this.physicalName;
        this.httpApi = props.httpApi;
        if (props.domainMapping) {
            new api_mapping_1.HttpApiMapping(this, `${props.domainMapping.domainName}${props.domainMapping.mappingKey}`, {
                api: props.httpApi,
                domainName: props.domainMapping.domainName,
                stage: this,
                apiMappingKey: props.domainMapping.mappingKey,
            });
            // ensure the dependency
            this.node.addDependency(props.domainMapping.domainName);
        }
    }
    /**
     * Import an existing stage into this CDK app.
     */
    static fromStageName(scope, id, stageName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.stageName = stageName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * The URL to this stage.
     */
    get url() {
        const s = core_1.Stack.of(this);
        const urlPath = this.stageName === DEFAULT_STAGE_NAME ? '' : this.stageName;
        return `https://${this.httpApi.httpApiId}.execute-api.${s.region}.${s.urlSuffix}/${urlPath}`;
    }
}
exports.HttpStage = HttpStage;
//# sourceMappingURL=data:application/json;base64,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