"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Schedule = void 0;
/**
 * Schedule for scheduled scaling actions
 */
class Schedule {
    constructor() {
    }
    /**
     * Construct a schedule from a literal schedule expression
     *
     * @param expression The expression to use. Must be in a format that Application AutoScaling will recognize
     */
    static expression(expression) {
        return new LiteralSchedule(expression);
    }
    /**
     * Construct a schedule from an interval and a time unit
     */
    static rate(duration) {
        if (duration.toSeconds() === 0) {
            throw new Error('Duration cannot be 0');
        }
        let rate = maybeRate(duration.toDays({ integral: false }), 'day');
        if (rate === undefined) {
            rate = maybeRate(duration.toHours({ integral: false }), 'hour');
        }
        if (rate === undefined) {
            rate = makeRate(duration.toMinutes({ integral: true }), 'minute');
        }
        return new LiteralSchedule(rate);
    }
    /**
     * Construct a Schedule from a moment in time
     */
    static at(moment) {
        return new LiteralSchedule(`at(${formatISO(moment)})`);
    }
    /**
     * Create a schedule from a set of cron fields
     */
    static cron(options) {
        if (options.weekDay !== undefined && options.day !== undefined) {
            throw new Error('Cannot supply both \'day\' and \'weekDay\', use at most one');
        }
        const minute = fallback(options.minute, '*');
        const hour = fallback(options.hour, '*');
        const month = fallback(options.month, '*');
        const year = fallback(options.year, '*');
        // Weekday defaults to '?' if not supplied. If it is supplied, day must become '?'
        const day = fallback(options.day, options.weekDay !== undefined ? '?' : '*');
        const weekDay = fallback(options.weekDay, '?');
        return new LiteralSchedule(`cron(${minute} ${hour} ${day} ${month} ${weekDay} ${year})`);
    }
}
exports.Schedule = Schedule;
class LiteralSchedule extends Schedule {
    constructor(expressionString) {
        super();
        this.expressionString = expressionString;
    }
}
function fallback(x, def) {
    return x === undefined ? def : x;
}
function formatISO(date) {
    if (!date) {
        return undefined;
    }
    return date.getUTCFullYear() +
        '-' + pad(date.getUTCMonth() + 1) +
        '-' + pad(date.getUTCDate()) +
        'T' + pad(date.getUTCHours()) +
        ':' + pad(date.getUTCMinutes()) +
        ':' + pad(date.getUTCSeconds());
    function pad(num) {
        if (num < 10) {
            return '0' + num;
        }
        return num;
    }
}
/**
 * Return the rate if the rate is whole number
 */
function maybeRate(interval, singular) {
    if (interval === 0 || !Number.isInteger(interval)) {
        return undefined;
    }
    return makeRate(interval, singular);
}
/**
 * Return 'rate(${interval} ${singular}(s))` for the interval
 */
function makeRate(interval, singular) {
    return interval === 1 ? `rate(1 ${singular})` : `rate(${interval} ${singular}s)`;
}
//# sourceMappingURL=data:application/json;base64,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