"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PredefinedMetric = exports.TargetTrackingScalingPolicy = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const applicationautoscaling_generated_1 = require("./applicationautoscaling.generated");
class TargetTrackingScalingPolicy extends cdk.Construct {
    constructor(scope, id, props) {
        if ((props.customMetric === undefined) === (props.predefinedMetric === undefined)) {
            throw new Error('Exactly one of \'customMetric\' or \'predefinedMetric\' must be specified.');
        }
        if (props.customMetric && !props.customMetric.toMetricConfig().metricStat) {
            throw new Error('Only direct metrics are supported for Target Tracking. Use Step Scaling or supply a Metric object.');
        }
        super(scope, id);
        const resource = new applicationautoscaling_generated_1.CfnScalingPolicy(this, 'Resource', {
            policyName: props.policyName || this.node.uniqueId,
            policyType: 'TargetTrackingScaling',
            scalingTargetId: props.scalingTarget.scalableTargetId,
            targetTrackingScalingPolicyConfiguration: {
                customizedMetricSpecification: renderCustomMetric(props.customMetric),
                disableScaleIn: props.disableScaleIn,
                predefinedMetricSpecification: props.predefinedMetric !== undefined ? {
                    predefinedMetricType: props.predefinedMetric,
                    resourceLabel: props.resourceLabel,
                } : undefined,
                scaleInCooldown: props.scaleInCooldown && props.scaleInCooldown.toSeconds(),
                scaleOutCooldown: props.scaleOutCooldown && props.scaleOutCooldown.toSeconds(),
                targetValue: props.targetValue,
            },
        });
        this.scalingPolicyArn = resource.ref;
    }
}
exports.TargetTrackingScalingPolicy = TargetTrackingScalingPolicy;
function renderCustomMetric(metric) {
    if (!metric) {
        return undefined;
    }
    const c = metric.toMetricConfig().metricStat;
    if (c.statistic.startsWith('p')) {
        throw new Error(`Cannot use statistic '${c.statistic}' for Target Tracking: only 'Average', 'Minimum', 'Maximum', 'SampleCount', and 'Sum' are supported.`);
    }
    return {
        dimensions: c.dimensions,
        metricName: c.metricName,
        namespace: c.namespace,
        statistic: c.statistic,
        unit: c.unitFilter,
    };
}
/**
 * One of the predefined autoscaling metrics
 */
var PredefinedMetric;
(function (PredefinedMetric) {
    /**
     * DYNAMODB_READ_CAPACITY_UTILIZATIO
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["DYNAMODB_READ_CAPACITY_UTILIZATION"] = "DynamoDBReadCapacityUtilization";
    /**
     * DYANMODB_WRITE_CAPACITY_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["DYANMODB_WRITE_CAPACITY_UTILIZATION"] = "DynamoDBWriteCapacityUtilization";
    /**
     * ALB_REQUEST_COUNT_PER_TARGET
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["ALB_REQUEST_COUNT_PER_TARGET"] = "ALBRequestCountPerTarget";
    /**
     * RDS_READER_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["RDS_READER_AVERAGE_CPU_UTILIZATION"] = "RDSReaderAverageCPUUtilization";
    /**
     * RDS_READER_AVERAGE_DATABASE_CONNECTIONS
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["RDS_READER_AVERAGE_DATABASE_CONNECTIONS"] = "RDSReaderAverageDatabaseConnections";
    /**
     * EC2_SPOT_FLEET_REQUEST_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_CPU_UTILIZATION"] = "EC2SpotFleetRequestAverageCPUUtilization";
    /**
     * EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_IN
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_IN"] = "EC2SpotFleetRequestAverageNetworkIn";
    /**
     * EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_OUT
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["EC2_SPOT_FLEET_REQUEST_AVERAGE_NETWORK_OUT"] = "EC2SpotFleetRequestAverageNetworkOut";
    /**
     * SAGEMAKER_VARIANT_INVOCATIONS_PER_INSTANCE
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["SAGEMAKER_VARIANT_INVOCATIONS_PER_INSTANCE"] = "SageMakerVariantInvocationsPerInstance";
    /**
     * ECS_SERVICE_AVERAGE_CPU_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["ECS_SERVICE_AVERAGE_CPU_UTILIZATION"] = "ECSServiceAverageCPUUtilization";
    /**
     * ECS_SERVICE_AVERAGE_MEMORY_UTILIZATION
     * @see https://docs.aws.amazon.com/autoscaling/application/APIReference/API_PredefinedMetricSpecification.html
     */
    PredefinedMetric["ECS_SERVICE_AVERAGE_MEMORY_UTILIZATION"] = "ECSServiceAverageMemoryUtilization";
    /**
     * LAMBDA_PROVISIONED_CONCURRENCY_UTILIZATION
     * @see https://docs.aws.amazon.com/lambda/latest/dg/monitoring-metrics.html#monitoring-metrics-concurrency
     */
    PredefinedMetric["LAMBDA_PROVISIONED_CONCURRENCY_UTILIZATION"] = "LambdaProvisionedConcurrencyUtilization";
})(PredefinedMetric = exports.PredefinedMetric || (exports.PredefinedMetric = {}));
//# sourceMappingURL=data:application/json;base64,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