"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Route = exports.RouteType = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
/**
 * Type of route
 */
var RouteType;
(function (RouteType) {
    /**
     * HTTP route
     */
    RouteType["HTTP"] = "http";
    /**
     * TCP route
     */
    RouteType["TCP"] = "tcp";
})(RouteType = exports.RouteType || (exports.RouteType = {}));
/**
 * Route represents a new or existing route attached to a VirtualRouter and Mesh
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/routes.html
 */
class Route extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.routeName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        this.weightedTargets = new Array();
        this.virtualRouter = props.virtualRouter;
        const routeType = props.routeType !== undefined ? props.routeType :
            props.prefix !== undefined ? RouteType.HTTP :
                RouteType.TCP;
        if (routeType === RouteType.HTTP) {
            this.httpRoute = this.renderHttpRoute(props);
        }
        else {
            this.tcpRoute = this.renderTcpRoute(props);
        }
        const route = new appmesh_generated_1.CfnRoute(this, 'Resource', {
            routeName: this.physicalName,
            meshName: this.virtualRouter.mesh.meshName,
            virtualRouterName: this.virtualRouter.virtualRouterName,
            spec: {
                tcpRoute: this.tcpRoute,
                httpRoute: this.httpRoute,
            },
        });
        this.routeName = this.getResourceNameAttribute(route.attrRouteName);
        this.routeArn = this.getResourceArnAttribute(route.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualRouter/${props.virtualRouter.virtualRouterName}/route`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing route given an ARN
     */
    static fromRouteArn(scope, id, routeArn) {
        return new ImportedRoute(scope, id, { routeArn });
    }
    /**
     * Import an existing route given its name
     */
    static fromRouteName(scope, id, meshName, virtualRouterName, routeName) {
        return new ImportedRoute(scope, id, { meshName, virtualRouterName, routeName });
    }
    /**
     * Utility method to add weighted route targets to an existing route
     */
    renderWeightedTargets(props) {
        for (const t of props) {
            this.weightedTargets.push({
                virtualNode: t.virtualNode.virtualNodeName,
                weight: t.weight || 1,
            });
        }
        return this.weightedTargets;
    }
    renderHttpRoute(props) {
        return {
            match: {
                prefix: props.prefix || '/',
            },
            action: {
                weightedTargets: this.renderWeightedTargets(props.routeTargets),
            },
        };
    }
    renderTcpRoute(props) {
        return {
            action: {
                weightedTargets: this.renderWeightedTargets(props.routeTargets),
            },
        };
    }
}
exports.Route = Route;
/**
 * Represents and imported IRoute
 */
class ImportedRoute extends cdk.Resource {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.routeArn) {
            this.routeArn = props.routeArn;
            this.routeName = cdk.Fn.select(4, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.routeArn).resourceName));
        }
        else if (props.routeName && props.meshName && props.virtualRouterName) {
            this.routeName = props.routeName;
            this.routeArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualRouter/${props.virtualRouterName}/route`,
                resourceName: this.routeName,
            });
        }
        else {
            throw new Error('Need either arn or three names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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