import * as cloudmap from "../../aws-servicediscovery";
import * as cdk from "../../core";
import { Construct } from 'constructs';
import { CfnVirtualNode } from './appmesh.generated';
import { IMesh } from './mesh';
import { AccessLog, VirtualNodeListener } from './shared-interfaces';
import { IVirtualService } from './virtual-service';
/**
 * Interface which all VirtualNode based classes must implement
 */
export interface IVirtualNode extends cdk.IResource {
    /**
     * The name of the VirtualNode
     *
     * @attribute
     */
    readonly virtualNodeName: string;
    /**
     * The Amazon Resource Name belonging to the VirtualNdoe
     *
     * Set this value as the APPMESH_VIRTUAL_NODE_NAME environment variable for
     * your task group's Envoy proxy container in your task definition or pod
     * spec.
     *
     * @attribute
     */
    readonly virtualNodeArn: string;
    /**
     * Utility method to add backends for existing or new VirtualNodes
     */
    addBackends(...props: IVirtualService[]): void;
    /**
     * Utility method to add Node Listeners for new or existing VirtualNodes
     */
    addListeners(...listeners: VirtualNodeListener[]): void;
}
/**
 * Basic configuration properties for a VirtualNode
 */
export interface VirtualNodeBaseProps {
    /**
     * The name of the VirtualNode
     *
     * @default - A name is automatically determined
     */
    readonly virtualNodeName?: string;
    /**
     * Host name of DNS record used to discover Virtual Node members
     *
     * The IP addresses returned by querying this DNS record will be considered
     * part of the Virtual Node.
     *
     * @default - Don't use DNS-based service discovery
     */
    readonly dnsHostName?: string;
    /**
     * CloudMap service where Virtual Node members register themselves
     *
     * Instances registering themselves into this CloudMap will
     * be considered part of the Virtual Node.
     *
     * @default - Don't use CloudMap-based service discovery
     */
    readonly cloudMapService?: cloudmap.IService;
    /**
     * Filter down the list of CloudMap service instance
     *
     * @default - No CloudMap instance filter
     */
    readonly cloudMapServiceInstanceAttributes?: {
        [key: string]: string;
    };
    /**
     * Virtual Services that this is node expected to send outbound traffic to
     *
     * @default - No backends
     */
    readonly backends?: IVirtualService[];
    /**
     * Initial listener for the virtual node
     *
     * @default - No listeners
     */
    readonly listener?: VirtualNodeListener;
    /**
     * Access Logging Configuration for the virtual node
     *
     * @default - No access logging
     */
    readonly accessLog?: AccessLog;
}
/**
 * The properties used when creating a new VirtualNode
 */
export interface VirtualNodeProps extends VirtualNodeBaseProps {
    /**
     * The name of the AppMesh which the virtual node belongs to
     */
    readonly mesh: IMesh;
}
declare abstract class VirtualNodeBase extends cdk.Resource implements IVirtualNode {
    /**
     * The name of the VirtualNode
     */
    abstract readonly virtualNodeName: string;
    /**
     * The Amazon Resource Name belonging to the VirtualNdoe
     */
    abstract readonly virtualNodeArn: string;
    protected readonly backends: CfnVirtualNode.BackendProperty[];
    protected readonly listeners: CfnVirtualNode.ListenerProperty[];
    /**
     * Add a Virtual Services that this node is expected to send outbound traffic to
     */
    addBackends(...props: IVirtualService[]): void;
    /**
     * Utility method to add an inbound listener for this virtual node
     */
    addListeners(...listeners: VirtualNodeListener[]): void;
}
/**
 * VirtualNode represents a newly defined AppMesh VirtualNode
 *
 * Any inbound traffic that your virtual node expects should be specified as a
 * listener. Any outbound traffic that your virtual node expects to reach
 * should be specified as a backend.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html
 */
export declare class VirtualNode extends VirtualNodeBase {
    /**
     * Import an existing VirtualNode given an ARN
     */
    static fromVirtualNodeArn(scope: Construct, id: string, virtualNodeArn: string): IVirtualNode;
    /**
     * Import an existing VirtualNode given its name
     */
    static fromVirtualNodeName(scope: Construct, id: string, meshName: string, virtualNodeName: string): IVirtualNode;
    /**
     * The name of the VirtualNode
     */
    readonly virtualNodeName: string;
    /**
     * The Amazon Resource Name belonging to the VirtualNdoe
     */
    readonly virtualNodeArn: string;
    /**
     * The service mesh that the virtual node resides in
     */
    readonly mesh: IMesh;
    constructor(scope: Construct, id: string, props: VirtualNodeProps);
}
export {};
