"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.VirtualNode = void 0;
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appmesh_generated_1 = require("./appmesh.generated");
const shared_interfaces_1 = require("./shared-interfaces");
class VirtualNodeBase extends cdk.Resource {
    constructor() {
        super(...arguments);
        this.backends = new Array();
        this.listeners = new Array();
    }
    /**
     * Add a Virtual Services that this node is expected to send outbound traffic to
     */
    addBackends(...props) {
        for (const s of props) {
            this.backends.push({
                virtualService: {
                    virtualServiceName: s.virtualServiceName,
                },
            });
        }
    }
    /**
     * Utility method to add an inbound listener for this virtual node
     */
    addListeners(...listeners) {
        if (this.listeners.length + listeners.length > 1) {
            throw new Error('VirtualNode may have at most one listener');
        }
        for (const listener of listeners) {
            const portMapping = listener.portMapping || { port: 8080, protocol: shared_interfaces_1.Protocol.HTTP };
            this.listeners.push({
                portMapping,
                healthCheck: renderHealthCheck(listener.healthCheck, portMapping),
            });
        }
    }
}
/**
 * Minimum and maximum thresholds for HeathCheck numeric properties
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/APIReference/API_HealthCheckPolicy.html
 */
const HEALTH_CHECK_PROPERTY_THRESHOLDS = {
    healthyThreshold: [2, 10],
    intervalMillis: [5000, 300000],
    port: [1, 65535],
    timeoutMillis: [2000, 60000],
    unhealthyThreshold: [2, 10],
};
function renderHealthCheck(hc, pm) {
    if (hc === undefined) {
        return undefined;
    }
    if (hc.protocol === shared_interfaces_1.Protocol.TCP && hc.path) {
        throw new Error('The path property cannot be set with Protocol.TCP');
    }
    if (hc.protocol === shared_interfaces_1.Protocol.GRPC && hc.path) {
        throw new Error('The path property cannot be set with Protocol.GRPC');
    }
    const healthCheck = {
        healthyThreshold: hc.healthyThreshold || 2,
        intervalMillis: (hc.interval || cdk.Duration.seconds(5)).toMilliseconds(),
        path: hc.path || (hc.protocol === shared_interfaces_1.Protocol.HTTP ? '/' : undefined),
        port: hc.port || pm.port,
        protocol: hc.protocol || pm.protocol,
        timeoutMillis: (hc.timeout || cdk.Duration.seconds(2)).toMilliseconds(),
        unhealthyThreshold: hc.unhealthyThreshold || 2,
    };
    Object.keys(healthCheck)
        .filter((key) => HEALTH_CHECK_PROPERTY_THRESHOLDS[key] &&
        typeof healthCheck[key] === 'number' &&
        !cdk.Token.isUnresolved(healthCheck[key])).map((key) => {
        const [min, max] = HEALTH_CHECK_PROPERTY_THRESHOLDS[key];
        const value = healthCheck[key];
        if (value < min) {
            throw new Error(`The value of '${key}' is below the minimum threshold (expected >=${min}, got ${value})`);
        }
        if (value > max) {
            throw new Error(`The value of '${key}' is above the maximum threshold (expected <=${max}, got ${value})`);
        }
    });
    return healthCheck;
}
/**
 * VirtualNode represents a newly defined AppMesh VirtualNode
 *
 * Any inbound traffic that your virtual node expects should be specified as a
 * listener. Any outbound traffic that your virtual node expects to reach
 * should be specified as a backend.
 *
 * @see https://docs.aws.amazon.com/app-mesh/latest/userguide/virtual_nodes.html
 */
class VirtualNode extends VirtualNodeBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id, {
            physicalName: props.virtualNodeName || cdk.Lazy.stringValue({ produce: () => this.node.uniqueId }),
        });
        this.mesh = props.mesh;
        this.addBackends(...props.backends || []);
        this.addListeners(...props.listener ? [props.listener] : []);
        const accessLogging = (_a = props.accessLog) === null || _a === void 0 ? void 0 : _a.bind(this);
        const node = new appmesh_generated_1.CfnVirtualNode(this, 'Resource', {
            virtualNodeName: this.physicalName,
            meshName: this.mesh.meshName,
            spec: {
                backends: cdk.Lazy.anyValue({ produce: () => this.backends }, { omitEmptyArray: true }),
                listeners: cdk.Lazy.anyValue({ produce: () => this.listeners }, { omitEmptyArray: true }),
                serviceDiscovery: {
                    dns: props.dnsHostName !== undefined ? { hostname: props.dnsHostName } : undefined,
                    awsCloudMap: props.cloudMapService !== undefined ? {
                        serviceName: props.cloudMapService.serviceName,
                        namespaceName: props.cloudMapService.namespace.namespaceName,
                        attributes: renderAttributes(props.cloudMapServiceInstanceAttributes),
                    } : undefined,
                },
                logging: accessLogging !== undefined ? {
                    accessLog: accessLogging.virtualNodeAccessLog,
                } : undefined,
            },
        });
        this.virtualNodeName = this.getResourceNameAttribute(node.attrVirtualNodeName);
        this.virtualNodeArn = this.getResourceArnAttribute(node.ref, {
            service: 'appmesh',
            resource: `mesh/${props.mesh.meshName}/virtualNode`,
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing VirtualNode given an ARN
     */
    static fromVirtualNodeArn(scope, id, virtualNodeArn) {
        return new ImportedVirtualNode(scope, id, { virtualNodeArn });
    }
    /**
     * Import an existing VirtualNode given its name
     */
    static fromVirtualNodeName(scope, id, meshName, virtualNodeName) {
        return new ImportedVirtualNode(scope, id, {
            meshName,
            virtualNodeName,
        });
    }
}
exports.VirtualNode = VirtualNode;
function renderAttributes(attrs) {
    if (attrs === undefined) {
        return undefined;
    }
    return Object.entries(attrs).map(([key, value]) => ({ key, value }));
}
/**
 * Used to import a VirtualNode and read it's properties
 */
class ImportedVirtualNode extends VirtualNodeBase {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.virtualNodeArn) {
            this.virtualNodeArn = props.virtualNodeArn;
            this.virtualNodeName = cdk.Fn.select(2, cdk.Fn.split('/', cdk.Stack.of(scope).parseArn(props.virtualNodeArn).resourceName));
        }
        else if (props.virtualNodeName && props.meshName) {
            this.virtualNodeName = props.virtualNodeName;
            this.virtualNodeArn = cdk.Stack.of(this).formatArn({
                service: 'appmesh',
                resource: `mesh/${props.meshName}/virtualNode`,
                resourceName: this.virtualNodeName,
            });
        }
        else {
            throw new Error('Need either arn or both names');
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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