"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaDataSource = exports.HttpDataSource = exports.DynamoDbDataSource = exports.NoneDataSource = exports.BackedDataSource = exports.BaseDataSource = void 0;
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appsync_generated_1 = require("./appsync.generated");
const resolver_1 = require("./resolver");
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for concrete datasources
 */
class BaseDataSource extends core_1.Construct {
    constructor(scope, id, props, extended) {
        var _a, _b;
        super(scope, id);
        if (extended.type !== 'NONE') {
            this.serviceRole = props.serviceRole || new aws_iam_1.Role(this, 'ServiceRole', { assumedBy: new aws_iam_1.ServicePrincipal('appsync') });
        }
        const name = (_a = props.name) !== null && _a !== void 0 ? _a : id;
        this.ds = new appsync_generated_1.CfnDataSource(this, 'Resource', {
            apiId: props.api.apiId,
            name: name,
            description: props.description,
            serviceRoleArn: (_b = this.serviceRole) === null || _b === void 0 ? void 0 : _b.roleArn,
            ...extended,
        });
        this.name = name;
        this.api = props.api;
    }
    /**
     * creates a new resolver for this datasource and API using the given properties
     */
    createResolver(props) {
        return new resolver_1.Resolver(this, `${props.typeName}${props.fieldName}Resolver`, {
            api: this.api,
            dataSource: this,
            ...props,
        });
    }
}
exports.BaseDataSource = BaseDataSource;
/**
 * Abstract AppSync datasource implementation. Do not use directly but use subclasses for resource backed datasources
 */
class BackedDataSource extends BaseDataSource {
    constructor(scope, id, props, extended) {
        super(scope, id, props, extended);
        this.grantPrincipal = this.serviceRole;
    }
}
exports.BackedDataSource = BackedDataSource;
/**
 * An AppSync dummy datasource
 */
class NoneDataSource extends BaseDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'NONE',
        });
    }
}
exports.NoneDataSource = NoneDataSource;
/**
 * An AppSync datasource backed by a DynamoDB table
 */
class DynamoDbDataSource extends BackedDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AMAZON_DYNAMODB',
            dynamoDbConfig: {
                tableName: props.table.tableName,
                awsRegion: props.table.stack.region,
                useCallerCredentials: props.useCallerCredentials,
            },
        });
        if (props.readOnlyAccess) {
            props.table.grantReadData(this);
        }
        else {
            props.table.grantReadWriteData(this);
        }
    }
}
exports.DynamoDbDataSource = DynamoDbDataSource;
/**
 * An AppSync datasource backed by a http endpoint
 */
class HttpDataSource extends BaseDataSource {
    constructor(scope, id, props) {
        const authorizationConfig = props.authorizationConfig ? {
            authorizationType: 'AWS_IAM',
            awsIamConfig: props.authorizationConfig,
        } : undefined;
        super(scope, id, props, {
            type: 'HTTP',
            httpConfig: {
                endpoint: props.endpoint,
                authorizationConfig,
            },
        });
    }
}
exports.HttpDataSource = HttpDataSource;
/**
 * An AppSync datasource backed by a Lambda function
 */
class LambdaDataSource extends BackedDataSource {
    constructor(scope, id, props) {
        super(scope, id, props, {
            type: 'AWS_LAMBDA',
            lambdaConfig: {
                lambdaFunctionArn: props.lambdaFunction.functionArn,
            },
        });
        props.lambdaFunction.grantInvoke(this);
    }
}
exports.LambdaDataSource = LambdaDataSource;
//# sourceMappingURL=data:application/json;base64,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