"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.GraphqlApi = exports.IamResource = exports.FieldLogLevel = exports.UserPoolDefaultAction = exports.AuthorizationType = void 0;
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const appsync_generated_1 = require("./appsync.generated");
const graphqlapi_base_1 = require("./graphqlapi-base");
const schema_1 = require("./schema");
/**
 * enum with all possible values for AppSync authorization type
 */
var AuthorizationType;
(function (AuthorizationType) {
    /**
     * API Key authorization type
     */
    AuthorizationType["API_KEY"] = "API_KEY";
    /**
     * AWS IAM authorization type. Can be used with Cognito Identity Pool federated credentials
     */
    AuthorizationType["IAM"] = "AWS_IAM";
    /**
     * Cognito User Pool authorization type
     */
    AuthorizationType["USER_POOL"] = "AMAZON_COGNITO_USER_POOLS";
    /**
     * OpenID Connect authorization type
     */
    AuthorizationType["OIDC"] = "OPENID_CONNECT";
})(AuthorizationType = exports.AuthorizationType || (exports.AuthorizationType = {}));
/**
 * enum with all possible values for Cognito user-pool default actions
 */
var UserPoolDefaultAction;
(function (UserPoolDefaultAction) {
    /**
     * ALLOW access to API
     */
    UserPoolDefaultAction["ALLOW"] = "ALLOW";
    /**
     * DENY access to API
     */
    UserPoolDefaultAction["DENY"] = "DENY";
})(UserPoolDefaultAction = exports.UserPoolDefaultAction || (exports.UserPoolDefaultAction = {}));
/**
 * log-level for fields in AppSync
 */
var FieldLogLevel;
(function (FieldLogLevel) {
    /**
     * No logging
     */
    FieldLogLevel["NONE"] = "NONE";
    /**
     * Error logging
     */
    FieldLogLevel["ERROR"] = "ERROR";
    /**
     * All logging
     */
    FieldLogLevel["ALL"] = "ALL";
})(FieldLogLevel = exports.FieldLogLevel || (exports.FieldLogLevel = {}));
/**
 * A class used to generate resource arns for AppSync
 */
class IamResource {
    constructor(arns) {
        this.arns = arns;
    }
    /**
     * Generate the resource names given custom arns
     *
     * @param arns The custom arns that need to be permissioned
     *
     * Example: custom('/types/Query/fields/getExample')
     */
    static custom(...arns) {
        if (arns.length === 0) {
            throw new Error('At least 1 custom ARN must be provided.');
        }
        return new IamResource(arns);
    }
    /**
     * Generate the resource names given a type and fields
     *
     * @param type The type that needs to be allowed
     * @param fields The fields that need to be allowed, if empty grant permissions to ALL fields
     *
     * Example: ofType('Query', 'GetExample')
     */
    static ofType(type, ...fields) {
        const arns = fields.length ? fields.map((field) => `types/${type}/fields/${field}`) : [`types/${type}/*`];
        return new IamResource(arns);
    }
    /**
     * Generate the resource names that accepts all types: `*`
     */
    static all() {
        return new IamResource(['*']);
    }
    /**
     * Return the Resource ARN
     *
     * @param api The GraphQL API to give permissions
     */
    resourceArns(api) {
        return this.arns.map((arn) => core_1.Stack.of(api).formatArn({
            service: 'appsync',
            resource: `apis/${api.apiId}`,
            sep: '/',
            resourceName: `${arn}`,
        }));
    }
}
exports.IamResource = IamResource;
/**
 * An AppSync GraphQL API
 *
 * @resource AWS::AppSync::GraphQLApi
 */
class GraphqlApi extends graphqlapi_base_1.GraphqlApiBase {
    constructor(scope, id, props) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id);
        const defaultMode = (_b = (_a = props.authorizationConfig) === null || _a === void 0 ? void 0 : _a.defaultAuthorization) !== null && _b !== void 0 ? _b : { authorizationType: AuthorizationType.API_KEY };
        const additionalModes = (_d = (_c = props.authorizationConfig) === null || _c === void 0 ? void 0 : _c.additionalAuthorizationModes) !== null && _d !== void 0 ? _d : [];
        const modes = [defaultMode, ...additionalModes];
        this.modes = modes.map((mode) => mode.authorizationType);
        this.validateAuthorizationProps(modes);
        this.api = new appsync_generated_1.CfnGraphQLApi(this, 'Resource', {
            name: props.name,
            authenticationType: defaultMode.authorizationType,
            logConfig: this.setupLogConfig(props.logConfig),
            openIdConnectConfig: this.setupOpenIdConnectConfig(defaultMode.openIdConnectConfig),
            userPoolConfig: this.setupUserPoolConfig(defaultMode.userPoolConfig),
            additionalAuthenticationProviders: this.setupAdditionalAuthorizationModes(additionalModes),
            xrayEnabled: props.xrayEnabled,
        });
        this.apiId = this.api.attrApiId;
        this.arn = this.api.attrArn;
        this.graphqlUrl = this.api.attrGraphQlUrl;
        this.name = this.api.name;
        this.schema = (_e = props.schema) !== null && _e !== void 0 ? _e : new schema_1.Schema();
        this.schemaResource = this.schema.bind(this);
        if (modes.some((mode) => mode.authorizationType === AuthorizationType.API_KEY)) {
            const config = (_f = modes.find((mode) => {
                return mode.authorizationType === AuthorizationType.API_KEY && mode.apiKeyConfig;
            })) === null || _f === void 0 ? void 0 : _f.apiKeyConfig;
            this.apiKeyResource = this.createAPIKey(config);
            this.apiKeyResource.addDependsOn(this.schemaResource);
            this.apiKey = this.apiKeyResource.attrApiKey;
        }
    }
    /**
     * Import a GraphQL API through this function
     *
     * @param scope scope
     * @param id id
     * @param attrs GraphQL API Attributes of an API
     */
    static fromGraphqlApiAttributes(scope, id, attrs) {
        var _a;
        const arn = (_a = attrs.graphqlApiArn) !== null && _a !== void 0 ? _a : core_1.Stack.of(scope).formatArn({
            service: 'appsync',
            resource: `apis/${attrs.graphqlApiId}`,
        });
        class Import extends graphqlapi_base_1.GraphqlApiBase {
            constructor(s, i) {
                super(s, i);
                this.apiId = attrs.graphqlApiId;
                this.arn = arn;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Adds an IAM policy statement associated with this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param resources The set of resources to allow (i.e. ...:[region]:[accountId]:apis/GraphQLId/...)
     * @param actions The actions that should be granted to the principal (i.e. appsync:graphql )
     */
    grant(grantee, resources, ...actions) {
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions,
            resourceArns: resources.resourceArns(this),
            scope: this,
        });
    }
    /**
     * Adds an IAM policy statement for Mutation access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Mutations (leave blank for all)
     */
    grantMutation(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Mutation', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Query access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Queries (leave blank for all)
     */
    grantQuery(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Query', ...fields), 'appsync:GraphQL');
    }
    /**
     * Adds an IAM policy statement for Subscription access to this GraphQLApi to an IAM
     * principal's policy.
     *
     * @param grantee The principal
     * @param fields The fields to grant access to that are Subscriptions (leave blank for all)
     */
    grantSubscription(grantee, ...fields) {
        return this.grant(grantee, IamResource.ofType('Subscription', ...fields), 'appsync:GraphQL');
    }
    validateAuthorizationProps(modes) {
        modes.map((mode) => {
            if (mode.authorizationType === AuthorizationType.OIDC && !mode.openIdConnectConfig) {
                throw new Error('Missing default OIDC Configuration');
            }
            if (mode.authorizationType === AuthorizationType.USER_POOL && !mode.userPoolConfig) {
                throw new Error('Missing default OIDC Configuration');
            }
        });
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.API_KEY).length > 1) {
            throw new Error('You can\'t duplicate API_KEY configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
        if (modes.filter((mode) => mode.authorizationType === AuthorizationType.IAM).length > 1) {
            throw new Error('You can\'t duplicate IAM configuration. See https://docs.aws.amazon.com/appsync/latest/devguide/security.html');
        }
    }
    /**
     * Add schema dependency to a given construct
     *
     * @param construct the dependee
     */
    addSchemaDependency(construct) {
        construct.addDependsOn(this.schemaResource);
        return true;
    }
    setupLogConfig(config) {
        if (!config)
            return undefined;
        const role = new aws_iam_1.Role(this, 'ApiLogsRole', {
            assumedBy: new aws_iam_1.ServicePrincipal('appsync.amazonaws.com'),
            managedPolicies: [
                aws_iam_1.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSAppSyncPushToCloudWatchLogs'),
            ],
        });
        return {
            cloudWatchLogsRoleArn: role.roleArn,
            excludeVerboseContent: config.excludeVerboseContent,
            fieldLogLevel: config.fieldLogLevel,
        };
    }
    setupOpenIdConnectConfig(config) {
        if (!config)
            return undefined;
        return {
            authTtl: config.tokenExpiryFromAuth,
            clientId: config.clientId,
            iatTtl: config.tokenExpiryFromIssue,
            issuer: config.oidcProvider,
        };
    }
    setupUserPoolConfig(config) {
        if (!config)
            return undefined;
        return {
            userPoolId: config.userPool.userPoolId,
            awsRegion: config.userPool.stack.region,
            appIdClientRegex: config.appIdClientRegex,
            defaultAction: config.defaultAction || UserPoolDefaultAction.ALLOW,
        };
    }
    setupAdditionalAuthorizationModes(modes) {
        if (!modes || modes.length === 0)
            return undefined;
        return modes.reduce((acc, mode) => [
            ...acc, {
                authenticationType: mode.authorizationType,
                userPoolConfig: this.setupUserPoolConfig(mode.userPoolConfig),
                openIdConnectConfig: this.setupOpenIdConnectConfig(mode.openIdConnectConfig),
            },
        ], []);
    }
    createAPIKey(config) {
        var _a, _b;
        if (((_a = config === null || config === void 0 ? void 0 : config.expires) === null || _a === void 0 ? void 0 : _a.isBefore(core_1.Duration.days(1))) || ((_b = config === null || config === void 0 ? void 0 : config.expires) === null || _b === void 0 ? void 0 : _b.isAfter(core_1.Duration.days(365)))) {
            throw Error('API key expiration must be between 1 and 365 days.');
        }
        const expires = (config === null || config === void 0 ? void 0 : config.expires) ? config === null || config === void 0 ? void 0 : config.expires.toEpoch() : undefined;
        return new appsync_generated_1.CfnApiKey(this, `${(config === null || config === void 0 ? void 0 : config.name) || 'Default'}ApiKey`, {
            expires,
            description: config === null || config === void 0 ? void 0 : config.description,
            apiId: this.apiId,
        });
    }
    /**
     * Escape hatch to append to Schema as desired. Will always result
     * in a newline.
     *
     * @param addition the addition to add to schema
     * @param delimiter the delimiter between schema and addition
     * @default - ''
     *
     * @experimental
     */
    addToSchema(addition, delimiter) {
        this.schema.addToSchema(addition, delimiter);
    }
    /**
     * Add type to the schema
     *
     * @param type the intermediate type to add to the schema
     *
     * @experimental
     */
    addType(type) {
        return this.schema.addType(type);
    }
    /**
     * Add a query field to the schema's Query. CDK will create an
     * Object Type called 'Query'. For example,
     *
     * type Query {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the query
     * @param field the resolvable field to for this query
     */
    addQuery(fieldName, field) {
        return this.schema.addQuery(fieldName, field);
    }
    /**
     * Add a mutation field to the schema's Mutation. CDK will create an
     * Object Type called 'Mutation'. For example,
     *
     * type Mutation {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Mutation
     * @param field the resolvable field to for this Mutation
     */
    addMutation(fieldName, field) {
        return this.schema.addMutation(fieldName, field);
    }
    /**
     * Add a subscription field to the schema's Subscription. CDK will create an
     * Object Type called 'Subscription'. For example,
     *
     * type Subscription {
     *   fieldName: Field.returnType
     * }
     *
     * @param fieldName the name of the Subscription
     * @param field the resolvable field to for this Subscription
     */
    addSubscription(fieldName, field) {
        return this.schema.addSubscription(fieldName, field);
    }
}
exports.GraphqlApi = GraphqlApi;
//# sourceMappingURL=data:application/json;base64,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