"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.MappingTemplate = void 0;
const fs_1 = require("fs");
/**
 * MappingTemplates for AppSync resolvers
 */
class MappingTemplate {
    /**
     * Create a mapping template from the given string
     */
    static fromString(template) {
        return new StringMappingTemplate(template);
    }
    /**
     * Create a mapping template from the given file
     */
    static fromFile(fileName) {
        return new StringMappingTemplate(fs_1.readFileSync(fileName).toString('utf-8'));
    }
    /**
     * Mapping template for a result list from DynamoDB
     */
    static dynamoDbResultList() {
        return this.fromString('$util.toJson($ctx.result.items)');
    }
    /**
     * Mapping template for a single result item from DynamoDB
     */
    static dynamoDbResultItem() {
        return this.fromString('$util.toJson($ctx.result)');
    }
    /**
     * Mapping template to scan a DynamoDB table to fetch all entries
     */
    static dynamoDbScanTable() {
        return this.fromString('{"version" : "2017-02-28", "operation" : "Scan"}');
    }
    /**
     * Mapping template to query a set of items from a DynamoDB table
     *
     * @param cond the key condition for the query
     */
    static dynamoDbQuery(cond) {
        return this.fromString(`{"version" : "2017-02-28", "operation" : "Query", ${cond.renderTemplate()}}`);
    }
    /**
     * Mapping template to get a single item from a DynamoDB table
     *
     * @param keyName the name of the hash key field
     * @param idArg the name of the Query argument
     */
    static dynamoDbGetItem(keyName, idArg) {
        return this.fromString(`{"version": "2017-02-28", "operation": "GetItem", "key": {"${keyName}": $util.dynamodb.toDynamoDBJson($ctx.args.${idArg})}}`);
    }
    /**
     * Mapping template to delete a single item from a DynamoDB table
     *
     * @param keyName the name of the hash key field
     * @param idArg the name of the Mutation argument
     */
    static dynamoDbDeleteItem(keyName, idArg) {
        return this.fromString(`{"version": "2017-02-28", "operation": "DeleteItem", "key": {"${keyName}": $util.dynamodb.toDynamoDBJson($ctx.args.${idArg})}}`);
    }
    /**
     * Mapping template to save a single item to a DynamoDB table
     *
     * @param key the assigment of Mutation values to the primary key
     * @param values the assignment of Mutation values to the table attributes
     */
    static dynamoDbPutItem(key, values) {
        return this.fromString(`
      ${values.renderVariables()}
      {
        "version": "2017-02-28",
        "operation": "PutItem",
        ${key.renderTemplate()},
        ${values.renderTemplate()}
      }`);
    }
    /**
     * Mapping template to invoke a Lambda function
     *
     * @param payload the VTL template snippet of the payload to send to the lambda.
     * If no payload is provided all available context fields are sent to the Lambda function
     */
    static lambdaRequest(payload = '$util.toJson($ctx)') {
        return this.fromString(`{"version": "2017-02-28", "operation": "Invoke", "payload": ${payload}}`);
    }
    /**
     * Mapping template to return the Lambda result to the caller
     */
    static lambdaResult() {
        return this.fromString('$util.toJson($ctx.result)');
    }
}
exports.MappingTemplate = MappingTemplate;
class StringMappingTemplate extends MappingTemplate {
    constructor(template) {
        super();
        this.template = template;
    }
    renderTemplate() {
        return this.template;
    }
}
//# sourceMappingURL=data:application/json;base64,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