"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ResolvableField = exports.Field = exports.GraphqlType = void 0;
const schema_base_1 = require("./schema-base");
/**
 * The GraphQL Types in AppSync's GraphQL. GraphQL Types are the
 * building blocks for object types, queries, mutations, etc. They are
 * types like String, Int, Id or even Object Types you create.
 *
 * i.e. `String`, `String!`, `[String]`, `[String!]`, `[String]!`
 *
 * GraphQL Types are used to define the entirety of schema.
 */
class GraphqlType {
    constructor(type, options) {
        var _a, _b, _c;
        this.type = type;
        this.isList = (_a = options === null || options === void 0 ? void 0 : options.isList) !== null && _a !== void 0 ? _a : false;
        this.isRequired = (_b = options === null || options === void 0 ? void 0 : options.isRequired) !== null && _b !== void 0 ? _b : false;
        this.isRequiredList = (_c = options === null || options === void 0 ? void 0 : options.isRequiredList) !== null && _c !== void 0 ? _c : false;
        this.intermediateType = options === null || options === void 0 ? void 0 : options.intermediateType;
    }
    /**
     * `ID` scalar type is a unique identifier. `ID` type is serialized similar to `String`.
     *
     * Often used as a key for a cache and not intended to be human-readable.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static id(options) {
        return new GraphqlType(schema_base_1.Type.ID, options);
    }
    /**
     * `String` scalar type is a free-form human-readable text.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static string(options) {
        return new GraphqlType(schema_base_1.Type.STRING, options);
    }
    /**
     * `Int` scalar type is a signed non-fractional numerical value.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static int(options) {
        return new GraphqlType(schema_base_1.Type.INT, options);
    }
    /**
     * `Float` scalar type is a signed double-precision fractional value.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static float(options) {
        return new GraphqlType(schema_base_1.Type.FLOAT, options);
    }
    /**
     * `Boolean` scalar type is a boolean value: true or false.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static boolean(options) {
        return new GraphqlType(schema_base_1.Type.BOOLEAN, options);
    }
    /**
     * `AWSDate` scalar type represents a valid extended `ISO 8601 Date` string.
     *
     * In other words, accepts date strings in the form of `YYYY-MM-DD`. It accepts time zone offsets.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsDate(options) {
        return new GraphqlType(schema_base_1.Type.AWS_DATE, options);
    }
    /**
     * `AWSTime` scalar type represents a valid extended `ISO 8601 Time` string.
     *
     * In other words, accepts date strings in the form of `hh:mm:ss.sss`. It accepts time zone offsets.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsTime(options) {
        return new GraphqlType(schema_base_1.Type.AWS_TIME, options);
    }
    /**
     * `AWSDateTime` scalar type represents a valid extended `ISO 8601 DateTime` string.
     *
     * In other words, accepts date strings in the form of `YYYY-MM-DDThh:mm:ss.sssZ`. It accepts time zone offsets.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsDateTime(options) {
        return new GraphqlType(schema_base_1.Type.AWS_DATE_TIME, options);
    }
    /**
     * `AWSTimestamp` scalar type represents the number of seconds since `1970-01-01T00:00Z`.
     *
     * Timestamps are serialized and deserialized as numbers.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsTimestamp(options) {
        return new GraphqlType(schema_base_1.Type.AWS_TIMESTAMP, options);
    }
    /**
     * `AWSEmail` scalar type represents an email address string (i.e.`username@example.com`)
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsEmail(options) {
        return new GraphqlType(schema_base_1.Type.AWS_EMAIL, options);
    }
    /**
     * `AWSJson` scalar type represents a JSON string.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsJson(options) {
        return new GraphqlType(schema_base_1.Type.AWS_JSON, options);
    }
    /**
     * `AWSURL` scalar type represetns a valid URL string.
     *
     * URLs wihtout schemes or contain double slashes are considered invalid.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsUrl(options) {
        return new GraphqlType(schema_base_1.Type.AWS_URL, options);
    }
    /**
     * `AWSPhone` scalar type represents a valid phone number. Phone numbers maybe be whitespace delimited or hyphenated.
     *
     * The number can specify a country code at the beginning, but is not required for US phone numbers.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsPhone(options) {
        return new GraphqlType(schema_base_1.Type.AWS_PHONE, options);
    }
    /**
     * `AWSIPAddress` scalar type respresents a valid `IPv4` of `IPv6` address string.
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     */
    static awsIpAddress(options) {
        return new GraphqlType(schema_base_1.Type.AWS_IP_ADDRESS, options);
    }
    /**
     * an intermediate type to be added as an attribute
     * (i.e. an interface or an object type)
     *
     * @param options the options to configure this attribute
     * - isList
     * - isRequired
     * - isRequiredList
     * - intermediateType
     */
    static intermediate(options) {
        if (!(options === null || options === void 0 ? void 0 : options.intermediateType)) {
            throw new Error('GraphQL Type of interface must be configured with corresponding Intermediate Type');
        }
        return new GraphqlType(schema_base_1.Type.INTERMEDIATE, options);
    }
    /**
     * Generate the string for this attribute
     */
    toString() {
        var _a;
        // If an Object Type, we use the name of the Object Type
        let type = this.intermediateType ? (_a = this.intermediateType) === null || _a === void 0 ? void 0 : _a.name : this.type;
        // If configured as required, the GraphQL Type becomes required
        type = this.isRequired ? `${type}!` : type;
        // If configured with isXxxList, the GraphQL Type becomes a list
        type = this.isList || this.isRequiredList ? `[${type}]` : type;
        // If configured with isRequiredList, the list becomes required
        type = this.isRequiredList ? `${type}!` : type;
        return type;
    }
    /**
     * Generate the arguments for this field
     */
    argsToString() {
        return '';
    }
    /**
     * Generate the directives for this field
     */
    directivesToString(_modes) {
        return '';
    }
}
exports.GraphqlType = GraphqlType;
/**
 * Fields build upon Graphql Types and provide typing
 * and arguments.
 */
class Field extends GraphqlType {
    constructor(options) {
        const props = {
            isList: options.returnType.isList,
            isRequired: options.returnType.isRequired,
            isRequiredList: options.returnType.isRequiredList,
            intermediateType: options.returnType.intermediateType,
        };
        super(options.returnType.type, props);
        this.fieldOptions = options;
    }
    /**
     * Generate the args string of this resolvable field
     */
    argsToString() {
        if (!this.fieldOptions || !this.fieldOptions.args) {
            return '';
        }
        return Object.keys(this.fieldOptions.args).reduce((acc, key) => { var _a, _b; return `${acc}${key}: ${(_b = (_a = this.fieldOptions) === null || _a === void 0 ? void 0 : _a.args) === null || _b === void 0 ? void 0 : _b[key].toString()} `; }, '(').slice(0, -1) + ')';
    }
    /**
     * Generate the directives for this field
     */
    directivesToString(modes) {
        if (!this.fieldOptions || !this.fieldOptions.directives) {
            return '';
        }
        return this.fieldOptions.directives.reduce((acc, directive) => `${acc}${directive._bindToAuthModes(modes).toString()} `, '\n  ').slice(0, -1);
    }
}
exports.Field = Field;
/**
 * Resolvable Fields build upon Graphql Types and provide fields
 * that can resolve into operations on a data source.
 */
class ResolvableField extends Field {
    constructor(options) {
        const props = {
            returnType: options.returnType,
            args: options.args,
        };
        super(props);
        this.fieldOptions = options;
    }
}
exports.ResolvableField = ResolvableField;
//# sourceMappingURL=data:application/json;base64,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