"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.EnumType = exports.UnionType = exports.InputType = exports.ObjectType = exports.InterfaceType = void 0;
const private_1 = require("./private");
const schema_field_1 = require("./schema-field");
/**
 * Interface Types are abstract types that includes a certain set of fields
 * that other types must include if they implement the interface.
 *
 * @experimental
 */
class InterfaceType {
    constructor(name, props) {
        this.name = name;
        this.definition = props.definition;
        this.directives = props.directives;
    }
    /**
     * Create a GraphQL Type representing this Intermediate Type
     *
     * @param options the options to configure this attribute
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * Generate the string of this object type
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'interface',
            name: this.name,
            directives: this.directives,
            fields: Object.keys(this.definition).map((key) => {
                const field = this.definition[key];
                return `${key}${field.argsToString()}: ${field.toString()}${field.directivesToString(this.modes)}`;
            }),
            modes: this.modes,
        });
    }
    /**
     * Add a field to this Interface Type.
     *
     * Interface Types must have both fieldName and field options.
     *
     * @param options the options to add a field
     */
    addField(options) {
        if (!options.fieldName || !options.field) {
            throw new Error('Interface Types must have both fieldName and field options.');
        }
        this.definition[options.fieldName] = options.field;
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.InterfaceType = InterfaceType;
/**
 * Object Types are types declared by you.
 *
 * @experimental
 */
class ObjectType extends InterfaceType {
    constructor(name, props) {
        var _a, _b;
        const options = {
            definition: (_b = (_a = props.interfaceTypes) === null || _a === void 0 ? void 0 : _a.reduce((def, interfaceType) => {
                return Object.assign({}, def, interfaceType.definition);
            }, props.definition)) !== null && _b !== void 0 ? _b : props.definition,
            directives: props.directives,
        };
        super(name, options);
        this.interfaceTypes = props.interfaceTypes;
        this.resolvers = [];
        Object.keys(this.definition).forEach((fieldName) => {
            const field = this.definition[fieldName];
            this.generateResolver(fieldName, field.fieldOptions);
        });
    }
    /**
     * Add a field to this Object Type.
     *
     * Object Types must have both fieldName and field options.
     *
     * @param options the options to add a field
     */
    addField(options) {
        if (!options.fieldName || !options.field) {
            throw new Error('Object Types must have both fieldName and field options.');
        }
        this.generateResolver(options.fieldName, options.field.fieldOptions);
        this.definition[options.fieldName] = options.field;
    }
    /**
     * Generate the string of this object type
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'type',
            name: this.name,
            interfaceTypes: this.interfaceTypes,
            directives: this.directives,
            fields: Object.keys(this.definition).map((key) => {
                const field = this.definition[key];
                return `${key}${field.argsToString()}: ${field.toString()}${field.directivesToString(this.modes)}`;
            }),
            modes: this.modes,
        });
    }
    /**
     * Generate the resolvers linked to this Object Type
     */
    generateResolver(fieldName, options) {
        if (!(options === null || options === void 0 ? void 0 : options.dataSource))
            return;
        if (!this.resolvers) {
            this.resolvers = [];
        }
        this.resolvers.push(options.dataSource.createResolver({
            typeName: this.name,
            fieldName: fieldName,
            pipelineConfig: options.pipelineConfig,
            requestMappingTemplate: options.requestMappingTemplate,
            responseMappingTemplate: options.responseMappingTemplate,
        }));
    }
}
exports.ObjectType = ObjectType;
/**
 * Input Types are abstract types that define complex objects.
 * They are used in arguments to represent
 *
 * @experimental
 */
class InputType {
    constructor(name, props) {
        this.name = name;
        this.definition = props.definition;
    }
    /**
     * Create a GraphQL Type representing this Input Type
     *
     * @param options the options to configure this attribute
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * Generate the string of this input type
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'input',
            name: this.name,
            fields: Object.keys(this.definition).map((key) => `${key}${this.definition[key].argsToString()}: ${this.definition[key].toString()}`),
            modes: this.modes,
        });
    }
    /**
     * Add a field to this Input Type.
     *
     * Input Types must have both fieldName and field options.
     *
     * @param options the options to add a field
     */
    addField(options) {
        if (!options.fieldName || !options.field) {
            throw new Error('Input Types must have both fieldName and field options.');
        }
        this.definition[options.fieldName] = options.field;
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.InputType = InputType;
/**
 * Union Types are abstract types that are similar to Interface Types,
 * but they cannot to specify any common fields between types.
 *
 * Note that fields of a union type need to be object types. In other words,
 * you can't create a union type out of interfaces, other unions, or inputs.
 *
 * @experimental
 */
class UnionType {
    constructor(name, options) {
        this.name = name;
        this.definition = {};
        options.definition.map((def) => this.addField({ field: def.attribute() }));
    }
    /**
     * Create a GraphQL Type representing this Union Type
     *
     * @param options the options to configure this attribute
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * Generate the string of this Union type
     */
    toString() {
        // Return a string that appends all Object Types for this Union Type
        // i.e. 'union Example = example1 | example2'
        return Object.values(this.definition).reduce((acc, field) => `${acc} ${field.toString()} |`, `union ${this.name} =`).slice(0, -2);
    }
    /**
     * Add a field to this Union Type
     *
     * Input Types must have field options and the IField must be an Object Type.
     *
     * @param options the options to add a field
     */
    addField(options) {
        var _a;
        if (options.fieldName) {
            throw new Error('Union Types cannot be configured with the fieldName option. Use the field option instead.');
        }
        if (!options.field) {
            throw new Error('Union Types must be configured with the field option.');
        }
        if (options.field && !(options.field.intermediateType instanceof ObjectType)) {
            throw new Error('Fields for Union Types must be Object Types.');
        }
        this.definition[((_a = options.field) === null || _a === void 0 ? void 0 : _a.toString()) + 'id'] = options.field;
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.UnionType = UnionType;
/**
 * Enum Types are abstract types that includes a set of fields
 * that represent the strings this type can create.
 *
 * @experimental
 */
class EnumType {
    constructor(name, options) {
        this.name = name;
        this.definition = {};
        options.definition.map((fieldName) => this.addField({ fieldName }));
    }
    /**
     * Create an GraphQL Type representing this Enum Type
     */
    attribute(options) {
        return schema_field_1.GraphqlType.intermediate({
            isList: options === null || options === void 0 ? void 0 : options.isList,
            isRequired: options === null || options === void 0 ? void 0 : options.isRequired,
            isRequiredList: options === null || options === void 0 ? void 0 : options.isRequiredList,
            intermediateType: this,
        });
    }
    /**
     * Generate the string of this enum type
     */
    toString() {
        return private_1.shapeAddition({
            prefix: 'enum',
            name: this.name,
            fields: Object.keys(this.definition),
            modes: this.modes,
        });
    }
    /**
     * Add a field to this Enum Type
     *
     * To add a field to this Enum Type, you must only configure
     * addField with the fieldName options.
     *
     * @param options the options to add a field
     */
    addField(options) {
        if (options.field) {
            throw new Error('Enum Type fields consist of strings. Use the fieldName option instead of the field option.');
        }
        if (!options.fieldName) {
            throw new Error('When adding a field to an Enum Type, you must configure the fieldName option.');
        }
        if (options.fieldName.indexOf(' ') > -1) {
            throw new Error(`Enum Type values cannot have whitespace. Received: ${options.fieldName}`);
        }
        this.definition[options.fieldName] = schema_field_1.GraphqlType.string();
    }
    /**
     * Method called when the stringifying Intermediate Types for schema generation
     *
     * @internal
     */
    _bindToGraphqlApi(api) {
        this.modes = api.modes;
        return this;
    }
}
exports.EnumType = EnumType;
//# sourceMappingURL=data:application/json;base64,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