"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.StepScalingPolicy = void 0;
const aws_autoscaling_common_1 = require("../../aws-autoscaling-common"); // Automatically re-written from '@aws-cdk/aws-autoscaling-common'
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const step_scaling_action_1 = require("./step-scaling-action");
/**
 * Define a acaling strategy which scales depending on absolute values of some metric.
 *
 * You can specify the scaling behavior for various values of the metric.
 *
 * Implemented using one or more CloudWatch alarms and Step Scaling Policies.
 */
class StepScalingPolicy extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        if (props.scalingSteps.length < 2) {
            throw new Error('You must supply at least 2 intervals for autoscaling');
        }
        const adjustmentType = props.adjustmentType || step_scaling_action_1.AdjustmentType.CHANGE_IN_CAPACITY;
        const changesAreAbsolute = adjustmentType === step_scaling_action_1.AdjustmentType.EXACT_CAPACITY;
        const intervals = aws_autoscaling_common_1.normalizeIntervals(props.scalingSteps, changesAreAbsolute);
        const alarms = aws_autoscaling_common_1.findAlarmThresholds(intervals);
        if (alarms.lowerAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.lowerAlarmIntervalIndex].upper;
            this.lowerAction = new step_scaling_action_1.StepScalingAction(this, 'LowerPolicy', {
                adjustmentType: props.adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                autoScalingGroup: props.autoScalingGroup,
            });
            for (let i = alarms.lowerAlarmIntervalIndex; i >= 0; i--) {
                this.lowerAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: i !== 0 ? intervals[i].lower - threshold : undefined,
                    upperBound: intervals[i].upper - threshold,
                });
            }
            this.lowerAlarm = new cloudwatch.Alarm(this, 'LowerAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Lower threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.LESS_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: 1,
                threshold,
            });
            this.lowerAlarm.addAlarmAction(new StepScalingAlarmAction(this.lowerAction));
        }
        if (alarms.upperAlarmIntervalIndex !== undefined) {
            const threshold = intervals[alarms.upperAlarmIntervalIndex].lower;
            this.upperAction = new step_scaling_action_1.StepScalingAction(this, 'UpperPolicy', {
                adjustmentType: props.adjustmentType,
                cooldown: props.cooldown,
                metricAggregationType: aggregationTypeFromMetric(props.metric),
                minAdjustmentMagnitude: props.minAdjustmentMagnitude,
                autoScalingGroup: props.autoScalingGroup,
            });
            for (let i = alarms.upperAlarmIntervalIndex; i < intervals.length; i++) {
                this.upperAction.addAdjustment({
                    adjustment: intervals[i].change,
                    lowerBound: intervals[i].lower - threshold,
                    upperBound: i !== intervals.length - 1 ? intervals[i].upper - threshold : undefined,
                });
            }
            this.upperAlarm = new cloudwatch.Alarm(this, 'UpperAlarm', {
                // Recommended by AutoScaling
                metric: props.metric,
                alarmDescription: 'Upper threshold scaling alarm',
                comparisonOperator: cloudwatch.ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD,
                evaluationPeriods: 1,
                threshold,
            });
            this.upperAlarm.addAlarmAction(new StepScalingAlarmAction(this.upperAction));
        }
    }
}
exports.StepScalingPolicy = StepScalingPolicy;
function aggregationTypeFromMetric(metric) {
    var _a;
    const statistic = (_a = metric.toMetricConfig().metricStat) === null || _a === void 0 ? void 0 : _a.statistic;
    if (statistic === undefined) {
        return undefined;
    } // Math expression, don't know aggregation, leave default
    switch (statistic) {
        case 'Average':
            return step_scaling_action_1.MetricAggregationType.AVERAGE;
        case 'Minimum':
            return step_scaling_action_1.MetricAggregationType.MINIMUM;
        case 'Maximum':
            return step_scaling_action_1.MetricAggregationType.MAXIMUM;
        default:
            throw new Error(`Cannot only scale on 'Minimum', 'Maximum', 'Average' metrics, got ${statistic}`);
    }
}
/**
 * Use a StepScalingAction as an Alarm Action
 *
 * This class is here and not in aws-cloudwatch-actions because this library
 * needs to use the class, and otherwise we'd have a circular dependency:
 *
 * aws-autoscaling -> aws-cloudwatch-actions (for using the Action)
 * aws-cloudwatch-actions -> aws-autoscaling (for the definition of IStepScalingAction)
 */
class StepScalingAlarmAction {
    constructor(stepScalingAction) {
        this.stepScalingAction = stepScalingAction;
    }
    bind(_scope, _alarm) {
        return { alarmActionArn: this.stepScalingAction.scalingPolicyArn };
    }
}
//# sourceMappingURL=data:application/json;base64,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