"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupPlanRule = void 0;
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * A backup plan rule
 */
class BackupPlanRule {
    /** @param props Rule properties */
    constructor(props) {
        this.props = props;
        if (props.deleteAfter && props.moveToColdStorageAfter &&
            props.deleteAfter.toSeconds() < props.moveToColdStorageAfter.toSeconds()) {
            throw new Error('`deleteAfter` must be greater than `moveToColdStorageAfter`');
        }
        if (props.scheduleExpression && !/^cron/.test(props.scheduleExpression.expressionString)) {
            throw new Error('`scheduleExpression` must be of type `cron`');
        }
    }
    /**
     * Daily with 35 days retention
     */
    static daily(backupVault) {
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Daily',
            scheduleExpression: events.Schedule.cron({
                hour: '5',
                minute: '0',
            }),
            deleteAfter: core_1.Duration.days(35),
        });
    }
    /**
     * Weekly with 3 months retention
     */
    static weekly(backupVault) {
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Weekly',
            scheduleExpression: events.Schedule.cron({
                hour: '5',
                minute: '0',
                weekDay: 'SAT',
            }),
            deleteAfter: core_1.Duration.days(30 * 3),
        });
    }
    /**
     * Monthly 1 year retention, move to cold storage after 1 month
     */
    static monthly1Year(backupVault) {
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Monthly1Year',
            scheduleExpression: events.Schedule.cron({
                day: '1',
                hour: '5',
                minute: '0',
            }),
            moveToColdStorageAfter: core_1.Duration.days(30),
            deleteAfter: core_1.Duration.days(365),
        });
    }
    /**
     * Monthly 5 year retention, move to cold storage after 3 months
     */
    static monthly5Year(backupVault) {
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Monthly5Year',
            scheduleExpression: events.Schedule.cron({
                day: '1',
                hour: '5',
                minute: '0',
            }),
            moveToColdStorageAfter: core_1.Duration.days(30 * 3),
            deleteAfter: core_1.Duration.days(365 * 5),
        });
    }
    /**
     * Monthly 7 year retention, move to cold storage after 3 months
     */
    static monthly7Year(backupVault) {
        return new BackupPlanRule({
            backupVault,
            ruleName: 'Monthly7Year',
            scheduleExpression: events.Schedule.cron({
                day: '1',
                hour: '5',
                minute: '0',
            }),
            moveToColdStorageAfter: core_1.Duration.days(30 * 3),
            deleteAfter: core_1.Duration.days(365 * 7),
        });
    }
}
exports.BackupPlanRule = BackupPlanRule;
//# sourceMappingURL=data:application/json;base64,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