"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.BackupVault = exports.BackupVaultEvents = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const backup_generated_1 = require("./backup.generated");
/**
 * Backup vault events
 */
var BackupVaultEvents;
(function (BackupVaultEvents) {
    /** BACKUP_JOB_STARTED */
    BackupVaultEvents["BACKUP_JOB_STARTED"] = "BACKUP_JOB_STARTED";
    /** BACKUP_JOB_COMPLETED */
    BackupVaultEvents["BACKUP_JOB_COMPLETED"] = "BACKUP_JOB_COMPLETED";
    /** BACKUP_JOB_SUCCESSFUL */
    BackupVaultEvents["BACKUP_JOB_SUCCESSFUL"] = "BACKUP_JOB_SUCCESSFUL";
    /** BACKUP_JOB_FAILED */
    BackupVaultEvents["BACKUP_JOB_FAILED"] = "BACKUP_JOB_FAILED";
    /** BACKUP_JOB_EXPIRED */
    BackupVaultEvents["BACKUP_JOB_EXPIRED"] = "BACKUP_JOB_EXPIRED";
    /** RESTORE_JOB_STARTED */
    BackupVaultEvents["RESTORE_JOB_STARTED"] = "RESTORE_JOB_STARTED";
    /** RESTORE_JOB_COMPLETED */
    BackupVaultEvents["RESTORE_JOB_COMPLETED"] = "RESTORE_JOB_COMPLETED";
    /** RESTORE_JOB_SUCCESSFUL */
    BackupVaultEvents["RESTORE_JOB_SUCCESSFUL"] = "RESTORE_JOB_SUCCESSFUL";
    /** RESTORE_JOB_FAILED */
    BackupVaultEvents["RESTORE_JOB_FAILED"] = "RESTORE_JOB_FAILED";
    /** COPY_JOB_STARTED */
    BackupVaultEvents["COPY_JOB_STARTED"] = "COPY_JOB_STARTED";
    /** COPY_JOB_SUCCESSFUL */
    BackupVaultEvents["COPY_JOB_SUCCESSFUL"] = "COPY_JOB_SUCCESSFUL";
    /** COPY_JOB_FAILED */
    BackupVaultEvents["COPY_JOB_FAILED"] = "COPY_JOB_FAILED";
    /** RECOVERY_POINT_MODIFIED */
    BackupVaultEvents["RECOVERY_POINT_MODIFIED"] = "RECOVERY_POINT_MODIFIED";
    /** BACKUP_PLAN_CREATED */
    BackupVaultEvents["BACKUP_PLAN_CREATED"] = "BACKUP_PLAN_CREATED";
    /** BACKUP_PLAN_MODIFIED */
    BackupVaultEvents["BACKUP_PLAN_MODIFIED"] = "BACKUP_PLAN_MODIFIED";
})(BackupVaultEvents = exports.BackupVaultEvents || (exports.BackupVaultEvents = {}));
/**
 * A backup vault
 */
class BackupVault extends core_1.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        if (props.backupVaultName && !/^[a-zA-Z0-9\-_]{2,50}$/.test(props.backupVaultName)) {
            throw new Error('Expected vault name to match pattern `^[a-zA-Z0-9\-_]{2,50}$`');
        }
        let notifications;
        if (props.notificationTopic) {
            notifications = {
                backupVaultEvents: props.notificationEvents || Object.values(BackupVaultEvents),
                snsTopicArn: props.notificationTopic.topicArn,
            };
            props.notificationTopic.grantPublish(new iam.ServicePrincipal('backup.amazonaws.com'));
        }
        const vault = new backup_generated_1.CfnBackupVault(this, 'Resource', {
            backupVaultName: props.backupVaultName || this.uniqueVaultName(),
            accessPolicy: props.accessPolicy && props.accessPolicy.toJSON(),
            encryptionKeyArn: props.encryptionKey && props.encryptionKey.keyArn,
            notifications,
        });
        vault.applyRemovalPolicy(props.removalPolicy);
        this.backupVaultName = vault.attrBackupVaultName;
        this.backupVaultArn = vault.attrBackupVaultArn;
    }
    /**
     * Import an existing backup vault
     */
    static fromBackupVaultName(scope, id, backupVaultName) {
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.backupVaultName = backupVaultName;
            }
        }
        return new Import(scope, id);
    }
    uniqueVaultName() {
        // Max length of 50 chars, get the last 50 chars
        const id = this.node.uniqueId;
        return id.substring(Math.max(id.length - 50, 0), id.length);
    }
}
exports.BackupVault = BackupVault;
//# sourceMappingURL=data:application/json;base64,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