"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobDefinition = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const batch_generated_1 = require("./batch.generated");
const job_definition_image_config_1 = require("./job-definition-image-config");
/**
 * Batch Job Definition.
 *
 * Defines a batch job definition to execute a specific batch job.
 */
class JobDefinition extends core_1.Resource {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.jobDefinitionName,
        });
        this.imageConfig = new job_definition_image_config_1.JobDefinitionImageConfig(this, props.container);
        const jobDef = new batch_generated_1.CfnJobDefinition(this, 'Resource', {
            jobDefinitionName: props.jobDefinitionName,
            containerProperties: this.buildJobContainer(props.container),
            type: 'container',
            nodeProperties: props.nodeProps
                ? {
                    mainNode: props.nodeProps.mainNode,
                    nodeRangeProperties: this.buildNodeRangeProps(props.nodeProps),
                    numNodes: props.nodeProps.count,
                }
                : undefined,
            parameters: props.parameters,
            retryStrategy: {
                attempts: props.retryAttempts || 1,
            },
            timeout: {
                attemptDurationSeconds: props.timeout ? props.timeout.toSeconds() : undefined,
            },
        });
        this.jobDefinitionArn = this.getResourceArnAttribute(jobDef.ref, {
            service: 'batch',
            resource: 'job-definition',
            resourceName: this.physicalName,
        });
        this.jobDefinitionName = this.getResourceNameAttribute(jobDef.ref);
    }
    /**
     * Imports an existing batch job definition by its amazon resource name.
     *
     * @param scope
     * @param id
     * @param jobDefinitionArn
     */
    static fromJobDefinitionArn(scope, id, jobDefinitionArn) {
        const stack = core_1.Stack.of(scope);
        const jobDefName = stack.parseArn(jobDefinitionArn).resourceName;
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefinitionArn;
                this.jobDefinitionName = jobDefName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Imports an existing batch job definition by its name.
     * If name is specified without a revision then the latest active revision is used.
     *
     * @param scope
     * @param id
     * @param jobDefinitionName
     */
    static fromJobDefinitionName(scope, id, jobDefinitionName) {
        const stack = core_1.Stack.of(scope);
        const jobDefArn = stack.formatArn({
            service: 'batch',
            resource: 'job-definition',
            sep: '/',
            resourceName: jobDefinitionName,
        });
        class Import extends core_1.Resource {
            constructor() {
                super(...arguments);
                this.jobDefinitionArn = jobDefArn;
                this.jobDefinitionName = jobDefinitionName;
            }
        }
        return new Import(scope, id);
    }
    deserializeEnvVariables(env) {
        const vars = new Array();
        if (env === undefined) {
            return undefined;
        }
        Object.keys(env).map((name) => {
            vars.push({ name, value: env[name] });
        });
        return vars;
    }
    buildJobContainer(container) {
        if (container === undefined) {
            return undefined;
        }
        return {
            command: container.command,
            environment: this.deserializeEnvVariables(container.environment),
            image: this.imageConfig.imageName,
            instanceType: container.instanceType && container.instanceType.toString(),
            jobRoleArn: container.jobRole && container.jobRole.roleArn,
            linuxParameters: container.linuxParams
                ? { devices: container.linuxParams.renderLinuxParameters().devices }
                : undefined,
            memory: container.memoryLimitMiB || 4,
            mountPoints: container.mountPoints,
            privileged: container.privileged || false,
            resourceRequirements: container.gpuCount
                ? [{ type: 'GPU', value: String(container.gpuCount) }]
                : undefined,
            readonlyRootFilesystem: container.readOnly || false,
            ulimits: container.ulimits,
            user: container.user,
            vcpus: container.vcpus || 1,
            volumes: container.volumes,
        };
    }
    buildNodeRangeProps(multiNodeProps) {
        const rangeProps = new Array();
        for (const prop of multiNodeProps.rangeProps) {
            rangeProps.push({
                container: this.buildJobContainer(prop.container),
                targetNodes: `${prop.fromNodeIndex || 0}:${prop.toNodeIndex || multiNodeProps.count}`,
            });
        }
        return rangeProps;
    }
}
exports.JobDefinition = JobDefinition;
//# sourceMappingURL=data:application/json;base64,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