"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.SlackChannelConfiguration = exports.LoggingLevel = void 0;
const cloudwatch = require("../../aws-cloudwatch"); // Automatically re-written from '@aws-cdk/aws-cloudwatch'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const chatbot_generated_1 = require("./chatbot.generated");
/**
 * Logging levels include ERROR, INFO, or NONE.
 */
var LoggingLevel;
(function (LoggingLevel) {
    /**
     * ERROR
     */
    LoggingLevel["ERROR"] = "ERROR";
    /**
     * INFO
     */
    LoggingLevel["INFO"] = "INFO";
    /**
     * NONE
     */
    LoggingLevel["NONE"] = "NONE";
})(LoggingLevel = exports.LoggingLevel || (exports.LoggingLevel = {}));
/**
 * Either a new or imported Slack channel configuration
 */
class SlackChannelConfigurationBase extends cdk.Resource {
    /**
     * Adds extra permission to iam-role of Slack channel configuration
     * @param statement
     */
    addToRolePolicy(statement) {
        if (!this.role) {
            return;
        }
        this.role.addToPrincipalPolicy(statement);
    }
    /**
     * Return the given named metric for this SlackChannelConfiguration
     */
    metric(metricName, props) {
        // AWS Chatbot publishes metrics to us-east-1 regardless of stack region
        // https://docs.aws.amazon.com/chatbot/latest/adminguide/monitoring-cloudwatch.html
        return new cloudwatch.Metric({
            namespace: 'AWS/Chatbot',
            region: 'us-east-1',
            dimensions: {
                ConfigurationName: this.slackChannelConfigurationName,
            },
            metricName,
            ...props,
        });
    }
}
/**
 * A new Slack channel configuration
 */
class SlackChannelConfiguration extends SlackChannelConfigurationBase {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id, {
            physicalName: props.slackChannelConfigurationName,
        });
        this.role = props.role || new iam.Role(this, 'ConfigurationRole', {
            assumedBy: new iam.ServicePrincipal('chatbot.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        const configuration = new chatbot_generated_1.CfnSlackChannelConfiguration(this, 'Resource', {
            configurationName: props.slackChannelConfigurationName,
            iamRoleArn: this.role.roleArn,
            slackWorkspaceId: props.slackWorkspaceId,
            slackChannelId: props.slackChannelId,
            snsTopicArns: (_a = props.notificationTopics) === null || _a === void 0 ? void 0 : _a.map(topic => topic.topicArn),
            loggingLevel: (_b = props.loggingLevel) === null || _b === void 0 ? void 0 : _b.toString(),
        });
        // Log retention
        // AWS Chatbot publishes logs to us-east-1 regardless of stack region https://docs.aws.amazon.com/chatbot/latest/adminguide/cloudwatch-logs.html
        if (props.logRetention) {
            new logs.LogRetention(this, 'LogRetention', {
                logGroupName: `/aws/chatbot/${props.slackChannelConfigurationName}`,
                retention: props.logRetention,
                role: props.logRetentionRole,
                logGroupRegion: 'us-east-1',
                logRetentionRetryOptions: props.logRetentionRetryOptions,
            });
        }
        this.slackChannelConfigurationArn = configuration.ref;
        this.slackChannelConfigurationName = props.slackChannelConfigurationName;
    }
    /**
     * Import an existing Slack channel configuration provided an ARN
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param slackChannelConfigurationArn configuration ARN (i.e. arn:aws:chatbot::1234567890:chat-configuration/slack-channel/my-slack)
     *
     * @returns a reference to the existing Slack channel configuration
     */
    static fromSlackChannelConfigurationArn(scope, id, slackChannelConfigurationArn) {
        const re = /^slack-channel\//;
        const resourceName = cdk.Stack.of(scope).parseArn(slackChannelConfigurationArn).resourceName;
        if (!re.test(resourceName)) {
            throw new Error('The ARN of a Slack integration must be in the form: arn:aws:chatbot:{region}:{account}:chat-configuration/slack-channel/{slackChannelName}');
        }
        class Import extends SlackChannelConfigurationBase {
            constructor(s, i) {
                super(s, i);
                /**
                 * @attribute
                 */
                this.slackChannelConfigurationArn = slackChannelConfigurationArn;
                this.role = undefined;
                /**
                 * Returns a name of Slack channel configuration
                 *
                 * NOTE:
                 * For example: arn:aws:chatbot::1234567890:chat-configuration/slack-channel/my-slack
                 * The ArnComponents API will return `slack-channel/my-slack`
                 * It need to handle that to gets a correct name.`my-slack`
                 */
                this.slackChannelConfigurationName = resourceName.substring('slack-channel/'.length);
                this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
            }
        }
        return new Import(scope, id);
    }
    /**
     * Return the given named metric for All SlackChannelConfigurations
     */
    static metricAll(metricName, props) {
        // AWS Chatbot publishes metrics to us-east-1 regardless of stack region
        // https://docs.aws.amazon.com/chatbot/latest/adminguide/monitoring-cloudwatch.html
        return new cloudwatch.Metric({
            namespace: 'AWS/Chatbot',
            region: 'us-east-1',
            metricName,
            ...props,
        });
    }
}
exports.SlackChannelConfiguration = SlackChannelConfiguration;
//# sourceMappingURL=data:application/json;base64,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