"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.OriginAccessIdentity = void 0;
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const cdk = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloudfront_generated_1 = require("./cloudfront.generated");
class OriginAccessIdentityBase extends cdk.Resource {
    /**
     * The ARN to include in S3 bucket policy to allow CloudFront access
     */
    arn() {
        return cdk.Stack.of(this).formatArn({
            service: 'iam',
            region: '',
            account: 'cloudfront',
            resource: 'user',
            resourceName: `CloudFront Origin Access Identity ${this.originAccessIdentityName}`,
        });
    }
}
/**
 * An origin access identity is a special CloudFront user that you can
 * associate with Amazon S3 origins, so that you can secure all or just some of
 * your Amazon S3 content.
 *
 * @resource AWS::CloudFront::CloudFrontOriginAccessIdentity
 */
class OriginAccessIdentity extends OriginAccessIdentityBase {
    constructor(scope, id, props) {
        var _a;
        super(scope, id);
        // Comment has a max length of 128.
        const comment = ((_a = props === null || props === void 0 ? void 0 : props.comment) !== null && _a !== void 0 ? _a : 'Allows CloudFront to reach the bucket').substr(0, 128);
        this.resource = new cloudfront_generated_1.CfnCloudFrontOriginAccessIdentity(this, 'Resource', {
            cloudFrontOriginAccessIdentityConfig: { comment },
        });
        // physical id - OAI name
        this.originAccessIdentityName = this.getResourceNameAttribute(this.resource.ref);
        // Canonical user to whitelist in S3 Bucket Policy
        this.cloudFrontOriginAccessIdentityS3CanonicalUserId = this.resource.attrS3CanonicalUserId;
        // The principal for must be either the canonical user or a special ARN
        // with the CloudFront Origin Access Id (see `arn()` method). For
        // import/export the OAI is anyway required so the principal is constructed
        // with it. But for the normal case the S3 Canonical User as a nicer
        // interface and does not require constructing the ARN.
        this.grantPrincipal = new iam.CanonicalUserPrincipal(this.cloudFrontOriginAccessIdentityS3CanonicalUserId);
    }
    /**
     * Creates a OriginAccessIdentity by providing the OriginAccessIdentityName
     */
    static fromOriginAccessIdentityName(scope, id, originAccessIdentityName) {
        class Import extends OriginAccessIdentityBase {
            constructor(s, i) {
                super(s, i, { physicalName: originAccessIdentityName });
                this.originAccessIdentityName = originAccessIdentityName;
                this.grantPrincipal = new iam.ArnPrincipal(this.arn());
            }
        }
        return new Import(scope, id);
    }
}
exports.OriginAccessIdentity = OriginAccessIdentity;
//# sourceMappingURL=data:application/json;base64,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