"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.DataResourceType = exports.Trail = exports.ReadWriteType = void 0;
const events = require("../../aws-events"); // Automatically re-written from '@aws-cdk/aws-events'
const iam = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const logs = require("../../aws-logs"); // Automatically re-written from '@aws-cdk/aws-logs'
const s3 = require("../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const cloudtrail_generated_1 = require("./cloudtrail.generated");
/**
 * Types of events that CloudTrail can log
 */
var ReadWriteType;
(function (ReadWriteType) {
    /**
     * Read-only events include API operations that read your resources,
     * but don't make changes.
     * For example, read-only events include the Amazon EC2 DescribeSecurityGroups
     * and DescribeSubnets API operations.
     */
    ReadWriteType["READ_ONLY"] = "ReadOnly";
    /**
     * Write-only events include API operations that modify (or might modify)
     * your resources.
     * For example, the Amazon EC2 RunInstances and TerminateInstances API
     * operations modify your instances.
     */
    ReadWriteType["WRITE_ONLY"] = "WriteOnly";
    /**
     * All events
     */
    ReadWriteType["ALL"] = "All";
    /**
     * No events
     */
    ReadWriteType["NONE"] = "None";
})(ReadWriteType = exports.ReadWriteType || (exports.ReadWriteType = {}));
/**
 * Cloud trail allows you to log events that happen in your AWS account
 * For example:
 *
 * import { CloudTrail } from '@aws-cdk/aws-cloudtrail'
 *
 * const cloudTrail = new CloudTrail(this, 'MyTrail');
 *
 * NOTE the above example creates an UNENCRYPTED bucket by default,
 * If you are required to use an Encrypted bucket you can supply a preconfigured bucket
 * via TrailProps
 *
 */
class Trail extends core_1.Resource {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e, _f;
        super(scope, id, {
            physicalName: props.trailName,
        });
        this.eventSelectors = [];
        const cloudTrailPrincipal = new iam.ServicePrincipal('cloudtrail.amazonaws.com');
        this.s3bucket = props.bucket || new s3.Bucket(this, 'S3', { encryption: s3.BucketEncryption.UNENCRYPTED });
        this.s3bucket.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.s3bucket.bucketArn],
            actions: ['s3:GetBucketAcl'],
            principals: [cloudTrailPrincipal],
        }));
        this.s3bucket.addToResourcePolicy(new iam.PolicyStatement({
            resources: [this.s3bucket.arnForObjects(`${props.s3KeyPrefix ? `${props.s3KeyPrefix}/` : ''}AWSLogs/${core_1.Stack.of(this).account}/*`)],
            actions: ['s3:PutObject'],
            principals: [cloudTrailPrincipal],
            conditions: {
                StringEquals: { 's3:x-amz-acl': 'bucket-owner-full-control' },
            },
        }));
        this.topic = props.snsTopic;
        if (this.topic) {
            this.topic.grantPublish(cloudTrailPrincipal);
        }
        let logsRole;
        if (props.sendToCloudWatchLogs) {
            if (props.cloudWatchLogGroup) {
                this.logGroup = props.cloudWatchLogGroup;
            }
            else {
                this.logGroup = new logs.LogGroup(this, 'LogGroup', {
                    retention: (_a = props.cloudWatchLogsRetention) !== null && _a !== void 0 ? _a : logs.RetentionDays.ONE_YEAR,
                });
            }
            logsRole = new iam.Role(this, 'LogsRole', { assumedBy: cloudTrailPrincipal });
            logsRole.addToPolicy(new iam.PolicyStatement({
                actions: ['logs:PutLogEvents', 'logs:CreateLogStream'],
                resources: [this.logGroup.logGroupArn],
            }));
        }
        if (props.managementEvents) {
            let managementEvent;
            if (props.managementEvents === ReadWriteType.NONE) {
                managementEvent = {
                    includeManagementEvents: false,
                };
            }
            else {
                managementEvent = {
                    includeManagementEvents: true,
                    readWriteType: props.managementEvents,
                };
            }
            this.eventSelectors.push(managementEvent);
        }
        if (props.kmsKey && props.encryptionKey) {
            throw new Error('Both kmsKey and encryptionKey must not be specified. Use only encryptionKey');
        }
        // TODO: not all regions support validation. Use service configuration data to fail gracefully
        const trail = new cloudtrail_generated_1.CfnTrail(this, 'Resource', {
            isLogging: true,
            enableLogFileValidation: props.enableFileValidation == null ? true : props.enableFileValidation,
            isMultiRegionTrail: props.isMultiRegionTrail == null ? true : props.isMultiRegionTrail,
            includeGlobalServiceEvents: props.includeGlobalServiceEvents == null ? true : props.includeGlobalServiceEvents,
            trailName: this.physicalName,
            kmsKeyId: (_c = (_b = props.encryptionKey) === null || _b === void 0 ? void 0 : _b.keyArn) !== null && _c !== void 0 ? _c : (_d = props.kmsKey) === null || _d === void 0 ? void 0 : _d.keyArn,
            s3BucketName: this.s3bucket.bucketName,
            s3KeyPrefix: props.s3KeyPrefix,
            cloudWatchLogsLogGroupArn: (_e = this.logGroup) === null || _e === void 0 ? void 0 : _e.logGroupArn,
            cloudWatchLogsRoleArn: logsRole === null || logsRole === void 0 ? void 0 : logsRole.roleArn,
            snsTopicName: (_f = this.topic) === null || _f === void 0 ? void 0 : _f.topicName,
            eventSelectors: this.eventSelectors,
        });
        this.trailArn = this.getResourceArnAttribute(trail.attrArn, {
            service: 'cloudtrail',
            resource: 'trail',
            resourceName: this.physicalName,
        });
        this.trailSnsTopicArn = trail.attrSnsTopicArn;
        // Add a dependency on the bucket policy being updated, CloudTrail will test this upon creation.
        if (this.s3bucket.policy) {
            trail.node.addDependency(this.s3bucket.policy);
        }
        // If props.sendToCloudWatchLogs is set to true then the trail needs to depend on the created logsRole
        // so that it can create the log stream for the log group. This ensures the logsRole is created and propagated
        // before the trail tries to create the log stream.
        if (logsRole !== undefined) {
            trail.node.addDependency(logsRole);
        }
    }
    /**
     * Create an event rule for when an event is recorded by any Trail in the account.
     *
     * Note that the event doesn't necessarily have to come from this Trail, it can
     * be captured from any one.
     *
     * Be sure to filter the event further down using an event pattern.
     */
    static onEvent(scope, id, options = {}) {
        const rule = new events.Rule(scope, id, options);
        rule.addTarget(options.target);
        rule.addEventPattern({
            detailType: ['AWS API Call via CloudTrail'],
        });
        return rule;
    }
    /**
     * When an event occurs in your account, CloudTrail evaluates whether the event matches the settings for your trails.
     * Only events that match your trail settings are delivered to your Amazon S3 bucket and Amazon CloudWatch Logs log group.
     *
     * This method adds an Event Selector for filtering events that match either S3 or Lambda function operations.
     *
     * Data events: These events provide insight into the resource operations performed on or within a resource.
     * These are also known as data plane operations.
     *
     * @param dataResourceValues the list of data resource ARNs to include in logging (maximum 250 entries).
     * @param options the options to configure logging of management and data events.
     */
    addEventSelector(dataResourceType, dataResourceValues, options = {}) {
        if (dataResourceValues.length > 250) {
            throw new Error('A maximum of 250 data elements can be in one event selector');
        }
        if (this.eventSelectors.length > 5) {
            throw new Error('A maximum of 5 event selectors are supported per trail.');
        }
        this.eventSelectors.push({
            dataResources: [{
                    type: dataResourceType,
                    values: dataResourceValues,
                }],
            includeManagementEvents: options.includeManagementEvents,
            readWriteType: options.readWriteType,
        });
    }
    /**
     * When an event occurs in your account, CloudTrail evaluates whether the event matches the settings for your trails.
     * Only events that match your trail settings are delivered to your Amazon S3 bucket and Amazon CloudWatch Logs log group.
     *
     * This method adds a Lambda Data Event Selector for filtering events that match Lambda function operations.
     *
     * Data events: These events provide insight into the resource operations performed on or within a resource.
     * These are also known as data plane operations.
     *
     * @param handlers the list of lambda function handlers whose data events should be logged (maximum 250 entries).
     * @param options the options to configure logging of management and data events.
     */
    addLambdaEventSelector(handlers, options = {}) {
        if (handlers.length === 0) {
            return;
        }
        const dataResourceValues = handlers.map((h) => h.functionArn);
        return this.addEventSelector(DataResourceType.LAMBDA_FUNCTION, dataResourceValues, options);
    }
    /**
     * Log all Lamda data events for all lambda functions the account.
     * @see https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html
     * @default false
     */
    logAllLambdaDataEvents(options = {}) {
        return this.addEventSelector(DataResourceType.LAMBDA_FUNCTION, [`arn:${this.stack.partition}:lambda`], options);
    }
    /**
     * When an event occurs in your account, CloudTrail evaluates whether the event matches the settings for your trails.
     * Only events that match your trail settings are delivered to your Amazon S3 bucket and Amazon CloudWatch Logs log group.
     *
     * This method adds an S3 Data Event Selector for filtering events that match S3 operations.
     *
     * Data events: These events provide insight into the resource operations performed on or within a resource.
     * These are also known as data plane operations.
     *
     * @param s3Selector the list of S3 bucket with optional prefix to include in logging (maximum 250 entries).
     * @param options the options to configure logging of management and data events.
     */
    addS3EventSelector(s3Selector, options = {}) {
        if (s3Selector.length === 0) {
            return;
        }
        const dataResourceValues = s3Selector.map((sel) => { var _a; return `${sel.bucket.bucketArn}/${(_a = sel.objectPrefix) !== null && _a !== void 0 ? _a : ''}`; });
        return this.addEventSelector(DataResourceType.S3_OBJECT, dataResourceValues, options);
    }
    /**
     * Log all S3 data events for all objects for all buckets in the account.
     * @see https://docs.aws.amazon.com/awscloudtrail/latest/userguide/logging-data-events-with-cloudtrail.html
     * @default false
     */
    logAllS3DataEvents(options = {}) {
        return this.addEventSelector(DataResourceType.S3_OBJECT, [`arn:${this.stack.partition}:s3:::`], options);
    }
    /**
     * Create an event rule for when an event is recorded by any Trail in the account.
     *
     * Note that the event doesn't necessarily have to come from this Trail, it can
     * be captured from any one.
     *
     * Be sure to filter the event further down using an event pattern.
     *
     * @deprecated - use Trail.onEvent()
     */
    onCloudTrailEvent(id, options = {}) {
        return Trail.onEvent(this, id, options);
    }
}
exports.Trail = Trail;
/**
 * Resource type for a data event
 */
var DataResourceType;
(function (DataResourceType) {
    /**
     * Data resource type for Lambda function
     */
    DataResourceType["LAMBDA_FUNCTION"] = "AWS::Lambda::Function";
    /**
     * Data resource type for S3 objects
     */
    DataResourceType["S3_OBJECT"] = "AWS::S3::Object";
})(DataResourceType = exports.DataResourceType || (exports.DataResourceType = {}));
//# sourceMappingURL=data:application/json;base64,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