"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Alarm = exports.TreatMissingData = exports.ComparisonOperator = void 0;
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const alarm_base_1 = require("./alarm-base");
const cloudwatch_generated_1 = require("./cloudwatch.generated");
const metric_util_1 = require("./private/metric-util");
const object_1 = require("./private/object");
const rendering_1 = require("./private/rendering");
const statistic_1 = require("./private/statistic");
/**
 * Comparison operator for evaluating alarms
 */
var ComparisonOperator;
(function (ComparisonOperator) {
    /**
     * Specified statistic is greater than or equal to the threshold
     */
    ComparisonOperator["GREATER_THAN_OR_EQUAL_TO_THRESHOLD"] = "GreaterThanOrEqualToThreshold";
    /**
     * Specified statistic is strictly greater than the threshold
     */
    ComparisonOperator["GREATER_THAN_THRESHOLD"] = "GreaterThanThreshold";
    /**
     * Specified statistic is strictly less than the threshold
     */
    ComparisonOperator["LESS_THAN_THRESHOLD"] = "LessThanThreshold";
    /**
     * Specified statistic is less than or equal to the threshold.
     */
    ComparisonOperator["LESS_THAN_OR_EQUAL_TO_THRESHOLD"] = "LessThanOrEqualToThreshold";
    /**
     * Specified statistic is lower than or greater than the anomaly model band.
     * Used only for alarms based on anomaly detection models
     */
    ComparisonOperator["LESS_THAN_LOWER_OR_GREATER_THAN_UPPER_THRESHOLD"] = "LessThanLowerOrGreaterThanUpperThreshold";
    /**
     * Specified statistic is greater than the anomaly model band.
     * Used only for alarms based on anomaly detection models
     */
    ComparisonOperator["GREATER_THAN_UPPER_THRESHOLD"] = "GreaterThanUpperThreshold";
    /**
     * Specified statistic is lower than the anomaly model band.
     * Used only for alarms based on anomaly detection models
     */
    ComparisonOperator["LESS_THAN_LOWER_THRESHOLD"] = "LessThanLowerThreshold";
})(ComparisonOperator = exports.ComparisonOperator || (exports.ComparisonOperator = {}));
const OPERATOR_SYMBOLS = {
    GreaterThanOrEqualToThreshold: '>=',
    GreaterThanThreshold: '>',
    LessThanThreshold: '<',
    LessThanOrEqualToThreshold: '<=',
};
/**
 * Specify how missing data points are treated during alarm evaluation
 */
var TreatMissingData;
(function (TreatMissingData) {
    /**
     * Missing data points are treated as breaching the threshold
     */
    TreatMissingData["BREACHING"] = "breaching";
    /**
     * Missing data points are treated as being within the threshold
     */
    TreatMissingData["NOT_BREACHING"] = "notBreaching";
    /**
     * The current alarm state is maintained
     */
    TreatMissingData["IGNORE"] = "ignore";
    /**
     * The alarm does not consider missing data points when evaluating whether to change state
     */
    TreatMissingData["MISSING"] = "missing";
})(TreatMissingData = exports.TreatMissingData || (exports.TreatMissingData = {}));
/**
 * An alarm on a CloudWatch metric
 */
class Alarm extends alarm_base_1.AlarmBase {
    constructor(scope, id, props) {
        super(scope, id, {
            physicalName: props.alarmName,
        });
        const comparisonOperator = props.comparisonOperator || ComparisonOperator.GREATER_THAN_OR_EQUAL_TO_THRESHOLD;
        // Render metric, process potential overrides from the alarm
        // (It would be preferable if the statistic etc. was worked into the metric,
        // but hey we're allowing overrides...)
        const metricProps = this.renderMetric(props.metric);
        if (props.period) {
            metricProps.period = props.period.toSeconds();
        }
        if (props.statistic) {
            // Will overwrite both fields if present
            Object.assign(metricProps, {
                statistic: renderIfSimpleStatistic(props.statistic),
                extendedStatistic: renderIfExtendedStatistic(props.statistic),
            });
        }
        const alarm = new cloudwatch_generated_1.CfnAlarm(this, 'Resource', {
            // Meta
            alarmDescription: props.alarmDescription,
            alarmName: this.physicalName,
            // Evaluation
            comparisonOperator,
            threshold: props.threshold,
            datapointsToAlarm: props.datapointsToAlarm,
            evaluateLowSampleCountPercentile: props.evaluateLowSampleCountPercentile,
            evaluationPeriods: props.evaluationPeriods,
            treatMissingData: props.treatMissingData,
            // Actions
            actionsEnabled: props.actionsEnabled,
            alarmActions: core_1.Lazy.listValue({ produce: () => this.alarmActionArns }),
            insufficientDataActions: core_1.Lazy.listValue({ produce: (() => this.insufficientDataActionArns) }),
            okActions: core_1.Lazy.listValue({ produce: () => this.okActionArns }),
            // Metric
            ...metricProps,
        });
        this.alarmArn = this.getResourceArnAttribute(alarm.attrArn, {
            service: 'cloudwatch',
            resource: 'alarm',
            resourceName: this.physicalName,
            sep: ':',
        });
        this.alarmName = this.getResourceNameAttribute(alarm.ref);
        this.metric = props.metric;
        const datapoints = props.datapointsToAlarm || props.evaluationPeriods;
        this.annotation = {
            // eslint-disable-next-line max-len
            label: `${this.metric} ${OPERATOR_SYMBOLS[comparisonOperator]} ${props.threshold} for ${datapoints} datapoints within ${describePeriod(props.evaluationPeriods * metric_util_1.metricPeriod(props.metric).toSeconds())}`,
            value: props.threshold,
        };
    }
    /**
     * Import an existing CloudWatch alarm provided an ARN
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name
     * @param alarmArn Alarm ARN (i.e. arn:aws:cloudwatch:<region>:<account-id>:alarm:Foo)
     */
    static fromAlarmArn(scope, id, alarmArn) {
        class Import extends alarm_base_1.AlarmBase {
            constructor() {
                super(...arguments);
                this.alarmArn = alarmArn;
                this.alarmName = core_1.Stack.of(scope).parseArn(alarmArn, ':').resourceName;
            }
        }
        return new Import(scope, id);
    }
    /**
     * Turn this alarm into a horizontal annotation
     *
     * This is useful if you want to represent an Alarm in a non-AlarmWidget.
     * An `AlarmWidget` can directly show an alarm, but it can only show a
     * single alarm and no other metrics. Instead, you can convert the alarm to
     * a HorizontalAnnotation and add it as an annotation to another graph.
     *
     * This might be useful if:
     *
     * - You want to show multiple alarms inside a single graph, for example if
     *   you have both a "small margin/long period" alarm as well as a
     *   "large margin/short period" alarm.
     *
     * - You want to show an Alarm line in a graph with multiple metrics in it.
     */
    toAnnotation() {
        return this.annotation;
    }
    renderMetric(metric) {
        const self = this;
        return metric_util_1.dispatchMetric(metric, {
            withStat(st) {
                var _a;
                self.validateMetricStat(st, metric);
                return object_1.dropUndefined({
                    dimensions: st.dimensions,
                    namespace: st.namespace,
                    metricName: st.metricName,
                    period: (_a = st.period) === null || _a === void 0 ? void 0 : _a.toSeconds(),
                    statistic: renderIfSimpleStatistic(st.statistic),
                    extendedStatistic: renderIfExtendedStatistic(st.statistic),
                    unit: st.unitFilter,
                });
            },
            withExpression() {
                // Expand the math expression metric into a set
                const mset = new rendering_1.MetricSet();
                mset.addTopLevel(true, metric);
                let eid = 0;
                function uniqueMetricId() {
                    return `expr_${++eid}`;
                }
                return {
                    metrics: mset.entries.map(entry => metric_util_1.dispatchMetric(entry.metric, {
                        withStat(stat, conf) {
                            var _a;
                            self.validateMetricStat(stat, entry.metric);
                            return {
                                metricStat: {
                                    metric: {
                                        metricName: stat.metricName,
                                        namespace: stat.namespace,
                                        dimensions: stat.dimensions,
                                    },
                                    period: stat.period.toSeconds(),
                                    stat: stat.statistic,
                                    unit: stat.unitFilter,
                                },
                                id: entry.id || uniqueMetricId(),
                                label: (_a = conf.renderingProperties) === null || _a === void 0 ? void 0 : _a.label,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                        withExpression(expr, conf) {
                            var _a;
                            return {
                                expression: expr.expression,
                                id: entry.id || uniqueMetricId(),
                                label: (_a = conf.renderingProperties) === null || _a === void 0 ? void 0 : _a.label,
                                period: mathExprHasSubmetrics(expr) ? undefined : expr.period,
                                returnData: entry.tag ? undefined : false,
                            };
                        },
                    })),
                };
            },
        });
    }
    /**
     * Validate that if a region and account are in the given stat config, they match the Alarm
     */
    validateMetricStat(stat, metric) {
        const stack = core_1.Stack.of(this);
        if (definitelyDifferent(stat.region, stack.region)) {
            throw new Error(`Cannot create an Alarm in region '${stack.region}' based on metric '${metric}' in '${stat.region}'`);
        }
        if (definitelyDifferent(stat.account, stack.account)) {
            throw new Error(`Cannot create an Alarm in account '${stack.account}' based on metric '${metric}' in '${stat.account}'`);
        }
    }
}
exports.Alarm = Alarm;
function definitelyDifferent(x, y) {
    return x && !core_1.Token.isUnresolved(y) && x !== y;
}
/**
 * Return a human readable string for this period
 *
 * We know the seconds are always one of a handful of allowed values.
 */
function describePeriod(seconds) {
    if (seconds === 60) {
        return '1 minute';
    }
    if (seconds === 1) {
        return '1 second';
    }
    if (seconds > 60) {
        return (seconds / 60) + ' minutes';
    }
    return seconds + ' seconds';
}
function renderIfSimpleStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'simple') {
        return parsed.statistic;
    }
    return undefined;
}
function renderIfExtendedStatistic(statistic) {
    if (statistic === undefined) {
        return undefined;
    }
    const parsed = statistic_1.parseStatistic(statistic);
    if (parsed.type === 'percentile') {
        // Already percentile. Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the p.
        return statistic.toLowerCase();
    }
    return undefined;
}
function mathExprHasSubmetrics(expr) {
    return Object.keys(expr.usingMetrics).length > 0;
}
//# sourceMappingURL=data:application/json;base64,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