"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.accountIfDifferentFromStack = exports.regionIfDifferentFromStack = void 0;
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
/**
 * Make a Token that renders to given region if used in a different stack, otherwise undefined
 */
function regionIfDifferentFromStack(region) {
    return cdk.Token.asString(new StackDependentToken(region, stack => stack.region));
}
exports.regionIfDifferentFromStack = regionIfDifferentFromStack;
/**
 * Make a Token that renders to given account if used in a different stack, otherwise undefined
 */
function accountIfDifferentFromStack(account) {
    return cdk.Token.asString(new StackDependentToken(account, stack => stack.account));
}
exports.accountIfDifferentFromStack = accountIfDifferentFromStack;
/**
 * A lazy token that requires an instance of Stack to evaluate
 */
class StackDependentToken {
    constructor(originalValue, fn) {
        this.originalValue = originalValue;
        this.fn = fn;
        this.creationStack = cdk.captureStackTrace();
    }
    resolve(context) {
        const stackValue = this.fn(cdk.Stack.of(context.scope));
        // Don't render if the values are definitely the same. If the stack
        // is unresolved we don't know, better output the value.
        if (!cdk.Token.isUnresolved(stackValue) && stackValue === this.originalValue) {
            return undefined;
        }
        return this.originalValue;
    }
    toString() {
        return cdk.Token.asString(this);
    }
    toJSON() {
        return this.originalValue;
    }
}
//# sourceMappingURL=data:application/json;base64,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