"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.normalizeStatistic = exports.parseStatistic = void 0;
const metric_types_1 = require("../metric-types");
/**
 * Parse a statistic, returning the type of metric that was used (simple or percentile)
 */
function parseStatistic(stat) {
    const lowerStat = stat.toLowerCase();
    // Simple statistics
    const statMap = {
        average: metric_types_1.Statistic.AVERAGE,
        avg: metric_types_1.Statistic.AVERAGE,
        minimum: metric_types_1.Statistic.MINIMUM,
        min: metric_types_1.Statistic.MINIMUM,
        maximum: metric_types_1.Statistic.MAXIMUM,
        max: metric_types_1.Statistic.MAXIMUM,
        samplecount: metric_types_1.Statistic.SAMPLE_COUNT,
        n: metric_types_1.Statistic.SAMPLE_COUNT,
        sum: metric_types_1.Statistic.SUM,
    };
    if (lowerStat in statMap) {
        return {
            type: 'simple',
            statistic: statMap[lowerStat],
        };
    }
    // Percentile statistics
    const re = /^p([\d.]+)$/;
    const m = re.exec(lowerStat);
    if (m) {
        return {
            type: 'percentile',
            percentile: parseFloat(m[1]),
        };
    }
    throw new Error(`Not a valid statistic: '${stat}', must be one of Average | Minimum | Maximum | SampleCount | Sum | pNN.NN`);
}
exports.parseStatistic = parseStatistic;
function normalizeStatistic(stat) {
    const parsed = parseStatistic(stat);
    if (parsed.type === 'simple') {
        return parsed.statistic;
    }
    else {
        // Already percentile. Avoid parsing because we might get into
        // floating point rounding issues, return as-is but lowercase the p.
        return stat.toLowerCase();
    }
}
exports.normalizeStatistic = normalizeStatistic;
//# sourceMappingURL=data:application/json;base64,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