"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ServerDeploymentGroup = exports.InstanceTagSet = void 0;
const ec2 = require("../../../aws-ec2"); // Automatically re-written from '@aws-cdk/aws-ec2'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const s3 = require("../../../aws-s3"); // Automatically re-written from '@aws-cdk/aws-s3'
const cdk = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const codedeploy_generated_1 = require("../codedeploy.generated");
const utils_1 = require("../utils");
const application_1 = require("./application");
const deployment_config_1 = require("./deployment-config");
const load_balancer_1 = require("./load-balancer");
/**
 * Represents a reference to a CodeDeploy EC2/on-premise Deployment Group.
 *
 * If you're managing the Deployment Group alongside the rest of your CDK resources,
 * use the {@link ServerDeploymentGroup} class.
 *
 * If you want to reference an already existing Deployment Group,
 * or one defined in a different CDK Stack,
 * use the {@link #import} method.
 */
class ServerDeploymentGroupBase extends cdk.Resource {
    constructor(scope, id, deploymentConfig, props) {
        super(scope, id, props);
        this.deploymentConfig = deploymentConfig || deployment_config_1.ServerDeploymentConfig.ONE_AT_A_TIME;
    }
}
class ImportedServerDeploymentGroup extends ServerDeploymentGroupBase {
    constructor(scope, id, props) {
        super(scope, id, props.deploymentConfig);
        this.role = undefined;
        this.autoScalingGroups = undefined;
        this.application = props.application;
        this.deploymentGroupName = props.deploymentGroupName;
        this.deploymentGroupArn = utils_1.arnForDeploymentGroup(props.application.applicationName, props.deploymentGroupName);
    }
}
/**
 * Represents a set of instance tag groups.
 * An instance will match a set if it matches all of the groups in the set -
 * in other words, sets follow 'and' semantics.
 * You can have a maximum of 3 tag groups inside a set.
 */
class InstanceTagSet {
    constructor(...instanceTagGroups) {
        if (instanceTagGroups.length > 3) {
            throw new Error('An instance tag set can have a maximum of 3 instance tag groups, ' +
                `but ${instanceTagGroups.length} were provided`);
        }
        this._instanceTagGroups = instanceTagGroups;
    }
    get instanceTagGroups() {
        return this._instanceTagGroups.slice();
    }
}
exports.InstanceTagSet = InstanceTagSet;
/**
 * A CodeDeploy Deployment Group that deploys to EC2/on-premise instances.
 * @resource AWS::CodeDeploy::DeploymentGroup
 */
class ServerDeploymentGroup extends ServerDeploymentGroupBase {
    constructor(scope, id, props = {}) {
        super(scope, id, props.deploymentConfig, {
            physicalName: props.deploymentGroupName,
        });
        this.application = props.application || new application_1.ServerApplication(this, 'Application');
        this.role = props.role || new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('codedeploy.amazonaws.com'),
            managedPolicies: [iam.ManagedPolicy.fromAwsManagedPolicyName('service-role/AWSCodeDeployRole')],
        });
        this._autoScalingGroups = props.autoScalingGroups || [];
        this.installAgent = props.installAgent === undefined ? true : props.installAgent;
        this.codeDeployBucket = s3.Bucket.fromBucketName(this, 'Bucket', `aws-codedeploy-${cdk.Stack.of(this).region}`);
        for (const asg of this._autoScalingGroups) {
            this.addCodeDeployAgentInstallUserData(asg);
        }
        this.alarms = props.alarms || [];
        const resource = new codedeploy_generated_1.CfnDeploymentGroup(this, 'Resource', {
            applicationName: this.application.applicationName,
            deploymentGroupName: this.physicalName,
            serviceRoleArn: this.role.roleArn,
            deploymentConfigName: props.deploymentConfig &&
                props.deploymentConfig.deploymentConfigName,
            autoScalingGroups: cdk.Lazy.listValue({ produce: () => this._autoScalingGroups.map(asg => asg.autoScalingGroupName) }, { omitEmpty: true }),
            loadBalancerInfo: this.loadBalancerInfo(props.loadBalancer),
            deploymentStyle: props.loadBalancer === undefined
                ? undefined
                : {
                    deploymentOption: 'WITH_TRAFFIC_CONTROL',
                },
            ec2TagSet: this.ec2TagSet(props.ec2InstanceTags),
            onPremisesTagSet: this.onPremiseTagSet(props.onPremiseInstanceTags),
            alarmConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAlarmConfiguration(this.alarms, props.ignorePollAlarmsFailure) }),
            autoRollbackConfiguration: cdk.Lazy.anyValue({ produce: () => utils_1.renderAutoRollbackConfiguration(this.alarms, props.autoRollback) }),
        });
        this.deploymentGroupName = this.getResourceNameAttribute(resource.ref);
        this.deploymentGroupArn = this.getResourceArnAttribute(utils_1.arnForDeploymentGroup(this.application.applicationName, resource.ref), {
            service: 'codedeploy',
            resource: 'deploymentgroup',
            resourceName: `${this.application.applicationName}/${this.physicalName}`,
            sep: ':',
        });
    }
    /**
     * Import an EC2/on-premise Deployment Group defined either outside the CDK app,
     * or in a different region.
     *
     * @param scope the parent Construct for this new Construct
     * @param id the logical ID of this new Construct
     * @param attrs the properties of the referenced Deployment Group
     * @returns a Construct representing a reference to an existing Deployment Group
     */
    static fromServerDeploymentGroupAttributes(scope, id, attrs) {
        return new ImportedServerDeploymentGroup(scope, id, attrs);
    }
    /**
     * Adds an additional auto-scaling group to this Deployment Group.
     *
     * @param asg the auto-scaling group to add to this Deployment Group.
     * [disable-awslint:ref-via-interface] is needed in order to install the code
     * deploy agent by updating the ASGs user data.
     */
    addAutoScalingGroup(asg) {
        this._autoScalingGroups.push(asg);
        this.addCodeDeployAgentInstallUserData(asg);
    }
    /**
     * Associates an additional alarm with this Deployment Group.
     *
     * @param alarm the alarm to associate with this Deployment Group
     */
    addAlarm(alarm) {
        this.alarms.push(alarm);
    }
    get autoScalingGroups() {
        return this._autoScalingGroups.slice();
    }
    addCodeDeployAgentInstallUserData(asg) {
        if (!this.installAgent) {
            return;
        }
        this.codeDeployBucket.grantRead(asg, 'latest/*');
        switch (asg.osType) {
            case ec2.OperatingSystemType.LINUX:
                asg.addUserData('PKG_CMD=`which yum 2>/dev/null`', 'if [ -z "$PKG_CMD" ]; then', 'PKG_CMD=apt-get', 'else', 'PKG=CMD=yum', 'fi', '$PKG_CMD update -y', '$PKG_CMD install -y ruby2.0', 'if [ $? -ne 0 ]; then', '$PKG_CMD install -y ruby', 'fi', '$PKG_CMD install -y awscli', 'TMP_DIR=`mktemp -d`', 'cd $TMP_DIR', `aws s3 cp s3://aws-codedeploy-${cdk.Stack.of(this).region}/latest/install . --region ${cdk.Stack.of(this).region}`, 'chmod +x ./install', './install auto', 'rm -fr $TMP_DIR');
                break;
            case ec2.OperatingSystemType.WINDOWS:
                asg.addUserData('Set-Variable -Name TEMPDIR -Value (New-TemporaryFile).DirectoryName', `aws s3 cp s3://aws-codedeploy-${cdk.Stack.of(this).region}/latest/codedeploy-agent.msi $TEMPDIR\\codedeploy-agent.msi`, '$TEMPDIR\\codedeploy-agent.msi /quiet /l c:\\temp\\host-agent-install-log.txt');
                break;
        }
    }
    loadBalancerInfo(loadBalancer) {
        if (!loadBalancer) {
            return undefined;
        }
        switch (loadBalancer.generation) {
            case load_balancer_1.LoadBalancerGeneration.FIRST:
                return {
                    elbInfoList: [
                        { name: loadBalancer.name },
                    ],
                };
            case load_balancer_1.LoadBalancerGeneration.SECOND:
                return {
                    targetGroupInfoList: [
                        { name: loadBalancer.name },
                    ],
                };
        }
    }
    ec2TagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            ec2TagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    ec2TagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    onPremiseTagSet(tagSet) {
        if (!tagSet || tagSet.instanceTagGroups.length === 0) {
            return undefined;
        }
        return {
            onPremisesTagSetList: tagSet.instanceTagGroups.map(tagGroup => {
                return {
                    onPremisesTagGroup: this.tagGroup2TagsArray(tagGroup),
                };
            }),
        };
    }
    tagGroup2TagsArray(tagGroup) {
        const tagsInGroup = new Array();
        for (const tagKey in tagGroup) {
            if (tagGroup.hasOwnProperty(tagKey)) {
                const tagValues = tagGroup[tagKey];
                if (tagKey.length > 0) {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                key: tagKey,
                                value: tagValue,
                                type: 'KEY_AND_VALUE',
                            });
                        }
                    }
                    else {
                        tagsInGroup.push({
                            key: tagKey,
                            type: 'KEY_ONLY',
                        });
                    }
                }
                else {
                    if (tagValues.length > 0) {
                        for (const tagValue of tagValues) {
                            tagsInGroup.push({
                                value: tagValue,
                                type: 'VALUE_ONLY',
                            });
                        }
                    }
                    else {
                        throw new Error('Cannot specify both an empty key and no values for an instance tag filter');
                    }
                }
            }
        }
        return tagsInGroup;
    }
}
exports.ServerDeploymentGroup = ServerDeploymentGroup;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVwbG95bWVudC1ncm91cC5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImRlcGxveW1lbnQtZ3JvdXAudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBRUEsd0NBQXdDLENBQUMsbURBQW1EO0FBQzVGLHdDQUF3QyxDQUFDLG1EQUFtRDtBQUM1RixzQ0FBc0MsQ0FBQyxrREFBa0Q7QUFDekYscUNBQXFDLENBQUMsZ0RBQWdEO0FBRXRGLGtFQUE2RDtBQUU3RCxvQ0FBNEc7QUFDNUcsK0NBQXNFO0FBQ3RFLDJEQUFzRjtBQUN0RixtREFBdUU7QUFzQ3ZFOzs7Ozs7Ozs7R0FTRztBQUNILE1BQWUseUJBQTBCLFNBQVEsR0FBRyxDQUFDLFFBQVE7SUFPekQsWUFBWSxLQUFnQixFQUFFLEVBQVUsRUFBRSxnQkFBMEMsRUFBRSxLQUF5QjtRQUMzRyxLQUFLLENBQUMsS0FBSyxFQUFFLEVBQUUsRUFBRSxLQUFLLENBQUMsQ0FBQztRQUN4QixJQUFJLENBQUMsZ0JBQWdCLEdBQUcsZ0JBQWdCLElBQUksMENBQXNCLENBQUMsYUFBYSxDQUFDO0lBQ3JGLENBQUM7Q0FDSjtBQUNELE1BQU0sNkJBQThCLFNBQVEseUJBQXlCO0lBTWpFLFlBQVksS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBc0M7UUFDNUUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUM7UUFMN0IsU0FBSSxHQUFjLFNBQVMsQ0FBQztRQUc1QixzQkFBaUIsR0FBb0MsU0FBUyxDQUFDO1FBRzNFLElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsQ0FBQztRQUNyQyxJQUFJLENBQUMsbUJBQW1CLEdBQUcsS0FBSyxDQUFDLG1CQUFtQixDQUFDO1FBQ3JELElBQUksQ0FBQyxrQkFBa0IsR0FBRyw2QkFBcUIsQ0FBQyxLQUFLLENBQUMsV0FBVyxDQUFDLGVBQWUsRUFBRSxLQUFLLENBQUMsbUJBQW1CLENBQUMsQ0FBQztJQUNsSCxDQUFDO0NBQ0o7QUFlRDs7Ozs7R0FLRztBQUNILE1BQWEsY0FBYztJQUV2QixZQUFZLEdBQUcsaUJBQXFDO1FBQ2hELElBQUksaUJBQWlCLENBQUMsTUFBTSxHQUFHLENBQUMsRUFBRTtZQUM5QixNQUFNLElBQUksS0FBSyxDQUFDLG1FQUFtRTtnQkFDL0UsT0FBTyxpQkFBaUIsQ0FBQyxNQUFNLGdCQUFnQixDQUFDLENBQUM7U0FDeEQ7UUFDRCxJQUFJLENBQUMsa0JBQWtCLEdBQUcsaUJBQWlCLENBQUM7SUFDaEQsQ0FBQztJQUNELElBQVcsaUJBQWlCO1FBQ3hCLE9BQU8sSUFBSSxDQUFDLGtCQUFrQixDQUFDLEtBQUssRUFBRSxDQUFDO0lBQzNDLENBQUM7Q0FDSjtBQVpELHdDQVlDO0FBNkZEOzs7R0FHRztBQUNILE1BQWEscUJBQXNCLFNBQVEseUJBQXlCO0lBcUJoRSxZQUFZLEtBQWdCLEVBQUUsRUFBVSxFQUFFLFFBQW9DLEVBQUU7UUFDNUUsS0FBSyxDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLGdCQUFnQixFQUFFO1lBQ3JDLFlBQVksRUFBRSxLQUFLLENBQUMsbUJBQW1CO1NBQzFDLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxXQUFXLEdBQUcsS0FBSyxDQUFDLFdBQVcsSUFBSSxJQUFJLCtCQUFpQixDQUFDLElBQUksRUFBRSxhQUFhLENBQUMsQ0FBQztRQUNuRixJQUFJLENBQUMsSUFBSSxHQUFHLEtBQUssQ0FBQyxJQUFJLElBQUksSUFBSSxHQUFHLENBQUMsSUFBSSxDQUFDLElBQUksRUFBRSxNQUFNLEVBQUU7WUFDakQsU0FBUyxFQUFFLElBQUksR0FBRyxDQUFDLGdCQUFnQixDQUFDLDBCQUEwQixDQUFDO1lBQy9ELGVBQWUsRUFBRSxDQUFDLEdBQUcsQ0FBQyxhQUFhLENBQUMsd0JBQXdCLENBQUMsZ0NBQWdDLENBQUMsQ0FBQztTQUNsRyxDQUFDLENBQUM7UUFDSCxJQUFJLENBQUMsa0JBQWtCLEdBQUcsS0FBSyxDQUFDLGlCQUFpQixJQUFJLEVBQUUsQ0FBQztRQUN4RCxJQUFJLENBQUMsWUFBWSxHQUFHLEtBQUssQ0FBQyxZQUFZLEtBQUssU0FBUyxDQUFDLENBQUMsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUM7UUFDakYsSUFBSSxDQUFDLGdCQUFnQixHQUFHLEVBQUUsQ0FBQyxNQUFNLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxRQUFRLEVBQUUsa0JBQWtCLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sRUFBRSxDQUFDLENBQUM7UUFDaEgsS0FBSyxNQUFNLEdBQUcsSUFBSSxJQUFJLENBQUMsa0JBQWtCLEVBQUU7WUFDdkMsSUFBSSxDQUFDLGlDQUFpQyxDQUFDLEdBQUcsQ0FBQyxDQUFDO1NBQy9DO1FBQ0QsSUFBSSxDQUFDLE1BQU0sR0FBRyxLQUFLLENBQUMsTUFBTSxJQUFJLEVBQUUsQ0FBQztRQUNqQyxNQUFNLFFBQVEsR0FBRyxJQUFJLHlDQUFrQixDQUFDLElBQUksRUFBRSxVQUFVLEVBQUU7WUFDdEQsZUFBZSxFQUFFLElBQUksQ0FBQyxXQUFXLENBQUMsZUFBZTtZQUNqRCxtQkFBbUIsRUFBRSxJQUFJLENBQUMsWUFBWTtZQUN0QyxjQUFjLEVBQUUsSUFBSSxDQUFDLElBQUksQ0FBQyxPQUFPO1lBQ2pDLG9CQUFvQixFQUFFLEtBQUssQ0FBQyxnQkFBZ0I7Z0JBQ3hDLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxvQkFBb0I7WUFDL0MsaUJBQWlCLEVBQUUsR0FBRyxDQUFDLElBQUksQ0FBQyxTQUFTLENBQUMsRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLENBQUMsSUFBSSxDQUFDLGtCQUFrQixDQUFDLEdBQUcsQ0FBQyxHQUFHLENBQUMsRUFBRSxDQUFDLEdBQUcsQ0FBQyxvQkFBb0IsQ0FBQyxFQUFFLEVBQUUsRUFBRSxTQUFTLEVBQUUsSUFBSSxFQUFFLENBQUM7WUFDM0ksZ0JBQWdCLEVBQUUsSUFBSSxDQUFDLGdCQUFnQixDQUFDLEtBQUssQ0FBQyxZQUFZLENBQUM7WUFDM0QsZUFBZSxFQUFFLEtBQUssQ0FBQyxZQUFZLEtBQUssU0FBUztnQkFDN0MsQ0FBQyxDQUFDLFNBQVM7Z0JBQ1gsQ0FBQyxDQUFDO29CQUNFLGdCQUFnQixFQUFFLHNCQUFzQjtpQkFDM0M7WUFDTCxTQUFTLEVBQUUsSUFBSSxDQUFDLFNBQVMsQ0FBQyxLQUFLLENBQUMsZUFBZSxDQUFDO1lBQ2hELGdCQUFnQixFQUFFLElBQUksQ0FBQyxlQUFlLENBQUMsS0FBSyxDQUFDLHFCQUFxQixDQUFDO1lBQ25FLGtCQUFrQixFQUFFLEdBQUcsQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsT0FBTyxFQUFFLEdBQUcsRUFBRSxDQUFDLGdDQUF3QixDQUFDLElBQUksQ0FBQyxNQUFNLEVBQUUsS0FBSyxDQUFDLHVCQUF1QixDQUFDLEVBQUUsQ0FBQztZQUM5SCx5QkFBeUIsRUFBRSxHQUFHLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsQ0FBQyx1Q0FBK0IsQ0FBQyxJQUFJLENBQUMsTUFBTSxFQUFFLEtBQUssQ0FBQyxZQUFZLENBQUMsRUFBRSxDQUFDO1NBQ3BJLENBQUMsQ0FBQztRQUNILElBQUksQ0FBQyxtQkFBbUIsR0FBRyxJQUFJLENBQUMsd0JBQXdCLENBQUMsUUFBUSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ3ZFLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxJQUFJLENBQUMsdUJBQXVCLENBQUMsNkJBQXFCLENBQUMsSUFBSSxDQUFDLFdBQVcsQ0FBQyxlQUFlLEVBQUUsUUFBUSxDQUFDLEdBQUcsQ0FBQyxFQUFFO1lBQzFILE9BQU8sRUFBRSxZQUFZO1lBQ3JCLFFBQVEsRUFBRSxpQkFBaUI7WUFDM0IsWUFBWSxFQUFFLEdBQUcsSUFBSSxDQUFDLFdBQVcsQ0FBQyxlQUFlLElBQUksSUFBSSxDQUFDLFlBQVksRUFBRTtZQUN4RSxHQUFHLEVBQUUsR0FBRztTQUNYLENBQUMsQ0FBQztJQUNQLENBQUM7SUE3REQ7Ozs7Ozs7O09BUUc7SUFDSSxNQUFNLENBQUMsbUNBQW1DLENBQUMsS0FBZ0IsRUFBRSxFQUFVLEVBQUUsS0FBc0M7UUFDbEgsT0FBTyxJQUFJLDZCQUE2QixDQUFDLEtBQUssRUFBRSxFQUFFLEVBQUUsS0FBSyxDQUFDLENBQUM7SUFDL0QsQ0FBQztJQW1ERDs7Ozs7O09BTUc7SUFDSSxtQkFBbUIsQ0FBQyxHQUFpQztRQUN4RCxJQUFJLENBQUMsa0JBQWtCLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxDQUFDO1FBQ2xDLElBQUksQ0FBQyxpQ0FBaUMsQ0FBQyxHQUFHLENBQUMsQ0FBQztJQUNoRCxDQUFDO0lBQ0Q7Ozs7T0FJRztJQUNJLFFBQVEsQ0FBQyxLQUF3QjtRQUNwQyxJQUFJLENBQUMsTUFBTSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsQ0FBQztJQUM1QixDQUFDO0lBQ0QsSUFBVyxpQkFBaUI7UUFDeEIsT0FBTyxJQUFJLENBQUMsa0JBQWtCLENBQUMsS0FBSyxFQUFFLENBQUM7SUFDM0MsQ0FBQztJQUNPLGlDQUFpQyxDQUFDLEdBQWtDO1FBQ3hFLElBQUksQ0FBQyxJQUFJLENBQUMsWUFBWSxFQUFFO1lBQ3BCLE9BQU87U0FDVjtRQUNELElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxTQUFTLENBQUMsR0FBRyxFQUFFLFVBQVUsQ0FBQyxDQUFDO1FBQ2pELFFBQVEsR0FBRyxDQUFDLE1BQU0sRUFBRTtZQUNoQixLQUFLLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxLQUFLO2dCQUM5QixHQUFHLENBQUMsV0FBVyxDQUFDLGlDQUFpQyxFQUFFLDRCQUE0QixFQUFFLGlCQUFpQixFQUFFLE1BQU0sRUFBRSxhQUFhLEVBQUUsSUFBSSxFQUFFLG9CQUFvQixFQUFFLDZCQUE2QixFQUFFLHVCQUF1QixFQUFFLDBCQUEwQixFQUFFLElBQUksRUFBRSw0QkFBNEIsRUFBRSxxQkFBcUIsRUFBRSxhQUFhLEVBQUUsaUNBQWlDLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sOEJBQThCLEdBQUcsQ0FBQyxLQUFLLENBQUMsRUFBRSxDQUFDLElBQUksQ0FBQyxDQUFDLE1BQU0sRUFBRSxFQUFFLG9CQUFvQixFQUFFLGdCQUFnQixFQUFFLGlCQUFpQixDQUFDLENBQUM7Z0JBQ3JlLE1BQU07WUFDVixLQUFLLEdBQUcsQ0FBQyxtQkFBbUIsQ0FBQyxPQUFPO2dCQUNoQyxHQUFHLENBQUMsV0FBVyxDQUFDLHFFQUFxRSxFQUFFLGlDQUFpQyxHQUFHLENBQUMsS0FBSyxDQUFDLEVBQUUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxNQUFNLDZEQUE2RCxFQUFFLCtFQUErRSxDQUFDLENBQUM7Z0JBQ2pTLE1BQU07U0FDYjtJQUNMLENBQUM7SUFDTyxnQkFBZ0IsQ0FBQyxZQUEyQjtRQUNoRCxJQUFJLENBQUMsWUFBWSxFQUFFO1lBQ2YsT0FBTyxTQUFTLENBQUM7U0FDcEI7UUFDRCxRQUFRLFlBQVksQ0FBQyxVQUFVLEVBQUU7WUFDN0IsS0FBSyxzQ0FBc0IsQ0FBQyxLQUFLO2dCQUM3QixPQUFPO29CQUNILFdBQVcsRUFBRTt3QkFDVCxFQUFFLElBQUksRUFBRSxZQUFZLENBQUMsSUFBSSxFQUFFO3FCQUM5QjtpQkFDSixDQUFDO1lBQ04sS0FBSyxzQ0FBc0IsQ0FBQyxNQUFNO2dCQUM5QixPQUFPO29CQUNILG1CQUFtQixFQUFFO3dCQUNqQixFQUFFLElBQUksRUFBRSxZQUFZLENBQUMsSUFBSSxFQUFFO3FCQUM5QjtpQkFDSixDQUFDO1NBQ1Q7SUFDTCxDQUFDO0lBQ08sU0FBUyxDQUFDLE1BQXVCO1FBQ3JDLElBQUksQ0FBQyxNQUFNLElBQUksTUFBTSxDQUFDLGlCQUFpQixDQUFDLE1BQU0sS0FBSyxDQUFDLEVBQUU7WUFDbEQsT0FBTyxTQUFTLENBQUM7U0FDcEI7UUFDRCxPQUFPO1lBQ0gsYUFBYSxFQUFFLE1BQU0sQ0FBQyxpQkFBaUIsQ0FBQyxHQUFHLENBQUMsUUFBUSxDQUFDLEVBQUU7Z0JBQ25ELE9BQU87b0JBQ0gsV0FBVyxFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRLENBQThDO2lCQUM5RixDQUFDO1lBQ04sQ0FBQyxDQUFDO1NBQ0wsQ0FBQztJQUNOLENBQUM7SUFDTyxlQUFlLENBQUMsTUFBdUI7UUFDM0MsSUFBSSxDQUFDLE1BQU0sSUFBSSxNQUFNLENBQUMsaUJBQWlCLENBQUMsTUFBTSxLQUFLLENBQUMsRUFBRTtZQUNsRCxPQUFPLFNBQVMsQ0FBQztTQUNwQjtRQUNELE9BQU87WUFDSCxvQkFBb0IsRUFBRSxNQUFNLENBQUMsaUJBQWlCLENBQUMsR0FBRyxDQUFDLFFBQVEsQ0FBQyxFQUFFO2dCQUMxRCxPQUFPO29CQUNILGtCQUFrQixFQUFFLElBQUksQ0FBQyxrQkFBa0IsQ0FBQyxRQUFRLENBQUM7aUJBQ3hELENBQUM7WUFDTixDQUFDLENBQUM7U0FDTCxDQUFDO0lBQ04sQ0FBQztJQUNPLGtCQUFrQixDQUFDLFFBQTBCO1FBQ2pELE1BQU0sV0FBVyxHQUFHLElBQUksS0FBSyxFQUF3QyxDQUFDO1FBQ3RFLEtBQUssTUFBTSxNQUFNLElBQUksUUFBUSxFQUFFO1lBQzNCLElBQUksUUFBUSxDQUFDLGNBQWMsQ0FBQyxNQUFNLENBQUMsRUFBRTtnQkFDakMsTUFBTSxTQUFTLEdBQUcsUUFBUSxDQUFDLE1BQU0sQ0FBQyxDQUFDO2dCQUNuQyxJQUFJLE1BQU0sQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO29CQUNuQixJQUFJLFNBQVMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO3dCQUN0QixLQUFLLE1BQU0sUUFBUSxJQUFJLFNBQVMsRUFBRTs0QkFDOUIsV0FBVyxDQUFDLElBQUksQ0FBQztnQ0FDYixHQUFHLEVBQUUsTUFBTTtnQ0FDWCxLQUFLLEVBQUUsUUFBUTtnQ0FDZixJQUFJLEVBQUUsZUFBZTs2QkFDeEIsQ0FBQyxDQUFDO3lCQUNOO3FCQUNKO3lCQUNJO3dCQUNELFdBQVcsQ0FBQyxJQUFJLENBQUM7NEJBQ2IsR0FBRyxFQUFFLE1BQU07NEJBQ1gsSUFBSSxFQUFFLFVBQVU7eUJBQ25CLENBQUMsQ0FBQztxQkFDTjtpQkFDSjtxQkFDSTtvQkFDRCxJQUFJLFNBQVMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO3dCQUN0QixLQUFLLE1BQU0sUUFBUSxJQUFJLFNBQVMsRUFBRTs0QkFDOUIsV0FBVyxDQUFDLElBQUksQ0FBQztnQ0FDYixLQUFLLEVBQUUsUUFBUTtnQ0FDZixJQUFJLEVBQUUsWUFBWTs2QkFDckIsQ0FBQyxDQUFDO3lCQUNOO3FCQUNKO3lCQUNJO3dCQUNELE1BQU0sSUFBSSxLQUFLLENBQUMsMkVBQTJFLENBQUMsQ0FBQztxQkFDaEc7aUJBQ0o7YUFDSjtTQUNKO1FBQ0QsT0FBTyxXQUFXLENBQUM7SUFDdkIsQ0FBQztDQUNKO0FBckxELHNEQXFMQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGF1dG9zY2FsaW5nIGZyb20gXCIuLi8uLi8uLi9hd3MtYXV0b3NjYWxpbmdcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1hdXRvc2NhbGluZydcbmltcG9ydCAqIGFzIGNsb3Vkd2F0Y2ggZnJvbSBcIi4uLy4uLy4uL2F3cy1jbG91ZHdhdGNoXCI7IC8vIEF1dG9tYXRpY2FsbHkgcmUtd3JpdHRlbiBmcm9tICdAYXdzLWNkay9hd3MtY2xvdWR3YXRjaCdcbmltcG9ydCAqIGFzIGVjMiBmcm9tIFwiLi4vLi4vLi4vYXdzLWVjMlwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWVjMidcbmltcG9ydCAqIGFzIGlhbSBmcm9tIFwiLi4vLi4vLi4vYXdzLWlhbVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSdcbmltcG9ydCAqIGFzIHMzIGZyb20gXCIuLi8uLi8uLi9hd3MtczNcIjsgLy8gQXV0b21hdGljYWxseSByZS13cml0dGVuIGZyb20gJ0Bhd3MtY2RrL2F3cy1zMydcbmltcG9ydCAqIGFzIGNkayBmcm9tIFwiLi4vLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IENvbnN0cnVjdCB9IGZyb20gJ2NvbnN0cnVjdHMnO1xuaW1wb3J0IHsgQ2ZuRGVwbG95bWVudEdyb3VwIH0gZnJvbSAnLi4vY29kZWRlcGxveS5nZW5lcmF0ZWQnO1xuaW1wb3J0IHsgQXV0b1JvbGxiYWNrQ29uZmlnIH0gZnJvbSAnLi4vcm9sbGJhY2stY29uZmlnJztcbmltcG9ydCB7IGFybkZvckRlcGxveW1lbnRHcm91cCwgcmVuZGVyQWxhcm1Db25maWd1cmF0aW9uLCByZW5kZXJBdXRvUm9sbGJhY2tDb25maWd1cmF0aW9uIH0gZnJvbSAnLi4vdXRpbHMnO1xuaW1wb3J0IHsgSVNlcnZlckFwcGxpY2F0aW9uLCBTZXJ2ZXJBcHBsaWNhdGlvbiB9IGZyb20gJy4vYXBwbGljYXRpb24nO1xuaW1wb3J0IHsgSVNlcnZlckRlcGxveW1lbnRDb25maWcsIFNlcnZlckRlcGxveW1lbnRDb25maWcgfSBmcm9tICcuL2RlcGxveW1lbnQtY29uZmlnJztcbmltcG9ydCB7IExvYWRCYWxhbmNlciwgTG9hZEJhbGFuY2VyR2VuZXJhdGlvbiB9IGZyb20gJy4vbG9hZC1iYWxhbmNlcic7XG5leHBvcnQgaW50ZXJmYWNlIElTZXJ2ZXJEZXBsb3ltZW50R3JvdXAgZXh0ZW5kcyBjZGsuSVJlc291cmNlIHtcbiAgICByZWFkb25seSBhcHBsaWNhdGlvbjogSVNlcnZlckFwcGxpY2F0aW9uO1xuICAgIHJlYWRvbmx5IHJvbGU/OiBpYW0uSVJvbGU7XG4gICAgLyoqXG4gICAgICogQGF0dHJpYnV0ZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGRlcGxveW1lbnRHcm91cE5hbWU6IHN0cmluZztcbiAgICAvKipcbiAgICAgKiBAYXR0cmlidXRlXG4gICAgICovXG4gICAgcmVhZG9ubHkgZGVwbG95bWVudEdyb3VwQXJuOiBzdHJpbmc7XG4gICAgcmVhZG9ubHkgZGVwbG95bWVudENvbmZpZzogSVNlcnZlckRlcGxveW1lbnRDb25maWc7XG4gICAgcmVhZG9ubHkgYXV0b1NjYWxpbmdHcm91cHM/OiBhdXRvc2NhbGluZy5JQXV0b1NjYWxpbmdHcm91cFtdO1xufVxuLyoqXG4gKiBQcm9wZXJ0aWVzIG9mIGEgcmVmZXJlbmNlIHRvIGEgQ29kZURlcGxveSBFQzIvb24tcHJlbWlzZSBEZXBsb3ltZW50IEdyb3VwLlxuICpcbiAqIEBzZWUgU2VydmVyRGVwbG95bWVudEdyb3VwI2ltcG9ydFxuICovXG5leHBvcnQgaW50ZXJmYWNlIFNlcnZlckRlcGxveW1lbnRHcm91cEF0dHJpYnV0ZXMge1xuICAgIC8qKlxuICAgICAqIFRoZSByZWZlcmVuY2UgdG8gdGhlIENvZGVEZXBsb3kgRUMyL29uLXByZW1pc2UgQXBwbGljYXRpb25cbiAgICAgKiB0aGF0IHRoaXMgRGVwbG95bWVudCBHcm91cCBiZWxvbmdzIHRvLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGFwcGxpY2F0aW9uOiBJU2VydmVyQXBwbGljYXRpb247XG4gICAgLyoqXG4gICAgICogVGhlIHBoeXNpY2FsLCBodW1hbi1yZWFkYWJsZSBuYW1lIG9mIHRoZSBDb2RlRGVwbG95IEVDMi9vbi1wcmVtaXNlIERlcGxveW1lbnQgR3JvdXBcbiAgICAgKiB0aGF0IHdlIGFyZSByZWZlcmVuY2luZy5cbiAgICAgKi9cbiAgICByZWFkb25seSBkZXBsb3ltZW50R3JvdXBOYW1lOiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIERlcGxveW1lbnQgQ29uZmlndXJhdGlvbiB0aGlzIERlcGxveW1lbnQgR3JvdXAgdXNlcy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IFNlcnZlckRlcGxveW1lbnRDb25maWcjT25lQXRBVGltZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGRlcGxveW1lbnRDb25maWc/OiBJU2VydmVyRGVwbG95bWVudENvbmZpZztcbn1cbi8qKlxuICogUmVwcmVzZW50cyBhIHJlZmVyZW5jZSB0byBhIENvZGVEZXBsb3kgRUMyL29uLXByZW1pc2UgRGVwbG95bWVudCBHcm91cC5cbiAqXG4gKiBJZiB5b3UncmUgbWFuYWdpbmcgdGhlIERlcGxveW1lbnQgR3JvdXAgYWxvbmdzaWRlIHRoZSByZXN0IG9mIHlvdXIgQ0RLIHJlc291cmNlcyxcbiAqIHVzZSB0aGUge0BsaW5rIFNlcnZlckRlcGxveW1lbnRHcm91cH0gY2xhc3MuXG4gKlxuICogSWYgeW91IHdhbnQgdG8gcmVmZXJlbmNlIGFuIGFscmVhZHkgZXhpc3RpbmcgRGVwbG95bWVudCBHcm91cCxcbiAqIG9yIG9uZSBkZWZpbmVkIGluIGEgZGlmZmVyZW50IENESyBTdGFjayxcbiAqIHVzZSB0aGUge0BsaW5rICNpbXBvcnR9IG1ldGhvZC5cbiAqL1xuYWJzdHJhY3QgY2xhc3MgU2VydmVyRGVwbG95bWVudEdyb3VwQmFzZSBleHRlbmRzIGNkay5SZXNvdXJjZSBpbXBsZW1lbnRzIElTZXJ2ZXJEZXBsb3ltZW50R3JvdXAge1xuICAgIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBhcHBsaWNhdGlvbjogSVNlcnZlckFwcGxpY2F0aW9uO1xuICAgIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuICAgIHB1YmxpYyBhYnN0cmFjdCByZWFkb25seSBkZXBsb3ltZW50R3JvdXBOYW1lOiBzdHJpbmc7XG4gICAgcHVibGljIGFic3RyYWN0IHJlYWRvbmx5IGRlcGxveW1lbnRHcm91cEFybjogc3RyaW5nO1xuICAgIHB1YmxpYyByZWFkb25seSBkZXBsb3ltZW50Q29uZmlnOiBJU2VydmVyRGVwbG95bWVudENvbmZpZztcbiAgICBwdWJsaWMgYWJzdHJhY3QgcmVhZG9ubHkgYXV0b1NjYWxpbmdHcm91cHM/OiBhdXRvc2NhbGluZy5JQXV0b1NjYWxpbmdHcm91cFtdO1xuICAgIGNvbnN0cnVjdG9yKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGRlcGxveW1lbnRDb25maWc/OiBJU2VydmVyRGVwbG95bWVudENvbmZpZywgcHJvcHM/OiBjZGsuUmVzb3VyY2VQcm9wcykge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzKTtcbiAgICAgICAgdGhpcy5kZXBsb3ltZW50Q29uZmlnID0gZGVwbG95bWVudENvbmZpZyB8fCBTZXJ2ZXJEZXBsb3ltZW50Q29uZmlnLk9ORV9BVF9BX1RJTUU7XG4gICAgfVxufVxuY2xhc3MgSW1wb3J0ZWRTZXJ2ZXJEZXBsb3ltZW50R3JvdXAgZXh0ZW5kcyBTZXJ2ZXJEZXBsb3ltZW50R3JvdXBCYXNlIHtcbiAgICBwdWJsaWMgcmVhZG9ubHkgYXBwbGljYXRpb246IElTZXJ2ZXJBcHBsaWNhdGlvbjtcbiAgICBwdWJsaWMgcmVhZG9ubHkgcm9sZT86IGlhbS5Sb2xlID0gdW5kZWZpbmVkO1xuICAgIHB1YmxpYyByZWFkb25seSBkZXBsb3ltZW50R3JvdXBOYW1lOiBzdHJpbmc7XG4gICAgcHVibGljIHJlYWRvbmx5IGRlcGxveW1lbnRHcm91cEFybjogc3RyaW5nO1xuICAgIHB1YmxpYyByZWFkb25seSBhdXRvU2NhbGluZ0dyb3Vwcz86IGF1dG9zY2FsaW5nLkF1dG9TY2FsaW5nR3JvdXBbXSA9IHVuZGVmaW5lZDtcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogU2VydmVyRGVwbG95bWVudEdyb3VwQXR0cmlidXRlcykge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzLmRlcGxveW1lbnRDb25maWcpO1xuICAgICAgICB0aGlzLmFwcGxpY2F0aW9uID0gcHJvcHMuYXBwbGljYXRpb247XG4gICAgICAgIHRoaXMuZGVwbG95bWVudEdyb3VwTmFtZSA9IHByb3BzLmRlcGxveW1lbnRHcm91cE5hbWU7XG4gICAgICAgIHRoaXMuZGVwbG95bWVudEdyb3VwQXJuID0gYXJuRm9yRGVwbG95bWVudEdyb3VwKHByb3BzLmFwcGxpY2F0aW9uLmFwcGxpY2F0aW9uTmFtZSwgcHJvcHMuZGVwbG95bWVudEdyb3VwTmFtZSk7XG4gICAgfVxufVxuLyoqXG4gKiBSZXByZXNlbnRzIGEgZ3JvdXAgb2YgaW5zdGFuY2UgdGFncy5cbiAqIEFuIGluc3RhbmNlIHdpbGwgbWF0Y2ggYSBncm91cCBpZiBpdCBoYXMgYSB0YWcgbWF0Y2hpbmdcbiAqIGFueSBvZiB0aGUgZ3JvdXAncyB0YWdzIGJ5IGtleSBhbmQgYW55IG9mIHRoZSBwcm92aWRlZCB2YWx1ZXMgLVxuICogaW4gb3RoZXIgd29yZHMsIHRhZyBncm91cHMgZm9sbG93ICdvcicgc2VtYW50aWNzLlxuICogSWYgdGhlIHZhbHVlIGZvciBhIGdpdmVuIGtleSBpcyBhbiBlbXB0eSBhcnJheSxcbiAqIGFuIGluc3RhbmNlIHdpbGwgbWF0Y2ggd2hlbiBpdCBoYXMgYSB0YWcgd2l0aCB0aGUgZ2l2ZW4ga2V5LFxuICogcmVnYXJkbGVzcyBvZiB0aGUgdmFsdWUuXG4gKiBJZiB0aGUga2V5IGlzIGFuIGVtcHR5IHN0cmluZywgYW55IHRhZyxcbiAqIHJlZ2FyZGxlc3Mgb2YgaXRzIGtleSwgd2l0aCBhbnkgb2YgdGhlIGdpdmVuIHZhbHVlcywgd2lsbCBtYXRjaC5cbiAqL1xuZXhwb3J0IHR5cGUgSW5zdGFuY2VUYWdHcm91cCA9IHtcbiAgICBba2V5OiBzdHJpbmddOiBzdHJpbmdbXTtcbn07XG4vKipcbiAqIFJlcHJlc2VudHMgYSBzZXQgb2YgaW5zdGFuY2UgdGFnIGdyb3Vwcy5cbiAqIEFuIGluc3RhbmNlIHdpbGwgbWF0Y2ggYSBzZXQgaWYgaXQgbWF0Y2hlcyBhbGwgb2YgdGhlIGdyb3VwcyBpbiB0aGUgc2V0IC1cbiAqIGluIG90aGVyIHdvcmRzLCBzZXRzIGZvbGxvdyAnYW5kJyBzZW1hbnRpY3MuXG4gKiBZb3UgY2FuIGhhdmUgYSBtYXhpbXVtIG9mIDMgdGFnIGdyb3VwcyBpbnNpZGUgYSBzZXQuXG4gKi9cbmV4cG9ydCBjbGFzcyBJbnN0YW5jZVRhZ1NldCB7XG4gICAgcHJpdmF0ZSByZWFkb25seSBfaW5zdGFuY2VUYWdHcm91cHM6IEluc3RhbmNlVGFnR3JvdXBbXTtcbiAgICBjb25zdHJ1Y3RvciguLi5pbnN0YW5jZVRhZ0dyb3VwczogSW5zdGFuY2VUYWdHcm91cFtdKSB7XG4gICAgICAgIGlmIChpbnN0YW5jZVRhZ0dyb3Vwcy5sZW5ndGggPiAzKSB7XG4gICAgICAgICAgICB0aHJvdyBuZXcgRXJyb3IoJ0FuIGluc3RhbmNlIHRhZyBzZXQgY2FuIGhhdmUgYSBtYXhpbXVtIG9mIDMgaW5zdGFuY2UgdGFnIGdyb3VwcywgJyArXG4gICAgICAgICAgICAgICAgYGJ1dCAke2luc3RhbmNlVGFnR3JvdXBzLmxlbmd0aH0gd2VyZSBwcm92aWRlZGApO1xuICAgICAgICB9XG4gICAgICAgIHRoaXMuX2luc3RhbmNlVGFnR3JvdXBzID0gaW5zdGFuY2VUYWdHcm91cHM7XG4gICAgfVxuICAgIHB1YmxpYyBnZXQgaW5zdGFuY2VUYWdHcm91cHMoKTogSW5zdGFuY2VUYWdHcm91cFtdIHtcbiAgICAgICAgcmV0dXJuIHRoaXMuX2luc3RhbmNlVGFnR3JvdXBzLnNsaWNlKCk7XG4gICAgfVxufVxuLyoqXG4gKiBDb25zdHJ1Y3Rpb24gcHJvcGVydGllcyBmb3Ige0BsaW5rIFNlcnZlckRlcGxveW1lbnRHcm91cH0uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgU2VydmVyRGVwbG95bWVudEdyb3VwUHJvcHMge1xuICAgIC8qKlxuICAgICAqIFRoZSBDb2RlRGVwbG95IEVDMi9vbi1wcmVtaXNlIEFwcGxpY2F0aW9uIHRoaXMgRGVwbG95bWVudCBHcm91cCBiZWxvbmdzIHRvLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBBIG5ldyBBcHBsaWNhdGlvbiB3aWxsIGJlIGNyZWF0ZWQuXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXBwbGljYXRpb24/OiBJU2VydmVyQXBwbGljYXRpb247XG4gICAgLyoqXG4gICAgICogVGhlIHNlcnZpY2UgUm9sZSBvZiB0aGlzIERlcGxveW1lbnQgR3JvdXAuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIEEgbmV3IFJvbGUgd2lsbCBiZSBjcmVhdGVkLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IHJvbGU/OiBpYW0uSVJvbGU7XG4gICAgLyoqXG4gICAgICogVGhlIHBoeXNpY2FsLCBodW1hbi1yZWFkYWJsZSBuYW1lIG9mIHRoZSBDb2RlRGVwbG95IERlcGxveW1lbnQgR3JvdXAuXG4gICAgICpcbiAgICAgKiBAZGVmYXVsdCAtIEFuIGF1dG8tZ2VuZXJhdGVkIG5hbWUgd2lsbCBiZSB1c2VkLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGRlcGxveW1lbnRHcm91cE5hbWU/OiBzdHJpbmc7XG4gICAgLyoqXG4gICAgICogVGhlIEVDMi9vbi1wcmVtaXNlIERlcGxveW1lbnQgQ29uZmlndXJhdGlvbiB0byB1c2UgZm9yIHRoaXMgRGVwbG95bWVudCBHcm91cC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IFNlcnZlckRlcGxveW1lbnRDb25maWcjT25lQXRBVGltZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGRlcGxveW1lbnRDb25maWc/OiBJU2VydmVyRGVwbG95bWVudENvbmZpZztcbiAgICAvKipcbiAgICAgKiBUaGUgYXV0by1zY2FsaW5nIGdyb3VwcyBiZWxvbmdpbmcgdG8gdGhpcyBEZXBsb3ltZW50IEdyb3VwLlxuICAgICAqXG4gICAgICogQXV0by1zY2FsaW5nIGdyb3VwcyBjYW4gYWxzbyBiZSBhZGRlZCBhZnRlciB0aGUgRGVwbG95bWVudCBHcm91cCBpcyBjcmVhdGVkXG4gICAgICogdXNpbmcgdGhlIHtAbGluayAjYWRkQXV0b1NjYWxpbmdHcm91cH0gbWV0aG9kLlxuICAgICAqXG4gICAgICogW2Rpc2FibGUtYXdzbGludDpyZWYtdmlhLWludGVyZmFjZV0gaXMgbmVlZGVkIGJlY2F1c2Ugd2UgdXBkYXRlIHVzZXJkYXRhXG4gICAgICogZm9yIEFTR3MgdG8gaW5zdGFsbCB0aGUgY29kZWRlcGxveSBhZ2VudC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IFtdXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXV0b1NjYWxpbmdHcm91cHM/OiBhdXRvc2NhbGluZy5JQXV0b1NjYWxpbmdHcm91cFtdO1xuICAgIC8qKlxuICAgICAqIElmIHlvdSd2ZSBwcm92aWRlZCBhbnkgYXV0by1zY2FsaW5nIGdyb3VwcyB3aXRoIHRoZSB7QGxpbmsgI2F1dG9TY2FsaW5nR3JvdXBzfSBwcm9wZXJ0eSxcbiAgICAgKiB5b3UgY2FuIHNldCB0aGlzIHByb3BlcnR5IHRvIGFkZCBVc2VyIERhdGEgdGhhdCBpbnN0YWxscyB0aGUgQ29kZURlcGxveSBhZ2VudCBvbiB0aGUgaW5zdGFuY2VzLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgdHJ1ZVxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVkZXBsb3kvbGF0ZXN0L3VzZXJndWlkZS9jb2RlZGVwbG95LWFnZW50LW9wZXJhdGlvbnMtaW5zdGFsbC5odG1sXG4gICAgICovXG4gICAgcmVhZG9ubHkgaW5zdGFsbEFnZW50PzogYm9vbGVhbjtcbiAgICAvKipcbiAgICAgKiBUaGUgbG9hZCBiYWxhbmNlciB0byBwbGFjZSBpbiBmcm9udCBvZiB0aGlzIERlcGxveW1lbnQgR3JvdXAuXG4gICAgICogQ2FuIGJlIGNyZWF0ZWQgZnJvbSBlaXRoZXIgYSBjbGFzc2ljIEVsYXN0aWMgTG9hZCBCYWxhbmNlcixcbiAgICAgKiBvciBhbiBBcHBsaWNhdGlvbiBMb2FkIEJhbGFuY2VyIC8gTmV0d29yayBMb2FkIEJhbGFuY2VyIFRhcmdldCBHcm91cC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gRGVwbG95bWVudCBHcm91cCB3aWxsIG5vdCBoYXZlIGEgbG9hZCBiYWxhbmNlciBkZWZpbmVkLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IGxvYWRCYWxhbmNlcj86IExvYWRCYWxhbmNlcjtcbiAgICAvKipcbiAgICAgKiBBbGwgRUMyIGluc3RhbmNlcyBtYXRjaGluZyB0aGUgZ2l2ZW4gc2V0IG9mIHRhZ3Mgd2hlbiBhIGRlcGxveW1lbnQgb2NjdXJzIHdpbGwgYmUgYWRkZWQgdG8gdGhpcyBEZXBsb3ltZW50IEdyb3VwLlxuICAgICAqXG4gICAgICogQGRlZmF1bHQgLSBObyBhZGRpdGlvbmFsIEVDMiBpbnN0YW5jZXMgd2lsbCBiZSBhZGRlZCB0byB0aGUgRGVwbG95bWVudCBHcm91cC5cbiAgICAgKi9cbiAgICByZWFkb25seSBlYzJJbnN0YW5jZVRhZ3M/OiBJbnN0YW5jZVRhZ1NldDtcbiAgICAvKipcbiAgICAgKiBBbGwgb24tcHJlbWlzZSBpbnN0YW5jZXMgbWF0Y2hpbmcgdGhlIGdpdmVuIHNldCBvZiB0YWdzIHdoZW4gYSBkZXBsb3ltZW50IG9jY3VycyB3aWxsIGJlIGFkZGVkIHRvIHRoaXMgRGVwbG95bWVudCBHcm91cC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gTm8gYWRkaXRpb25hbCBvbi1wcmVtaXNlIGluc3RhbmNlcyB3aWxsIGJlIGFkZGVkIHRvIHRoZSBEZXBsb3ltZW50IEdyb3VwLlxuICAgICAqL1xuICAgIHJlYWRvbmx5IG9uUHJlbWlzZUluc3RhbmNlVGFncz86IEluc3RhbmNlVGFnU2V0O1xuICAgIC8qKlxuICAgICAqIFRoZSBDbG91ZFdhdGNoIGFsYXJtcyBhc3NvY2lhdGVkIHdpdGggdGhpcyBEZXBsb3ltZW50IEdyb3VwLlxuICAgICAqIENvZGVEZXBsb3kgd2lsbCBzdG9wIChhbmQgb3B0aW9uYWxseSByb2xsIGJhY2spXG4gICAgICogYSBkZXBsb3ltZW50IGlmIGR1cmluZyBpdCBhbnkgb2YgdGhlIGFsYXJtcyB0cmlnZ2VyLlxuICAgICAqXG4gICAgICogQWxhcm1zIGNhbiBhbHNvIGJlIGFkZGVkIGFmdGVyIHRoZSBEZXBsb3ltZW50IEdyb3VwIGlzIGNyZWF0ZWQgdXNpbmcgdGhlIHtAbGluayAjYWRkQWxhcm19IG1ldGhvZC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IFtdXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZWRlcGxveS9sYXRlc3QvdXNlcmd1aWRlL21vbml0b3JpbmctY3JlYXRlLWFsYXJtcy5odG1sXG4gICAgICovXG4gICAgcmVhZG9ubHkgYWxhcm1zPzogY2xvdWR3YXRjaC5JQWxhcm1bXTtcbiAgICAvKipcbiAgICAgKiBXaGV0aGVyIHRvIGNvbnRpbnVlIGEgZGVwbG95bWVudCBldmVuIGlmIGZldGNoaW5nIHRoZSBhbGFybSBzdGF0dXMgZnJvbSBDbG91ZFdhdGNoIGZhaWxlZC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IGZhbHNlXG4gICAgICovXG4gICAgcmVhZG9ubHkgaWdub3JlUG9sbEFsYXJtc0ZhaWx1cmU/OiBib29sZWFuO1xuICAgIC8qKlxuICAgICAqIFRoZSBhdXRvLXJvbGxiYWNrIGNvbmZpZ3VyYXRpb24gZm9yIHRoaXMgRGVwbG95bWVudCBHcm91cC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IC0gZGVmYXVsdCBBdXRvUm9sbGJhY2tDb25maWcuXG4gICAgICovXG4gICAgcmVhZG9ubHkgYXV0b1JvbGxiYWNrPzogQXV0b1JvbGxiYWNrQ29uZmlnO1xufVxuLyoqXG4gKiBBIENvZGVEZXBsb3kgRGVwbG95bWVudCBHcm91cCB0aGF0IGRlcGxveXMgdG8gRUMyL29uLXByZW1pc2UgaW5zdGFuY2VzLlxuICogQHJlc291cmNlIEFXUzo6Q29kZURlcGxveTo6RGVwbG95bWVudEdyb3VwXG4gKi9cbmV4cG9ydCBjbGFzcyBTZXJ2ZXJEZXBsb3ltZW50R3JvdXAgZXh0ZW5kcyBTZXJ2ZXJEZXBsb3ltZW50R3JvdXBCYXNlIHtcbiAgICAvKipcbiAgICAgKiBJbXBvcnQgYW4gRUMyL29uLXByZW1pc2UgRGVwbG95bWVudCBHcm91cCBkZWZpbmVkIGVpdGhlciBvdXRzaWRlIHRoZSBDREsgYXBwLFxuICAgICAqIG9yIGluIGEgZGlmZmVyZW50IHJlZ2lvbi5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBzY29wZSB0aGUgcGFyZW50IENvbnN0cnVjdCBmb3IgdGhpcyBuZXcgQ29uc3RydWN0XG4gICAgICogQHBhcmFtIGlkIHRoZSBsb2dpY2FsIElEIG9mIHRoaXMgbmV3IENvbnN0cnVjdFxuICAgICAqIEBwYXJhbSBhdHRycyB0aGUgcHJvcGVydGllcyBvZiB0aGUgcmVmZXJlbmNlZCBEZXBsb3ltZW50IEdyb3VwXG4gICAgICogQHJldHVybnMgYSBDb25zdHJ1Y3QgcmVwcmVzZW50aW5nIGEgcmVmZXJlbmNlIHRvIGFuIGV4aXN0aW5nIERlcGxveW1lbnQgR3JvdXBcbiAgICAgKi9cbiAgICBwdWJsaWMgc3RhdGljIGZyb21TZXJ2ZXJEZXBsb3ltZW50R3JvdXBBdHRyaWJ1dGVzKHNjb3BlOiBDb25zdHJ1Y3QsIGlkOiBzdHJpbmcsIGF0dHJzOiBTZXJ2ZXJEZXBsb3ltZW50R3JvdXBBdHRyaWJ1dGVzKTogSVNlcnZlckRlcGxveW1lbnRHcm91cCB7XG4gICAgICAgIHJldHVybiBuZXcgSW1wb3J0ZWRTZXJ2ZXJEZXBsb3ltZW50R3JvdXAoc2NvcGUsIGlkLCBhdHRycyk7XG4gICAgfVxuICAgIHB1YmxpYyByZWFkb25seSBhcHBsaWNhdGlvbjogSVNlcnZlckFwcGxpY2F0aW9uO1xuICAgIHB1YmxpYyByZWFkb25seSByb2xlPzogaWFtLklSb2xlO1xuICAgIHB1YmxpYyByZWFkb25seSBkZXBsb3ltZW50R3JvdXBBcm46IHN0cmluZztcbiAgICBwdWJsaWMgcmVhZG9ubHkgZGVwbG95bWVudEdyb3VwTmFtZTogc3RyaW5nO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgX2F1dG9TY2FsaW5nR3JvdXBzOiBhdXRvc2NhbGluZy5JQXV0b1NjYWxpbmdHcm91cFtdO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgaW5zdGFsbEFnZW50OiBib29sZWFuO1xuICAgIHByaXZhdGUgcmVhZG9ubHkgY29kZURlcGxveUJ1Y2tldDogczMuSUJ1Y2tldDtcbiAgICBwcml2YXRlIHJlYWRvbmx5IGFsYXJtczogY2xvdWR3YXRjaC5JQWxhcm1bXTtcbiAgICBjb25zdHJ1Y3RvcihzY29wZTogQ29uc3RydWN0LCBpZDogc3RyaW5nLCBwcm9wczogU2VydmVyRGVwbG95bWVudEdyb3VwUHJvcHMgPSB7fSkge1xuICAgICAgICBzdXBlcihzY29wZSwgaWQsIHByb3BzLmRlcGxveW1lbnRDb25maWcsIHtcbiAgICAgICAgICAgIHBoeXNpY2FsTmFtZTogcHJvcHMuZGVwbG95bWVudEdyb3VwTmFtZSxcbiAgICAgICAgfSk7XG4gICAgICAgIHRoaXMuYXBwbGljYXRpb24gPSBwcm9wcy5hcHBsaWNhdGlvbiB8fCBuZXcgU2VydmVyQXBwbGljYXRpb24odGhpcywgJ0FwcGxpY2F0aW9uJyk7XG4gICAgICAgIHRoaXMucm9sZSA9IHByb3BzLnJvbGUgfHwgbmV3IGlhbS5Sb2xlKHRoaXMsICdSb2xlJywge1xuICAgICAgICAgICAgYXNzdW1lZEJ5OiBuZXcgaWFtLlNlcnZpY2VQcmluY2lwYWwoJ2NvZGVkZXBsb3kuYW1hem9uYXdzLmNvbScpLFxuICAgICAgICAgICAgbWFuYWdlZFBvbGljaWVzOiBbaWFtLk1hbmFnZWRQb2xpY3kuZnJvbUF3c01hbmFnZWRQb2xpY3lOYW1lKCdzZXJ2aWNlLXJvbGUvQVdTQ29kZURlcGxveVJvbGUnKV0sXG4gICAgICAgIH0pO1xuICAgICAgICB0aGlzLl9hdXRvU2NhbGluZ0dyb3VwcyA9IHByb3BzLmF1dG9TY2FsaW5nR3JvdXBzIHx8IFtdO1xuICAgICAgICB0aGlzLmluc3RhbGxBZ2VudCA9IHByb3BzLmluc3RhbGxBZ2VudCA9PT0gdW5kZWZpbmVkID8gdHJ1ZSA6IHByb3BzLmluc3RhbGxBZ2VudDtcbiAgICAgICAgdGhpcy5jb2RlRGVwbG95QnVja2V0ID0gczMuQnVja2V0LmZyb21CdWNrZXROYW1lKHRoaXMsICdCdWNrZXQnLCBgYXdzLWNvZGVkZXBsb3ktJHtjZGsuU3RhY2sub2YodGhpcykucmVnaW9ufWApO1xuICAgICAgICBmb3IgKGNvbnN0IGFzZyBvZiB0aGlzLl9hdXRvU2NhbGluZ0dyb3Vwcykge1xuICAgICAgICAgICAgdGhpcy5hZGRDb2RlRGVwbG95QWdlbnRJbnN0YWxsVXNlckRhdGEoYXNnKTtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLmFsYXJtcyA9IHByb3BzLmFsYXJtcyB8fCBbXTtcbiAgICAgICAgY29uc3QgcmVzb3VyY2UgPSBuZXcgQ2ZuRGVwbG95bWVudEdyb3VwKHRoaXMsICdSZXNvdXJjZScsIHtcbiAgICAgICAgICAgIGFwcGxpY2F0aW9uTmFtZTogdGhpcy5hcHBsaWNhdGlvbi5hcHBsaWNhdGlvbk5hbWUsXG4gICAgICAgICAgICBkZXBsb3ltZW50R3JvdXBOYW1lOiB0aGlzLnBoeXNpY2FsTmFtZSxcbiAgICAgICAgICAgIHNlcnZpY2VSb2xlQXJuOiB0aGlzLnJvbGUucm9sZUFybixcbiAgICAgICAgICAgIGRlcGxveW1lbnRDb25maWdOYW1lOiBwcm9wcy5kZXBsb3ltZW50Q29uZmlnICYmXG4gICAgICAgICAgICAgICAgcHJvcHMuZGVwbG95bWVudENvbmZpZy5kZXBsb3ltZW50Q29uZmlnTmFtZSxcbiAgICAgICAgICAgIGF1dG9TY2FsaW5nR3JvdXBzOiBjZGsuTGF6eS5saXN0VmFsdWUoeyBwcm9kdWNlOiAoKSA9PiB0aGlzLl9hdXRvU2NhbGluZ0dyb3Vwcy5tYXAoYXNnID0+IGFzZy5hdXRvU2NhbGluZ0dyb3VwTmFtZSkgfSwgeyBvbWl0RW1wdHk6IHRydWUgfSksXG4gICAgICAgICAgICBsb2FkQmFsYW5jZXJJbmZvOiB0aGlzLmxvYWRCYWxhbmNlckluZm8ocHJvcHMubG9hZEJhbGFuY2VyKSxcbiAgICAgICAgICAgIGRlcGxveW1lbnRTdHlsZTogcHJvcHMubG9hZEJhbGFuY2VyID09PSB1bmRlZmluZWRcbiAgICAgICAgICAgICAgICA/IHVuZGVmaW5lZFxuICAgICAgICAgICAgICAgIDoge1xuICAgICAgICAgICAgICAgICAgICBkZXBsb3ltZW50T3B0aW9uOiAnV0lUSF9UUkFGRklDX0NPTlRST0wnLFxuICAgICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICBlYzJUYWdTZXQ6IHRoaXMuZWMyVGFnU2V0KHByb3BzLmVjMkluc3RhbmNlVGFncyksXG4gICAgICAgICAgICBvblByZW1pc2VzVGFnU2V0OiB0aGlzLm9uUHJlbWlzZVRhZ1NldChwcm9wcy5vblByZW1pc2VJbnN0YW5jZVRhZ3MpLFxuICAgICAgICAgICAgYWxhcm1Db25maWd1cmF0aW9uOiBjZGsuTGF6eS5hbnlWYWx1ZSh7IHByb2R1Y2U6ICgpID0+IHJlbmRlckFsYXJtQ29uZmlndXJhdGlvbih0aGlzLmFsYXJtcywgcHJvcHMuaWdub3JlUG9sbEFsYXJtc0ZhaWx1cmUpIH0pLFxuICAgICAgICAgICAgYXV0b1JvbGxiYWNrQ29uZmlndXJhdGlvbjogY2RrLkxhenkuYW55VmFsdWUoeyBwcm9kdWNlOiAoKSA9PiByZW5kZXJBdXRvUm9sbGJhY2tDb25maWd1cmF0aW9uKHRoaXMuYWxhcm1zLCBwcm9wcy5hdXRvUm9sbGJhY2spIH0pLFxuICAgICAgICB9KTtcbiAgICAgICAgdGhpcy5kZXBsb3ltZW50R3JvdXBOYW1lID0gdGhpcy5nZXRSZXNvdXJjZU5hbWVBdHRyaWJ1dGUocmVzb3VyY2UucmVmKTtcbiAgICAgICAgdGhpcy5kZXBsb3ltZW50R3JvdXBBcm4gPSB0aGlzLmdldFJlc291cmNlQXJuQXR0cmlidXRlKGFybkZvckRlcGxveW1lbnRHcm91cCh0aGlzLmFwcGxpY2F0aW9uLmFwcGxpY2F0aW9uTmFtZSwgcmVzb3VyY2UucmVmKSwge1xuICAgICAgICAgICAgc2VydmljZTogJ2NvZGVkZXBsb3knLFxuICAgICAgICAgICAgcmVzb3VyY2U6ICdkZXBsb3ltZW50Z3JvdXAnLFxuICAgICAgICAgICAgcmVzb3VyY2VOYW1lOiBgJHt0aGlzLmFwcGxpY2F0aW9uLmFwcGxpY2F0aW9uTmFtZX0vJHt0aGlzLnBoeXNpY2FsTmFtZX1gLFxuICAgICAgICAgICAgc2VwOiAnOicsXG4gICAgICAgIH0pO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBZGRzIGFuIGFkZGl0aW9uYWwgYXV0by1zY2FsaW5nIGdyb3VwIHRvIHRoaXMgRGVwbG95bWVudCBHcm91cC5cbiAgICAgKlxuICAgICAqIEBwYXJhbSBhc2cgdGhlIGF1dG8tc2NhbGluZyBncm91cCB0byBhZGQgdG8gdGhpcyBEZXBsb3ltZW50IEdyb3VwLlxuICAgICAqIFtkaXNhYmxlLWF3c2xpbnQ6cmVmLXZpYS1pbnRlcmZhY2VdIGlzIG5lZWRlZCBpbiBvcmRlciB0byBpbnN0YWxsIHRoZSBjb2RlXG4gICAgICogZGVwbG95IGFnZW50IGJ5IHVwZGF0aW5nIHRoZSBBU0dzIHVzZXIgZGF0YS5cbiAgICAgKi9cbiAgICBwdWJsaWMgYWRkQXV0b1NjYWxpbmdHcm91cChhc2c6IGF1dG9zY2FsaW5nLkF1dG9TY2FsaW5nR3JvdXApOiB2b2lkIHtcbiAgICAgICAgdGhpcy5fYXV0b1NjYWxpbmdHcm91cHMucHVzaChhc2cpO1xuICAgICAgICB0aGlzLmFkZENvZGVEZXBsb3lBZ2VudEluc3RhbGxVc2VyRGF0YShhc2cpO1xuICAgIH1cbiAgICAvKipcbiAgICAgKiBBc3NvY2lhdGVzIGFuIGFkZGl0aW9uYWwgYWxhcm0gd2l0aCB0aGlzIERlcGxveW1lbnQgR3JvdXAuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gYWxhcm0gdGhlIGFsYXJtIHRvIGFzc29jaWF0ZSB3aXRoIHRoaXMgRGVwbG95bWVudCBHcm91cFxuICAgICAqL1xuICAgIHB1YmxpYyBhZGRBbGFybShhbGFybTogY2xvdWR3YXRjaC5JQWxhcm0pOiB2b2lkIHtcbiAgICAgICAgdGhpcy5hbGFybXMucHVzaChhbGFybSk7XG4gICAgfVxuICAgIHB1YmxpYyBnZXQgYXV0b1NjYWxpbmdHcm91cHMoKTogYXV0b3NjYWxpbmcuSUF1dG9TY2FsaW5nR3JvdXBbXSB8IHVuZGVmaW5lZCB7XG4gICAgICAgIHJldHVybiB0aGlzLl9hdXRvU2NhbGluZ0dyb3Vwcy5zbGljZSgpO1xuICAgIH1cbiAgICBwcml2YXRlIGFkZENvZGVEZXBsb3lBZ2VudEluc3RhbGxVc2VyRGF0YShhc2c6IGF1dG9zY2FsaW5nLklBdXRvU2NhbGluZ0dyb3VwKTogdm9pZCB7XG4gICAgICAgIGlmICghdGhpcy5pbnN0YWxsQWdlbnQpIHtcbiAgICAgICAgICAgIHJldHVybjtcbiAgICAgICAgfVxuICAgICAgICB0aGlzLmNvZGVEZXBsb3lCdWNrZXQuZ3JhbnRSZWFkKGFzZywgJ2xhdGVzdC8qJyk7XG4gICAgICAgIHN3aXRjaCAoYXNnLm9zVHlwZSkge1xuICAgICAgICAgICAgY2FzZSBlYzIuT3BlcmF0aW5nU3lzdGVtVHlwZS5MSU5VWDpcbiAgICAgICAgICAgICAgICBhc2cuYWRkVXNlckRhdGEoJ1BLR19DTUQ9YHdoaWNoIHl1bSAyPi9kZXYvbnVsbGAnLCAnaWYgWyAteiBcIiRQS0dfQ01EXCIgXTsgdGhlbicsICdQS0dfQ01EPWFwdC1nZXQnLCAnZWxzZScsICdQS0c9Q01EPXl1bScsICdmaScsICckUEtHX0NNRCB1cGRhdGUgLXknLCAnJFBLR19DTUQgaW5zdGFsbCAteSBydWJ5Mi4wJywgJ2lmIFsgJD8gLW5lIDAgXTsgdGhlbicsICckUEtHX0NNRCBpbnN0YWxsIC15IHJ1YnknLCAnZmknLCAnJFBLR19DTUQgaW5zdGFsbCAteSBhd3NjbGknLCAnVE1QX0RJUj1gbWt0ZW1wIC1kYCcsICdjZCAkVE1QX0RJUicsIGBhd3MgczMgY3AgczM6Ly9hd3MtY29kZWRlcGxveS0ke2Nkay5TdGFjay5vZih0aGlzKS5yZWdpb259L2xhdGVzdC9pbnN0YWxsIC4gLS1yZWdpb24gJHtjZGsuU3RhY2sub2YodGhpcykucmVnaW9ufWAsICdjaG1vZCAreCAuL2luc3RhbGwnLCAnLi9pbnN0YWxsIGF1dG8nLCAncm0gLWZyICRUTVBfRElSJyk7XG4gICAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgICAgICBjYXNlIGVjMi5PcGVyYXRpbmdTeXN0ZW1UeXBlLldJTkRPV1M6XG4gICAgICAgICAgICAgICAgYXNnLmFkZFVzZXJEYXRhKCdTZXQtVmFyaWFibGUgLU5hbWUgVEVNUERJUiAtVmFsdWUgKE5ldy1UZW1wb3JhcnlGaWxlKS5EaXJlY3RvcnlOYW1lJywgYGF3cyBzMyBjcCBzMzovL2F3cy1jb2RlZGVwbG95LSR7Y2RrLlN0YWNrLm9mKHRoaXMpLnJlZ2lvbn0vbGF0ZXN0L2NvZGVkZXBsb3ktYWdlbnQubXNpICRURU1QRElSXFxcXGNvZGVkZXBsb3ktYWdlbnQubXNpYCwgJyRURU1QRElSXFxcXGNvZGVkZXBsb3ktYWdlbnQubXNpIC9xdWlldCAvbCBjOlxcXFx0ZW1wXFxcXGhvc3QtYWdlbnQtaW5zdGFsbC1sb2cudHh0Jyk7XG4gICAgICAgICAgICAgICAgYnJlYWs7XG4gICAgICAgIH1cbiAgICB9XG4gICAgcHJpdmF0ZSBsb2FkQmFsYW5jZXJJbmZvKGxvYWRCYWxhbmNlcj86IExvYWRCYWxhbmNlcik6IENmbkRlcGxveW1lbnRHcm91cC5Mb2FkQmFsYW5jZXJJbmZvUHJvcGVydHkgfCB1bmRlZmluZWQge1xuICAgICAgICBpZiAoIWxvYWRCYWxhbmNlcikge1xuICAgICAgICAgICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgICAgICAgfVxuICAgICAgICBzd2l0Y2ggKGxvYWRCYWxhbmNlci5nZW5lcmF0aW9uKSB7XG4gICAgICAgICAgICBjYXNlIExvYWRCYWxhbmNlckdlbmVyYXRpb24uRklSU1Q6XG4gICAgICAgICAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgICAgICAgICAgZWxiSW5mb0xpc3Q6IFtcbiAgICAgICAgICAgICAgICAgICAgICAgIHsgbmFtZTogbG9hZEJhbGFuY2VyLm5hbWUgfSxcbiAgICAgICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgICB9O1xuICAgICAgICAgICAgY2FzZSBMb2FkQmFsYW5jZXJHZW5lcmF0aW9uLlNFQ09ORDpcbiAgICAgICAgICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgICAgICAgICB0YXJnZXRHcm91cEluZm9MaXN0OiBbXG4gICAgICAgICAgICAgICAgICAgICAgICB7IG5hbWU6IGxvYWRCYWxhbmNlci5uYW1lIH0sXG4gICAgICAgICAgICAgICAgICAgIF0sXG4gICAgICAgICAgICAgICAgfTtcbiAgICAgICAgfVxuICAgIH1cbiAgICBwcml2YXRlIGVjMlRhZ1NldCh0YWdTZXQ/OiBJbnN0YW5jZVRhZ1NldCk6IENmbkRlcGxveW1lbnRHcm91cC5FQzJUYWdTZXRQcm9wZXJ0eSB8IHVuZGVmaW5lZCB7XG4gICAgICAgIGlmICghdGFnU2V0IHx8IHRhZ1NldC5pbnN0YW5jZVRhZ0dyb3Vwcy5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICAgIHJldHVybiB1bmRlZmluZWQ7XG4gICAgICAgIH1cbiAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgIGVjMlRhZ1NldExpc3Q6IHRhZ1NldC5pbnN0YW5jZVRhZ0dyb3Vwcy5tYXAodGFnR3JvdXAgPT4ge1xuICAgICAgICAgICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICAgICAgICAgIGVjMlRhZ0dyb3VwOiB0aGlzLnRhZ0dyb3VwMlRhZ3NBcnJheSh0YWdHcm91cCkgYXMgQ2ZuRGVwbG95bWVudEdyb3VwLkVDMlRhZ0ZpbHRlclByb3BlcnR5W10sXG4gICAgICAgICAgICAgICAgfTtcbiAgICAgICAgICAgIH0pLFxuICAgICAgICB9O1xuICAgIH1cbiAgICBwcml2YXRlIG9uUHJlbWlzZVRhZ1NldCh0YWdTZXQ/OiBJbnN0YW5jZVRhZ1NldCk6IENmbkRlcGxveW1lbnRHcm91cC5PblByZW1pc2VzVGFnU2V0UHJvcGVydHkgfCB1bmRlZmluZWQge1xuICAgICAgICBpZiAoIXRhZ1NldCB8fCB0YWdTZXQuaW5zdGFuY2VUYWdHcm91cHMubGVuZ3RoID09PSAwKSB7XG4gICAgICAgICAgICByZXR1cm4gdW5kZWZpbmVkO1xuICAgICAgICB9XG4gICAgICAgIHJldHVybiB7XG4gICAgICAgICAgICBvblByZW1pc2VzVGFnU2V0TGlzdDogdGFnU2V0Lmluc3RhbmNlVGFnR3JvdXBzLm1hcCh0YWdHcm91cCA9PiB7XG4gICAgICAgICAgICAgICAgcmV0dXJuIHtcbiAgICAgICAgICAgICAgICAgICAgb25QcmVtaXNlc1RhZ0dyb3VwOiB0aGlzLnRhZ0dyb3VwMlRhZ3NBcnJheSh0YWdHcm91cCksXG4gICAgICAgICAgICAgICAgfTtcbiAgICAgICAgICAgIH0pLFxuICAgICAgICB9O1xuICAgIH1cbiAgICBwcml2YXRlIHRhZ0dyb3VwMlRhZ3NBcnJheSh0YWdHcm91cDogSW5zdGFuY2VUYWdHcm91cCk6IENmbkRlcGxveW1lbnRHcm91cC5UYWdGaWx0ZXJQcm9wZXJ0eVtdIHtcbiAgICAgICAgY29uc3QgdGFnc0luR3JvdXAgPSBuZXcgQXJyYXk8Q2ZuRGVwbG95bWVudEdyb3VwLlRhZ0ZpbHRlclByb3BlcnR5PigpO1xuICAgICAgICBmb3IgKGNvbnN0IHRhZ0tleSBpbiB0YWdHcm91cCkge1xuICAgICAgICAgICAgaWYgKHRhZ0dyb3VwLmhhc093blByb3BlcnR5KHRhZ0tleSkpIHtcbiAgICAgICAgICAgICAgICBjb25zdCB0YWdWYWx1ZXMgPSB0YWdHcm91cFt0YWdLZXldO1xuICAgICAgICAgICAgICAgIGlmICh0YWdLZXkubGVuZ3RoID4gMCkge1xuICAgICAgICAgICAgICAgICAgICBpZiAodGFnVmFsdWVzLmxlbmd0aCA+IDApIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIGZvciAoY29uc3QgdGFnVmFsdWUgb2YgdGFnVmFsdWVzKSB7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgdGFnc0luR3JvdXAucHVzaCh7XG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIGtleTogdGFnS2V5LFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICB2YWx1ZTogdGFnVmFsdWUsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIHR5cGU6ICdLRVlfQU5EX1ZBTFVFJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICB9KTtcbiAgICAgICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIHRhZ3NJbkdyb3VwLnB1c2goe1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIGtleTogdGFnS2V5LFxuICAgICAgICAgICAgICAgICAgICAgICAgICAgIHR5cGU6ICdLRVlfT05MWScsXG4gICAgICAgICAgICAgICAgICAgICAgICB9KTtcbiAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgICAgICAgICAgaWYgKHRhZ1ZhbHVlcy5sZW5ndGggPiAwKSB7XG4gICAgICAgICAgICAgICAgICAgICAgICBmb3IgKGNvbnN0IHRhZ1ZhbHVlIG9mIHRhZ1ZhbHVlcykge1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgIHRhZ3NJbkdyb3VwLnB1c2goe1xuICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICB2YWx1ZTogdGFnVmFsdWUsXG4gICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgIHR5cGU6ICdWQUxVRV9PTkxZJyxcbiAgICAgICAgICAgICAgICAgICAgICAgICAgICB9KTtcbiAgICAgICAgICAgICAgICAgICAgICAgIH1cbiAgICAgICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgICAgICAgICBlbHNlIHtcbiAgICAgICAgICAgICAgICAgICAgICAgIHRocm93IG5ldyBFcnJvcignQ2Fubm90IHNwZWNpZnkgYm90aCBhbiBlbXB0eSBrZXkgYW5kIG5vIHZhbHVlcyBmb3IgYW4gaW5zdGFuY2UgdGFnIGZpbHRlcicpO1xuICAgICAgICAgICAgICAgICAgICB9XG4gICAgICAgICAgICAgICAgfVxuICAgICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICAgIHJldHVybiB0YWdzSW5Hcm91cDtcbiAgICB9XG59XG4iXX0=