"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LoadBalancer = exports.LoadBalancerGeneration = void 0;
/**
 * The generations of AWS load balancing solutions.
 */
var LoadBalancerGeneration;
(function (LoadBalancerGeneration) {
    /**
     * The first generation (ELB Classic).
     */
    LoadBalancerGeneration[LoadBalancerGeneration["FIRST"] = 0] = "FIRST";
    /**
     * The second generation (ALB and NLB).
     */
    LoadBalancerGeneration[LoadBalancerGeneration["SECOND"] = 1] = "SECOND";
})(LoadBalancerGeneration = exports.LoadBalancerGeneration || (exports.LoadBalancerGeneration = {}));
/**
 * An interface of an abstract load balancer, as needed by CodeDeploy.
 * Create instances using the static factory methods:
 * {@link #classic}, {@link #application} and {@link #network}.
 */
class LoadBalancer {
    /**
     * Creates a new CodeDeploy load balancer from a Classic ELB Load Balancer.
     *
     * @param loadBalancer a classic ELB Load Balancer
     */
    static classic(loadBalancer) {
        class ClassicLoadBalancer extends LoadBalancer {
            constructor() {
                super(...arguments);
                this.generation = LoadBalancerGeneration.FIRST;
                this.name = loadBalancer.loadBalancerName;
            }
        }
        return new ClassicLoadBalancer();
    }
    /**
     * Creates a new CodeDeploy load balancer from an Application Load Balancer Target Group.
     *
     * @param albTargetGroup an ALB Target Group
     */
    static application(albTargetGroup) {
        class AlbLoadBalancer extends LoadBalancer {
            constructor() {
                super(...arguments);
                this.generation = LoadBalancerGeneration.SECOND;
                this.name = albTargetGroup.targetGroupName;
            }
        }
        return new AlbLoadBalancer();
    }
    /**
     * Creates a new CodeDeploy load balancer from a Network Load Balancer Target Group.
     *
     * @param nlbTargetGroup an NLB Target Group
     */
    static network(nlbTargetGroup) {
        class NlbLoadBalancer extends LoadBalancer {
            constructor() {
                super(...arguments);
                this.generation = LoadBalancerGeneration.SECOND;
                this.name = nlbTargetGroup.targetGroupName;
            }
        }
        return new NlbLoadBalancer();
    }
}
exports.LoadBalancer = LoadBalancer;
//# sourceMappingURL=data:application/json;base64,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