"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProfilingGroup = exports.ComputePlatform = void 0;
const aws_iam_1 = require("../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../core"); // Automatically re-written from '@aws-cdk/core'
const codeguruprofiler_generated_1 = require("./codeguruprofiler.generated");
/**
 * The compute platform of the profiling group.
 */
var ComputePlatform;
(function (ComputePlatform) {
    /**
     * Use AWS_LAMBDA if your application runs on AWS Lambda.
     */
    ComputePlatform["AWS_LAMBDA"] = "AWSLambda";
    /**
     * Use Default if your application runs on a compute platform that is not AWS Lambda,
     * such an Amazon EC2 instance, an on-premises server, or a different platform.
     */
    ComputePlatform["DEFAULT"] = "Default";
})(ComputePlatform = exports.ComputePlatform || (exports.ComputePlatform = {}));
class ProfilingGroupBase extends core_1.Resource {
    /**
     * Grant access to publish profiling information to the Profiling Group to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - codeguru-profiler:ConfigureAgent
     *  - codeguru-profiler:PostAgentProfile
     *
     * @param grantee Principal to grant publish rights to
     */
    grantPublish(grantee) {
        // https://docs.aws.amazon.com/codeguru/latest/profiler-ug/security-iam.html#security-iam-access-control
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['codeguru-profiler:ConfigureAgent', 'codeguru-profiler:PostAgentProfile'],
            resourceArns: [this.profilingGroupArn],
        });
    }
    /**
     * Grant access to read profiling information from the Profiling Group to the given identity.
     *
     * This will grant the following permissions:
     *
     *  - codeguru-profiler:GetProfile
     *  - codeguru-profiler:DescribeProfilingGroup
     *
     * @param grantee Principal to grant read rights to
     */
    grantRead(grantee) {
        // https://docs.aws.amazon.com/codeguru/latest/profiler-ug/security-iam.html#security-iam-access-control
        return aws_iam_1.Grant.addToPrincipal({
            grantee,
            actions: ['codeguru-profiler:GetProfile', 'codeguru-profiler:DescribeProfilingGroup'],
            resourceArns: [this.profilingGroupArn],
        });
    }
}
/**
 * A new Profiling Group.
 */
class ProfilingGroup extends ProfilingGroupBase {
    constructor(scope, id, props = {}) {
        var _a;
        super(scope, id, {
            physicalName: (_a = props.profilingGroupName) !== null && _a !== void 0 ? _a : core_1.Lazy.stringValue({ produce: () => this.generateUniqueId() }),
        });
        const profilingGroup = new codeguruprofiler_generated_1.CfnProfilingGroup(this, 'ProfilingGroup', {
            profilingGroupName: this.physicalName,
            computePlatform: props.computePlatform,
        });
        this.profilingGroupName = this.getResourceNameAttribute(profilingGroup.ref);
        this.profilingGroupArn = this.getResourceArnAttribute(profilingGroup.attrArn, {
            service: 'codeguru-profiler',
            resource: 'profilingGroup',
            resourceName: this.physicalName,
        });
    }
    /**
     * Import an existing Profiling Group provided a Profiling Group Name.
     *
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param profilingGroupName Profiling Group Name
     */
    static fromProfilingGroupName(scope, id, profilingGroupName) {
        const stack = core_1.Stack.of(scope);
        return this.fromProfilingGroupArn(scope, id, stack.formatArn({
            service: 'codeguru-profiler',
            resource: 'profilingGroup',
            resourceName: profilingGroupName,
        }));
    }
    /**
     * Import an existing Profiling Group provided an ARN.
     *
     * @param scope The parent creating construct
     * @param id The construct's name
     * @param profilingGroupArn Profiling Group ARN
     */
    static fromProfilingGroupArn(scope, id, profilingGroupArn) {
        class Import extends ProfilingGroupBase {
            constructor() {
                super(...arguments);
                this.profilingGroupName = core_1.Stack.of(scope).parseArn(profilingGroupArn).resource;
                this.profilingGroupArn = profilingGroupArn;
            }
        }
        return new Import(scope, id);
    }
    generateUniqueId() {
        const name = this.node.uniqueId;
        if (name.length > 240) {
            return name.substring(0, 120) + name.substring(name.length - 120);
        }
        return name;
    }
}
exports.ProfilingGroup = ProfilingGroup;
//# sourceMappingURL=data:application/json;base64,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