"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CodeCommitSourceAction = exports.CodeCommitTrigger = void 0;
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const targets = require("../../../aws-events-targets"); // Automatically re-written from '@aws-cdk/aws-events-targets'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const action_1 = require("../action");
const common_1 = require("../common");
/**
 * How should the CodeCommit Action detect changes.
 * This is the type of the {@link CodeCommitSourceAction.trigger} property.
 */
var CodeCommitTrigger;
(function (CodeCommitTrigger) {
    /**
     * The Action will never detect changes -
     * the Pipeline it's part of will only begin a run when explicitly started.
     */
    CodeCommitTrigger["NONE"] = "None";
    /**
     * CodePipeline will poll the repository to detect changes.
     */
    CodeCommitTrigger["POLL"] = "Poll";
    /**
     * CodePipeline will use CloudWatch Events to be notified of changes.
     * This is the default method of detecting changes.
     */
    CodeCommitTrigger["EVENTS"] = "Events";
})(CodeCommitTrigger = exports.CodeCommitTrigger || (exports.CodeCommitTrigger = {}));
/**
 * CodePipeline Source that is provided by an AWS CodeCommit repository.
 */
class CodeCommitSourceAction extends action_1.Action {
    constructor(props) {
        var _a;
        const branch = (_a = props.branch) !== null && _a !== void 0 ? _a : 'master';
        if (!branch) {
            throw new Error("'branch' parameter cannot be an empty string");
        }
        super({
            ...props,
            resource: props.repository,
            category: codepipeline.ActionCategory.SOURCE,
            provider: 'CodeCommit',
            artifactBounds: common_1.sourceArtifactBounds(),
            outputs: [props.output],
        });
        this.branch = branch;
        this.props = props;
    }
    /** The variables emitted by this action. */
    get variables() {
        return {
            repositoryName: this.variableExpression('RepositoryName'),
            branchName: this.variableExpression('BranchName'),
            authorDate: this.variableExpression('AuthorDate'),
            committerDate: this.variableExpression('CommitterDate'),
            commitId: this.variableExpression('CommitId'),
            commitMessage: this.variableExpression('CommitMessage'),
        };
    }
    bound(_scope, stage, options) {
        const createEvent = this.props.trigger === undefined ||
            this.props.trigger === CodeCommitTrigger.EVENTS;
        if (createEvent) {
            const eventId = this.generateEventId(stage);
            this.props.repository.onCommit(eventId, {
                target: new targets.CodePipeline(stage.pipeline),
                branches: [this.branch],
            });
        }
        // the Action will write the contents of the Git repository to the Bucket,
        // so its Role needs write permissions to the Pipeline Bucket
        options.bucket.grantReadWrite(options.role);
        // https://docs.aws.amazon.com/codecommit/latest/userguide/auth-and-access-control-permissions-reference.html#aa-acp
        options.role.addToPolicy(new iam.PolicyStatement({
            resources: [this.props.repository.repositoryArn],
            actions: [
                'codecommit:GetBranch',
                'codecommit:GetCommit',
                'codecommit:UploadArchive',
                'codecommit:GetUploadArchiveStatus',
                'codecommit:CancelUploadArchive',
            ],
        }));
        return {
            configuration: {
                RepositoryName: this.props.repository.repositoryName,
                BranchName: this.branch,
                PollForSourceChanges: this.props.trigger === CodeCommitTrigger.POLL,
            },
        };
    }
    generateEventId(stage) {
        const baseId = stage.pipeline.node.uniqueId;
        if (core_1.Token.isUnresolved(this.branch)) {
            let candidate = '';
            let counter = 0;
            do {
                candidate = this.eventIdFromPrefix(`${baseId}${counter}`);
                counter += 1;
            } while (this.props.repository.node.tryFindChild(candidate) !== undefined);
            return candidate;
        }
        else {
            const branchIdDisambiguator = this.branch === 'master' ? '' : '-${this.branch}-';
            return this.eventIdFromPrefix(`${baseId}${branchIdDisambiguator}`);
        }
    }
    eventIdFromPrefix(eventIdPrefix) {
        return `${eventIdPrefix}EventRule`;
    }
}
exports.CodeCommitSourceAction = CodeCommitSourceAction;
//# sourceMappingURL=data:application/json;base64,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