"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.LambdaInvokeAction = void 0;
const codepipeline = require("../../../aws-codepipeline"); // Automatically re-written from '@aws-cdk/aws-codepipeline'
const iam = require("../../../aws-iam"); // Automatically re-written from '@aws-cdk/aws-iam'
const core_1 = require("../../../core"); // Automatically re-written from '@aws-cdk/core'
const action_1 = require("../action");
/**
 * CodePipeline invoke Action that is provided by an AWS Lambda function.
 *
 * @see https://docs.aws.amazon.com/codepipeline/latest/userguide/actions-invoke-lambda-function.html
 */
class LambdaInvokeAction extends action_1.Action {
    constructor(props) {
        super({
            ...props,
            resource: props.lambda,
            category: codepipeline.ActionCategory.INVOKE,
            provider: 'Lambda',
            artifactBounds: {
                minInputs: 0,
                maxInputs: 5,
                minOutputs: 0,
                maxOutputs: 5,
            },
        });
        this.props = props;
    }
    /**
     * Reference a CodePipeline variable defined by the Lambda function this action points to.
     * Variables in Lambda invoke actions are defined by calling the PutJobSuccessResult CodePipeline API call
     * with the 'outputVariables' property filled.
     *
     * @param variableName the name of the variable to reference.
     *   A variable by this name must be present in the 'outputVariables' section of the PutJobSuccessResult
     *   request that the Lambda function calls when the action is invoked
     *
     * @see https://docs.aws.amazon.com/codepipeline/latest/APIReference/API_PutJobSuccessResult.html
     */
    variable(variableName) {
        return this.variableExpression(variableName);
    }
    bound(scope, _stage, options) {
        // allow pipeline to list functions
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['lambda:ListFunctions'],
            resources: ['*'],
        }));
        // allow pipeline to invoke this lambda functionn
        options.role.addToPolicy(new iam.PolicyStatement({
            actions: ['lambda:InvokeFunction'],
            resources: [this.props.lambda.functionArn],
        }));
        // allow the Role access to the Bucket, if there are any inputs/outputs
        if ((this.actionProperties.inputs || []).length > 0) {
            options.bucket.grantRead(options.role);
        }
        if ((this.actionProperties.outputs || []).length > 0) {
            options.bucket.grantWrite(options.role);
        }
        // allow lambda to put job results for this pipeline
        // CodePipeline requires this to be granted to '*'
        // (the Pipeline ARN will not be enough)
        this.props.lambda.addToRolePolicy(new iam.PolicyStatement({
            resources: ['*'],
            actions: ['codepipeline:PutJobSuccessResult', 'codepipeline:PutJobFailureResult'],
        }));
        return {
            configuration: {
                FunctionName: this.props.lambda.functionName,
                UserParameters: core_1.Stack.of(scope).toJsonString(this.props.userParameters),
            },
        };
    }
}
exports.LambdaInvokeAction = LambdaInvokeAction;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiaW52b2tlLWFjdGlvbi5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbImludm9rZS1hY3Rpb24udHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7O0FBQUEsMERBQTBELENBQUMsNERBQTREO0FBQ3ZILHdDQUF3QyxDQUFDLG1EQUFtRDtBQUU1Rix3Q0FBaUQsQ0FBQyxnREFBZ0Q7QUFDbEcsc0NBQW1DO0FBdUNuQzs7OztHQUlHO0FBQ0gsTUFBYSxrQkFBbUIsU0FBUSxlQUFNO0lBRTFDLFlBQVksS0FBOEI7UUFDdEMsS0FBSyxDQUFDO1lBQ0YsR0FBRyxLQUFLO1lBQ1IsUUFBUSxFQUFFLEtBQUssQ0FBQyxNQUFNO1lBQ3RCLFFBQVEsRUFBRSxZQUFZLENBQUMsY0FBYyxDQUFDLE1BQU07WUFDNUMsUUFBUSxFQUFFLFFBQVE7WUFDbEIsY0FBYyxFQUFFO2dCQUNaLFNBQVMsRUFBRSxDQUFDO2dCQUNaLFNBQVMsRUFBRSxDQUFDO2dCQUNaLFVBQVUsRUFBRSxDQUFDO2dCQUNiLFVBQVUsRUFBRSxDQUFDO2FBQ2hCO1NBQ0osQ0FBQyxDQUFDO1FBQ0gsSUFBSSxDQUFDLEtBQUssR0FBRyxLQUFLLENBQUM7SUFDdkIsQ0FBQztJQUNEOzs7Ozs7Ozs7O09BVUc7SUFDSSxRQUFRLENBQUMsWUFBb0I7UUFDaEMsT0FBTyxJQUFJLENBQUMsa0JBQWtCLENBQUMsWUFBWSxDQUFDLENBQUM7SUFDakQsQ0FBQztJQUNTLEtBQUssQ0FBQyxLQUFnQixFQUFFLE1BQTJCLEVBQUUsT0FBdUM7UUFDbEcsbUNBQW1DO1FBQ25DLE9BQU8sQ0FBQyxJQUFJLENBQUMsV0FBVyxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUM3QyxPQUFPLEVBQUUsQ0FBQyxzQkFBc0IsQ0FBQztZQUNqQyxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7U0FDbkIsQ0FBQyxDQUFDLENBQUM7UUFDSixpREFBaUQ7UUFDakQsT0FBTyxDQUFDLElBQUksQ0FBQyxXQUFXLENBQUMsSUFBSSxHQUFHLENBQUMsZUFBZSxDQUFDO1lBQzdDLE9BQU8sRUFBRSxDQUFDLHVCQUF1QixDQUFDO1lBQ2xDLFNBQVMsRUFBRSxDQUFDLElBQUksQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLFdBQVcsQ0FBQztTQUM3QyxDQUFDLENBQUMsQ0FBQztRQUNKLHVFQUF1RTtRQUN2RSxJQUFJLENBQUMsSUFBSSxDQUFDLGdCQUFnQixDQUFDLE1BQU0sSUFBSSxFQUFFLENBQUMsQ0FBQyxNQUFNLEdBQUcsQ0FBQyxFQUFFO1lBQ2pELE9BQU8sQ0FBQyxNQUFNLENBQUMsU0FBUyxDQUFDLE9BQU8sQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUMxQztRQUNELElBQUksQ0FBQyxJQUFJLENBQUMsZ0JBQWdCLENBQUMsT0FBTyxJQUFJLEVBQUUsQ0FBQyxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7WUFDbEQsT0FBTyxDQUFDLE1BQU0sQ0FBQyxVQUFVLENBQUMsT0FBTyxDQUFDLElBQUksQ0FBQyxDQUFDO1NBQzNDO1FBQ0Qsb0RBQW9EO1FBQ3BELGtEQUFrRDtRQUNsRCx3Q0FBd0M7UUFDeEMsSUFBSSxDQUFDLEtBQUssQ0FBQyxNQUFNLENBQUMsZUFBZSxDQUFDLElBQUksR0FBRyxDQUFDLGVBQWUsQ0FBQztZQUN0RCxTQUFTLEVBQUUsQ0FBQyxHQUFHLENBQUM7WUFDaEIsT0FBTyxFQUFFLENBQUMsa0NBQWtDLEVBQUUsa0NBQWtDLENBQUM7U0FDcEYsQ0FBQyxDQUFDLENBQUM7UUFDSixPQUFPO1lBQ0gsYUFBYSxFQUFFO2dCQUNYLFlBQVksRUFBRSxJQUFJLENBQUMsS0FBSyxDQUFDLE1BQU0sQ0FBQyxZQUFZO2dCQUM1QyxjQUFjLEVBQUUsWUFBSyxDQUFDLEVBQUUsQ0FBQyxLQUFLLENBQUMsQ0FBQyxZQUFZLENBQUMsSUFBSSxDQUFDLEtBQUssQ0FBQyxjQUFjLENBQUM7YUFDMUU7U0FDSixDQUFDO0lBQ04sQ0FBQztDQUNKO0FBL0RELGdEQStEQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNvZGVwaXBlbGluZSBmcm9tIFwiLi4vLi4vLi4vYXdzLWNvZGVwaXBlbGluZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWNvZGVwaXBlbGluZSdcbmltcG9ydCAqIGFzIGlhbSBmcm9tIFwiLi4vLi4vLi4vYXdzLWlhbVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWlhbSdcbmltcG9ydCAqIGFzIGxhbWJkYSBmcm9tIFwiLi4vLi4vLi4vYXdzLWxhbWJkYVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvYXdzLWxhbWJkYSdcbmltcG9ydCB7IENvbnN0cnVjdCwgU3RhY2sgfSBmcm9tIFwiLi4vLi4vLi4vY29yZVwiOyAvLyBBdXRvbWF0aWNhbGx5IHJlLXdyaXR0ZW4gZnJvbSAnQGF3cy1jZGsvY29yZSdcbmltcG9ydCB7IEFjdGlvbiB9IGZyb20gJy4uL2FjdGlvbic7XG4vKipcbiAqIENvbnN0cnVjdGlvbiBwcm9wZXJ0aWVzIG9mIHRoZSB7QGxpbmsgTGFtYmRhSW52b2tlQWN0aW9uIExhbWJkYSBpbnZva2UgQ29kZVBpcGVsaW5lIEFjdGlvbn0uXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTGFtYmRhSW52b2tlQWN0aW9uUHJvcHMgZXh0ZW5kcyBjb2RlcGlwZWxpbmUuQ29tbW9uQXdzQWN0aW9uUHJvcHMge1xuICAgIC8qKlxuICAgICAqIFRoZSBvcHRpb25hbCBpbnB1dCBBcnRpZmFjdHMgb2YgdGhlIEFjdGlvbi5cbiAgICAgKiBBIExhbWJkYSBBY3Rpb24gY2FuIGhhdmUgdXAgdG8gNSBpbnB1dHMuXG4gICAgICogVGhlIGlucHV0cyB3aWxsIGFwcGVhciBpbiB0aGUgZXZlbnQgcGFzc2VkIHRvIHRoZSBMYW1iZGEsXG4gICAgICogdW5kZXIgdGhlIGAnQ29kZVBpcGVsaW5lLmpvYicuZGF0YS5pbnB1dEFydGlmYWN0c2AgcGF0aC5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IHRoZSBBY3Rpb24gd2lsbCBub3QgaGF2ZSBhbnkgaW5wdXRzXG4gICAgICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZXBpcGVsaW5lL2xhdGVzdC91c2VyZ3VpZGUvYWN0aW9ucy1pbnZva2UtbGFtYmRhLWZ1bmN0aW9uLmh0bWwjYWN0aW9ucy1pbnZva2UtbGFtYmRhLWZ1bmN0aW9uLWpzb24tZXZlbnQtZXhhbXBsZVxuICAgICAqL1xuICAgIHJlYWRvbmx5IGlucHV0cz86IGNvZGVwaXBlbGluZS5BcnRpZmFjdFtdO1xuICAgIC8qKlxuICAgICAqIFRoZSBvcHRpb25hbCBuYW1lcyBvZiB0aGUgb3V0cHV0IEFydGlmYWN0cyBvZiB0aGUgQWN0aW9uLlxuICAgICAqIEEgTGFtYmRhIEFjdGlvbiBjYW4gaGF2ZSB1cCB0byA1IG91dHB1dHMuXG4gICAgICogVGhlIG91dHB1dHMgd2lsbCBhcHBlYXIgaW4gdGhlIGV2ZW50IHBhc3NlZCB0byB0aGUgTGFtYmRhLFxuICAgICAqIHVuZGVyIHRoZSBgJ0NvZGVQaXBlbGluZS5qb2InLmRhdGEub3V0cHV0QXJ0aWZhY3RzYCBwYXRoLlxuICAgICAqIEl0IGlzIHRoZSByZXNwb25zaWJpbGl0eSBvZiB0aGUgTGFtYmRhIHRvIHVwbG9hZCBaSVAgZmlsZXMgd2l0aCB0aGUgQXJ0aWZhY3QgY29udGVudHMgdG8gdGhlIHByb3ZpZGVkIGxvY2F0aW9ucy5cbiAgICAgKlxuICAgICAqIEBkZWZhdWx0IHRoZSBBY3Rpb24gd2lsbCBub3QgaGF2ZSBhbnkgb3V0cHV0c1xuICAgICAqL1xuICAgIHJlYWRvbmx5IG91dHB1dHM/OiBjb2RlcGlwZWxpbmUuQXJ0aWZhY3RbXTtcbiAgICAvKipcbiAgICAgKiBBIHNldCBvZiBrZXktdmFsdWUgcGFpcnMgdGhhdCB3aWxsIGJlIGFjY2Vzc2libGUgdG8gdGhlIGludm9rZWQgTGFtYmRhXG4gICAgICogaW5zaWRlIHRoZSBldmVudCB0aGF0IHRoZSBQaXBlbGluZSB3aWxsIGNhbGwgaXQgd2l0aC5cbiAgICAgKlxuICAgICAqIEBzZWUgaHR0cHM6Ly9kb2NzLmF3cy5hbWF6b24uY29tL2NvZGVwaXBlbGluZS9sYXRlc3QvdXNlcmd1aWRlL2FjdGlvbnMtaW52b2tlLWxhbWJkYS1mdW5jdGlvbi5odG1sI2FjdGlvbnMtaW52b2tlLWxhbWJkYS1mdW5jdGlvbi1qc29uLWV2ZW50LWV4YW1wbGVcbiAgICAgKi9cbiAgICByZWFkb25seSB1c2VyUGFyYW1ldGVycz86IHtcbiAgICAgICAgW2tleTogc3RyaW5nXTogYW55O1xuICAgIH07XG4gICAgLyoqXG4gICAgICogVGhlIGxhbWJkYSBmdW5jdGlvbiB0byBpbnZva2UuXG4gICAgICovXG4gICAgcmVhZG9ubHkgbGFtYmRhOiBsYW1iZGEuSUZ1bmN0aW9uO1xufVxuLyoqXG4gKiBDb2RlUGlwZWxpbmUgaW52b2tlIEFjdGlvbiB0aGF0IGlzIHByb3ZpZGVkIGJ5IGFuIEFXUyBMYW1iZGEgZnVuY3Rpb24uXG4gKlxuICogQHNlZSBodHRwczovL2RvY3MuYXdzLmFtYXpvbi5jb20vY29kZXBpcGVsaW5lL2xhdGVzdC91c2VyZ3VpZGUvYWN0aW9ucy1pbnZva2UtbGFtYmRhLWZ1bmN0aW9uLmh0bWxcbiAqL1xuZXhwb3J0IGNsYXNzIExhbWJkYUludm9rZUFjdGlvbiBleHRlbmRzIEFjdGlvbiB7XG4gICAgcHJpdmF0ZSByZWFkb25seSBwcm9wczogTGFtYmRhSW52b2tlQWN0aW9uUHJvcHM7XG4gICAgY29uc3RydWN0b3IocHJvcHM6IExhbWJkYUludm9rZUFjdGlvblByb3BzKSB7XG4gICAgICAgIHN1cGVyKHtcbiAgICAgICAgICAgIC4uLnByb3BzLFxuICAgICAgICAgICAgcmVzb3VyY2U6IHByb3BzLmxhbWJkYSxcbiAgICAgICAgICAgIGNhdGVnb3J5OiBjb2RlcGlwZWxpbmUuQWN0aW9uQ2F0ZWdvcnkuSU5WT0tFLFxuICAgICAgICAgICAgcHJvdmlkZXI6ICdMYW1iZGEnLFxuICAgICAgICAgICAgYXJ0aWZhY3RCb3VuZHM6IHtcbiAgICAgICAgICAgICAgICBtaW5JbnB1dHM6IDAsXG4gICAgICAgICAgICAgICAgbWF4SW5wdXRzOiA1LFxuICAgICAgICAgICAgICAgIG1pbk91dHB1dHM6IDAsXG4gICAgICAgICAgICAgICAgbWF4T3V0cHV0czogNSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgIH0pO1xuICAgICAgICB0aGlzLnByb3BzID0gcHJvcHM7XG4gICAgfVxuICAgIC8qKlxuICAgICAqIFJlZmVyZW5jZSBhIENvZGVQaXBlbGluZSB2YXJpYWJsZSBkZWZpbmVkIGJ5IHRoZSBMYW1iZGEgZnVuY3Rpb24gdGhpcyBhY3Rpb24gcG9pbnRzIHRvLlxuICAgICAqIFZhcmlhYmxlcyBpbiBMYW1iZGEgaW52b2tlIGFjdGlvbnMgYXJlIGRlZmluZWQgYnkgY2FsbGluZyB0aGUgUHV0Sm9iU3VjY2Vzc1Jlc3VsdCBDb2RlUGlwZWxpbmUgQVBJIGNhbGxcbiAgICAgKiB3aXRoIHRoZSAnb3V0cHV0VmFyaWFibGVzJyBwcm9wZXJ0eSBmaWxsZWQuXG4gICAgICpcbiAgICAgKiBAcGFyYW0gdmFyaWFibGVOYW1lIHRoZSBuYW1lIG9mIHRoZSB2YXJpYWJsZSB0byByZWZlcmVuY2UuXG4gICAgICogICBBIHZhcmlhYmxlIGJ5IHRoaXMgbmFtZSBtdXN0IGJlIHByZXNlbnQgaW4gdGhlICdvdXRwdXRWYXJpYWJsZXMnIHNlY3Rpb24gb2YgdGhlIFB1dEpvYlN1Y2Nlc3NSZXN1bHRcbiAgICAgKiAgIHJlcXVlc3QgdGhhdCB0aGUgTGFtYmRhIGZ1bmN0aW9uIGNhbGxzIHdoZW4gdGhlIGFjdGlvbiBpcyBpbnZva2VkXG4gICAgICpcbiAgICAgKiBAc2VlIGh0dHBzOi8vZG9jcy5hd3MuYW1hem9uLmNvbS9jb2RlcGlwZWxpbmUvbGF0ZXN0L0FQSVJlZmVyZW5jZS9BUElfUHV0Sm9iU3VjY2Vzc1Jlc3VsdC5odG1sXG4gICAgICovXG4gICAgcHVibGljIHZhcmlhYmxlKHZhcmlhYmxlTmFtZTogc3RyaW5nKTogc3RyaW5nIHtcbiAgICAgICAgcmV0dXJuIHRoaXMudmFyaWFibGVFeHByZXNzaW9uKHZhcmlhYmxlTmFtZSk7XG4gICAgfVxuICAgIHByb3RlY3RlZCBib3VuZChzY29wZTogQ29uc3RydWN0LCBfc3RhZ2U6IGNvZGVwaXBlbGluZS5JU3RhZ2UsIG9wdGlvbnM6IGNvZGVwaXBlbGluZS5BY3Rpb25CaW5kT3B0aW9ucyk6IGNvZGVwaXBlbGluZS5BY3Rpb25Db25maWcge1xuICAgICAgICAvLyBhbGxvdyBwaXBlbGluZSB0byBsaXN0IGZ1bmN0aW9uc1xuICAgICAgICBvcHRpb25zLnJvbGUuYWRkVG9Qb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgYWN0aW9uczogWydsYW1iZGE6TGlzdEZ1bmN0aW9ucyddLFxuICAgICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgfSkpO1xuICAgICAgICAvLyBhbGxvdyBwaXBlbGluZSB0byBpbnZva2UgdGhpcyBsYW1iZGEgZnVuY3Rpb25uXG4gICAgICAgIG9wdGlvbnMucm9sZS5hZGRUb1BvbGljeShuZXcgaWFtLlBvbGljeVN0YXRlbWVudCh7XG4gICAgICAgICAgICBhY3Rpb25zOiBbJ2xhbWJkYTpJbnZva2VGdW5jdGlvbiddLFxuICAgICAgICAgICAgcmVzb3VyY2VzOiBbdGhpcy5wcm9wcy5sYW1iZGEuZnVuY3Rpb25Bcm5dLFxuICAgICAgICB9KSk7XG4gICAgICAgIC8vIGFsbG93IHRoZSBSb2xlIGFjY2VzcyB0byB0aGUgQnVja2V0LCBpZiB0aGVyZSBhcmUgYW55IGlucHV0cy9vdXRwdXRzXG4gICAgICAgIGlmICgodGhpcy5hY3Rpb25Qcm9wZXJ0aWVzLmlucHV0cyB8fCBbXSkubGVuZ3RoID4gMCkge1xuICAgICAgICAgICAgb3B0aW9ucy5idWNrZXQuZ3JhbnRSZWFkKG9wdGlvbnMucm9sZSk7XG4gICAgICAgIH1cbiAgICAgICAgaWYgKCh0aGlzLmFjdGlvblByb3BlcnRpZXMub3V0cHV0cyB8fCBbXSkubGVuZ3RoID4gMCkge1xuICAgICAgICAgICAgb3B0aW9ucy5idWNrZXQuZ3JhbnRXcml0ZShvcHRpb25zLnJvbGUpO1xuICAgICAgICB9XG4gICAgICAgIC8vIGFsbG93IGxhbWJkYSB0byBwdXQgam9iIHJlc3VsdHMgZm9yIHRoaXMgcGlwZWxpbmVcbiAgICAgICAgLy8gQ29kZVBpcGVsaW5lIHJlcXVpcmVzIHRoaXMgdG8gYmUgZ3JhbnRlZCB0byAnKidcbiAgICAgICAgLy8gKHRoZSBQaXBlbGluZSBBUk4gd2lsbCBub3QgYmUgZW5vdWdoKVxuICAgICAgICB0aGlzLnByb3BzLmxhbWJkYS5hZGRUb1JvbGVQb2xpY3kobmV3IGlhbS5Qb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgICAgICAgcmVzb3VyY2VzOiBbJyonXSxcbiAgICAgICAgICAgIGFjdGlvbnM6IFsnY29kZXBpcGVsaW5lOlB1dEpvYlN1Y2Nlc3NSZXN1bHQnLCAnY29kZXBpcGVsaW5lOlB1dEpvYkZhaWx1cmVSZXN1bHQnXSxcbiAgICAgICAgfSkpO1xuICAgICAgICByZXR1cm4ge1xuICAgICAgICAgICAgY29uZmlndXJhdGlvbjoge1xuICAgICAgICAgICAgICAgIEZ1bmN0aW9uTmFtZTogdGhpcy5wcm9wcy5sYW1iZGEuZnVuY3Rpb25OYW1lLFxuICAgICAgICAgICAgICAgIFVzZXJQYXJhbWV0ZXJzOiBTdGFjay5vZihzY29wZSkudG9Kc29uU3RyaW5nKHRoaXMucHJvcHMudXNlclBhcmFtZXRlcnMpLFxuICAgICAgICAgICAgfSxcbiAgICAgICAgfTtcbiAgICB9XG59XG4iXX0=